#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
Defines a custom objects to manage "wildcard filter" used by
file chooser dialogs (such as open/save dialogs).

Require
  Python        2.3.4
  wxPython      2.5.2

Classes
  class         wildcard(string filter)
  class         wildcardList(sequence sequence) extends list
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.3 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  types

import  py_netasq.commonlib.asqDebug                    as asqDebug
import  py_netasq.wx                                    as asqWx

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


class wildcard(object) :
    """\
      Defines a custom object which holds a "wildcard filter" used by
      file chooser dialogs (such as open/save dialogs).
      wildcard filter is made of a descriptive string and a file pattern
      
        aFilter = wildcard('GIF files (*.gif)|*.gif')
        aFilter.description     => 'GIF files (*.gif)'
        aFilter.filePattern     => '*.gif'
      
      To get/set the filter expression, use the "string" property
        aFilter.string          => 'GIF files (*.gif)|*.gif'
      
      Note "str(aFilter)" and "aFilter.string" will return the same
      string.
      
      If no description is provided, then the pattern is used instead :
        aFilter = wildcard('*.*')
        aFilter.description = '*.*'
        str(aFilter)            => '*.*|*.*'
     
     
      constructor wildcard(string filter)
      
      void      assign(object source)
      void      debug(void)
      
      property string description : filter description
      property string filePattern : filter pattern
      property string string      : wildcard filter as a string
    """

    SEPARATOR = '|'

  # Constructor / Destructor  - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  
    def __init__(self, filter) :
        """\
          constructor wildcard(string filter)
         
          Sets wildcard filter's decription and file pattern.
          
          param filter                file filter expression
        """
        assert type(filter) in types.StringTypes                        ,\
                "Wrong parameter type, StringTypes expected"
        
        self._description = ''
        self._filePattern = ''
        self._setString(filter)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Special methods - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __nonzero__(self) :
        """\
          bool __nonzero__(void)
          
          Called to implement truth value testing, and the built-in operation
          bool(); should return False or True, or their integer equivalents
          0 or 1.
          Tells whether or not the filter is "empty": an empty filter has
          no description neither a file pattern.
          
          return                      is wildcard filter not empty
        """
        return bool(self._description) and bool(self._filePattern)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __str__(self) :
        """\
          String __str__(void)
          
          Called by the str() built-in function and by the print statement
          to compute the ``informal'' string representation of an object.
          This differs from __repr__() in that it does not have to be a
          valid Python expression: a more convenient or concise 
          representation may be used instead.
          Return wildcard filter as a single string.
          Example :
            self.description = 'GIF files (*.gif)'
            self.filePattern = '*.gif'
            str(self)           => 'GIF files (*.gif)|*.gif'
          
          return                      informal string representation
        """
        return self._getString()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def assign(self, source) :
        """\
          void assign(object source)
          
          Replace object's wildcard filter with the one provided by given
          'source'. 'source' should be an instance of wildcard.
          
          param source                wildcard source
        """
        assert isinstance(source, wildcard)                            ,\
                "Wrong parameter type, wildcard object expected"
        self._description = source.description
        self._filePattern = source.filePattern


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def debug(self, tab=0) :
        """\
          void debug(int tab)
          
          Output object properties, using asqWx's package logger.
         
          param tab                   left margin length, defaults to 0
        """
        asqWx.logger.debug(asqDebug.debugMessage('{ wildcard object } - - - - - - - - - - - - - -', tab))
        asqWx.logger.debug(asqDebug.debugValue('description', self._description, tab))
        asqWx.logger.debug(asqDebug.debugValue('filePattern', self._filePattern, tab))


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _getDescription(self) :
        """\
          string _getDescription(void)
          
          Returns wildcard filter description string;
         
          return                      filter description
        """
        return self._description


    def _getFilePattern(self) :
        """\
          string _getFilePattern(void)
          
          Returns wildcard filter pattern string;
         
          return                      filter pattern
        """
        return self._filePattern


    def _getString(self) :
        """\
          string _getString(void)
          
          Returns wildcard filter as a string "description | filter";
          exemple :
            'GIF files (*.gif)|*.gif'
         
          return                      filter string
        """
        if not bool(self) :
          buffer = tuple()
        else :
          buffer = (self._description, self._filePattern)
        return wildcard.SEPARATOR.join(buffer)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _setDescription(self, value) :
        """\
          void _setDescription(string value)
          
          Sets filter description string.
          
          param value                 filter description
        """
        self._description = value


    def _setFilePattern(self, value) :
        """\
          void _setFilePattern(string value)
          
          Sets filter file pattern.
          
          param value                 filter pattern
        """
        self._filePattern = value


    def _setString(self, value) :
        """\
          void _setString(string value)

          Sets filter description and file pattern.

          param value                 filter expression
        """
        assert type(value) in types.StringTypes                         ,\
                "Wrong parameter type, StringTypes expected"
        
        chunks = value.split(wildcard.SEPARATOR)
        if 1 < len(chunks) :
          self._description = chunks[0]
          self._filePattern = chunks[1]
        elif 1 == len(chunks) :
          self._description = chunks[0]
          self._filePattern = chunks[0]
        else :
          self._description = ''
          self._filePattern = ''


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    description = property(
                doc  = "filter description",
                fget = _getDescription,
                fset = _setDescription,
                fdel = None )

    filePattern = property(
                doc  = "filter pattern",
                fget = _getFilePattern,
                fset = _setFilePattern,
                fdel = None )

    string      = property(
                doc  = "filter expression as a string",
                fget = _getString,
                fset = _setString,
                fdel = None )


  # /class wildcard - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -



class wildcardList(list) :
    """\
      Defines a custom list which holds wildcard objects.

      Defines a list which holds "wildcard filters" used by file chooser
      dialogs (such as open/save dialogs).
      Instead of a single string, like
        "BMP files (*.bmp)|*.bmp|GIF files (*.gif)|*.gif"
      there will be a list of tuples, like
        [('BMP files (*.bmp)', '*.bmp'), ('GIF files (*.gif)', '*.gif')]
     
     
      constructor wildcardList(sequence sequence)
      
      void      debug(void)
      
    """

    SEPARATOR = '|'

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Class methods - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


  # Constructor / Destructor  - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __init__(self, sequence) :
        """\
        """
        super(wildcardList, self).__init__()
        
        gender = type(sequence)
        if gender in types.StringTypes :
          buffer = wildcard(sequence)
          super(wildcardList, self).append(buffer)
        elif gender in (types.TupleType, types.ListType) :
          buffer = [wildcard(x) for x in sequence]
          super(wildcardList, self).extend(buffer)
        else :
          assert False, "Wrong parameter type, sequence type expected"


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Special methods - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __str__(self) :
        """\
          String __str__(void)
          
          Called by the str() built-in function and by the print statement
          to compute the ``informal'' string representation of an object.
          This differs from __repr__() in that it does not have to be a
          valid Python expression: a more convenient or concise 
          representation may be used instead.
          Please note, result string format does *NOT* comply to the
          regular list representation, example :
          
            aList = wildcardList(['GIF files (*.gif)|*.gif', '*.*'])
          the result of "str(aList)" will be
            "GIF files (*.gif)|*.gif|*.*|*.*"
          instead of
            "['GIF files (*.gif)|*.gif', '*.*|*.*']"
          
          return                      informal string representation
        """
        return wildcardList.SEPARATOR.join([str(x) for x in self])


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __setitem__(self, key, value) :
        """\
          void __setitem__(int key, var value)
          
          Called to implement assignment to "self[key]". Converts given
          'value' to an wildcard object before assignement. If conversion
          cannot be done then a "ValueError" exception will be raised.
          
          param key                   new item index
          param value                 new item value
        """
        value = self._convertItem(value)
        super(wildcardList, self).__setitem__(key, value)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __setslice__(self, i, j, value) :
        """\
          void __setslice__(int i, int j, sequence value)
          
          Called to implement assignment to self[i:j]. Converts values to
          wildcard objects before assignement. If a conversion cannot be
          done then a "ValueError" exception will be raised.
          Note : Even though this method is flagged as deprecated, it seems
          that the regular list still relies on it for slice assignement.
          
          param i                     low index bound
          param j                     high index bound
          param value                 a sequence
        """
        value = self._convertSequence(value)
        super(wildcardList, self).__setslice__(i, j, value)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _convertItem(self, value) :
        """\
          object _convertItem(var value)
          
          Tries to convert given 'value' to an wildcard object. If
          conversion cannot be done then a "ValueError" exception will
          be raised.
          
          param value                 value to convert
          return                      wildcard object
        """
        if isinstance(value, wildcard) :
          return value
        elif type(value) in types.StringTypes :
          return wildcard(value)
        else :
          raise ValueError, "wildcard object or string expected"


    def _convertSequence(self, value) :
        """\
          sequence _convertSequence(var value)
          
          Tries to convert each items of given 'value' sequence to wildcard
          objects. If a conversion cannot be done then a "ValueError"
          exception will be raised.
          
          param value                 sequence to convert
          return                      sequence of wildcard objects
        """
        if isinstance(value, wildcardList) :
          return value
        elif type(value) in (types.TupleType, types.ListType) :
          return map(self._convertItem, value)
        else :
          raise ValueError, "wildcardList object or sequence expected"


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def append(self, value) :
        """\
          void append(var value)
          
          Appends given 'value' to list. Converts given 'value' to an
          wildcard object before assignement. If conversion cannot be
          done then a "ValueError" exception will be raised.
          
          param value                 new item value
        """
        value = self._convertItem(value)
        super(wildcardList, self).append(value)


    def insert(self, index, value) :
        """\
          void insert(var value)
          
          Inserts given 'value' at given 'index'. Converts given 'value'
          to an wildcard object before assignement. If conversion cannot
          be done then a "ValueError" exception will be raised.
          
          param index                 index where to insert value
          param value                 item to insert
        """
        value = self._convertItem(value)
        super(wildcardList, self).insert(index, value)


    def extend(self, value) :
        """\
          void extend(var value)
          
          Extends list with given 'value' sequence. Converts values to
          wildcard objects before assignement. If a conversion cannot be
          done then a "ValueError" exception will be raised.
          
          param value                 a sequence
        """
        value = self._convertSequence(value)
        super(wildcardList, self).extend(value)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def debug(self, tab=0) :
        """\
          void debug(int tab)
          
          Output object properties, using asqWx's package logger.
         
          param tab                   left margin length, defaults to 0
        """
        asqWx.logger.debug('{ wildcardList object } - - - - - - - - - - - -')
        margin = 1 + tab
        for item in self :
          item.debug(margin)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


  # /class wildcardList - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -