#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
Defines function to bind/unbind event table to event handlers.

Require
  Python        2.3.4
  wxPython      2.5.2

Functions
  void          bindEventTable(object window, list evtTable)
  void          unBindEventTable(object window, list evtTable)
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.4 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  wx
import  py_netasq.wx                                    as asqWx

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
#
# Event table entries do comply to the prototype of the Bind method
#
# def Bind(self, event, handler, source=None, id=wxID_ANY, id2=wxID_ANY)
#
#     event     One of the EVT_* objects that specifies the
#               type of event to bind.
#
#     handler   A callable object to be invoked when the event
#               is delivered to self.  Pass None to disconnect an
#               event handler.
#
#     source    Sometimes the event originates from a different window
#               than self, but you still want to catch it in self.  (For
#               example, a button event delivered to a frame.)  By
#               passing the source of the event, the event handling
#               system is able to differentiate between the same event
#               type from different controls.
#
#     id,id2    Used for menu IDs or for event types that require a
#               range of IDs  
#
# A typical event table would look like :
#
#  _eventTable = [
#      ( wx.EVT_SIZE, 'OnResize', None, wx.ID_ANY, wx.ID_ANY ),
#      { 'event' : wx.EVT_MOTION, 'handler' : 'OnMouseMove' }
#    ]
#
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def __convertToDict(value) :
    """\
      dict __convertToDict(sequence value)
      
      HIDDEN - Converts given sequence (list or tuple) to a dictionnary
      that could be passed to the "Bind" method.
      
      param  value                sequence
      return                      a dictionnary
    """
    assert type(value) in (list, tuple, dict)                           ,\
            'sequence or mapping expected, got %s' % (str(type(value)),)
    assert 1 < len(value)                                               ,\
            'not enough items found in given value %s' % (str(value),)
    
    if isinstance(value, dict) :
      return value
    
    length = len(value)
    result = {
        'event'    : None,
        'handler'  : None,
        'source'   : None,
        'id'       : wx.ID_ANY,
        'id2'      : wx.ID_ANY
      }
     
    result['event'    ] = value[0]
    result['handler'  ] = value[1]
    if 2 < length :
      result['source' ] = value[2]
    if 3 < length :
      result['id'     ] = value[3]
    if 4 < length :
      result['id2'    ] = value[4]
    
    return result


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def __debugDict(value) :
    """\
      string __debugDict(dict value)
      
      HIDDEN - Builds a string from given event table entry.
      
      param  value                sequence
      return                      a string
    """
    assert isinstance(value, dict)                                      ,\
            'dictionnary expected , got %s' % (str(type(value)),)
    
    result = []
    result.append('event : %s'   % value['event'].__class__.__name__)
    result.append('handler : %s' % value['handler'])
    result.append('source : %s'  % value.get('source', None))
    result.append('id : %s'      % value.get('id', wx.ID_ANY))
    result.append('id2 : %s'     % value.get('id', wx.ID_ANY))
    return ', '.join(result)


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def bindEventTable(window, evtTable) :
    """\
      void bindEventTable(object window, list evtTable)
      
      Bind events to window object, using given event table. Event table's
      items can either be dictionnaries or sequences (lists/tuples), as long
      as they can be passed to the "Bind" method.
      
      param  window               wx.EvtHandler object
      param  evtTable             event table
    """
    assert hasattr(window, 'Bind')                                      ,\
            'bindEventTable - object has no "Bind" attribute'
    
    asqWx.logger.trace('asqWxCore.bindEventTable')
    asqWx.logger.incDepthTrace()
    
    for entry in evtTable :
      entry  = __convertToDict(entry)
      buffer = __debugDict(entry)
      assert hasattr(window, entry['handler'])                       ,\
              'bindEventTable - object has no "%s" attribute' % (entry['handler'],)
      asqWx.logger.debug('%s.Bind( %s )', window.__class__.__name__, buffer)
      entry['handler'] = getattr(window, entry['handler'])
      window.Bind(**entry)
      
    asqWx.logger.decDepthTrace()


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def unBindEventTable(window, evtTable) :
    """\
      void unBindEventTable(object window, list evtTable)
      
      Disconnect events found in given event table.
      This is a wxPython 2.5.2.x feature.
      
      param  window               wx.EvtHandler object
      param  evtTable             event table
    """
    assert hasattr(window, 'UnBind')                                    ,\
            'unBindEventTable - object has no "UnBind" attribute'
    
    asqWx.logger.trace('asqWxCore.unBindEventTable')
    asqWx.logger.incDepthTrace()
    
    for entry in evtTable :
      entry  = __convertToDict(entry)
      buffer = __debugDict(entry)
      asqWx.logger.debug('%s.Unbind( %s )', window.__class__.__name__, buffer)
      del entry['handler']
      window.Unbind(**entry)
      
    asqWx.logger.decDepthTrace()


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -