#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
Nice function set to build wx.Menu object from static descriptive lists.

Require
  Python        2.3.4
  wxPython      2.5.2

Functions
  Menu          createMenu(list menuTree)
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.2 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  wx

import  py_netasq.commonlib.asqMapping                  as asqMapping


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
#
# a typical menuTree list would look like :
#
#   [
#     { 'id' : ID_ITEM1, 'text' : 'item1', 'help' : 'help item1', 'kind' : wx.ITEM_NORMAL },
#     { 'id' : ID_ITEM2, 'text' : 'item2', 'submenu' :
#       [
#         { 'id' : ID_SUB1, 'text' : 'subItem1', 'help' : 'help subItem1' },
#         { 'kind' : wx.ITEM_SEPARATOR },
#         { 'id' : ID_SUB2, 'text' : 'subItem2', 'help' : 'help subItem2' },
#         { 'id' : ID_SUB3, 'text' : 'subItem3', 'help' : 'help subItem3' }
#       ]
#     }
#   ]
#
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def __convertItem(aItem) :
    """\
      dict __convertItem(var aItem)
      
      HIDDEN - Converts given sequence (list or tuple) to a dictionnary
      that could be passed to the wx.Menu.Append method.
      
      param  aItem                sequence
      return                      a dictionnary
    """
    assert type(aItem) in (list, tuple, dict)                           ,\
            'sequence or mapping expected, got %s' % (str(type(aItem)),)
    assert 2 < len(aItem)                                               ,\
            'At least two elements expected, got only %d' % (len(aItem),)
    
    if isinstance(aItem, dict) :
      return aItem
    
    length = len(aItem)
    result = {
        'id'      : wx.ID_ANY,
        'text'    : '',
        'help'    : '',
        'kind'    : wx.ITEM_NORMAL,
        'submenu' : []
      }
     
    result['id'  ] = aItem[0]
    result['text'] = aItem[1]
    if 2 < length :
      result['help'] = aItem[2]
    if 3 < length :
      result['kind'] = aItem[3]
    if 4 < length :
      assert type(aItem[4]) in (list, tuple, dict)                      ,\
              'item %s unexpected type : %s' % (aItem[1], str(type(aItem[4])))
      result['submenu'][:] = __convertTree(aItem[4])
    
    return result



# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def __convertTree(aTree) :
    """\
      list __convertTree(var aItem)
      
      HIDDEN - Converts given menu tree (list or tuple) to a list of
      dictionnary items (each menu entry is stored as a dict).
      
      param  aTree                list
      return                      list
    """
    assert type(aTree) in (list, tuple)                                 ,\
            'sequence expected, got %s' % (str(type(aTree)),)
    return [__convertItem(x) for x in aTree]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def __debugItem(aItem, depth=0) :
    """\
      string __debugItem(dict aItem, int depth)
      
      HIDDEN - Builds a string from given menu tree entry. Depth is
      used to build a "left margin".
      
      param  aItem                dictionnary
      param  depth                left margin size, defaults to 0
      return                      a string
    """
    result = []
    margin = '  ' * depth
    
    assert isinstance(aItem, dict)                                      ,\
            'dictionnary expected , got %s' % (str(type(aItem)),)
    
    if wx.ITEM_SEPARATOR == aItem.get('kind', wx.ITEM_NORMAL) :
      return '%s----' % (margin,)
    
    result.append('%s%-10s : %d' % (margin, 'id'  , aItem['id']))
    result.append('%s%-10s : %s' % (margin, 'text', aItem['text']))
    result.append('%s%-10s : %s' % (margin, 'help', aItem.get('help', '')))
    result.append('%s%-10s : %d' % (margin, 'kind', aItem.get('kind', wx.ITEM_NORMAL)))
    
    buffer = aItem.get('submenu', None)
    if buffer is None :
      result.append('%s%-10s : %s' % (margin, 'submenu', 'None'))
    else :
      assert isinstance(buffer, list)                                   ,\
              'item %s menu list expected , got %s' % (aItem['text'], str(type(buffer)))
      result.append('%s%-10s : %d item(s)' % (margin, 'submenu', len(buffer)))
      if 0 < len(buffer) :
        result.append(__debugTree(buffer, 1+depth))
   
    return '\n'.join(result)


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def __debugTree(aTree, depth=0) :
    """\
      string __debugTree(list aTree, inc depth)
      
      HIDDEN - Builds a string from given menu tree. Depth is used to
      build a "left margin".
      
      param  aTree                list
      param  depth                left margin size, defaults to 0
      return                      a string
    """
    result = []
    margin = '  ' * depth
    
    if aTree is None :
      return '%s%s' % (margin, 'None')
    
    assert isinstance(aTree, list)                                      ,\
            'list expected , got %s' % (str(type(aTree)),)
    
    for item in aTree :
      result.append(__debugItem(item, depth))
      result.append(margin)
    
    return '\n'.join(result)


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def createMenu(menuTree) :
    """\
      Menu createMenu(list menuTree)
      
      Builds up a wx.Menu object from given 'menuTree' list. A neat and
      simple way to simplify GUI code.
      
      param  menuTree             menu items list (as dictionnaries)
      return                      a wx.Menu
    """
    result = wx.Menu()
    
    for item in __convertTree(menuTree) :
      if wx.ITEM_SEPARATOR == item.get('kind', wx.ITEM_NORMAL) :
        result.AppendSeparator()
      elif 0 == len(item.get('submenu', [])) :
        asqMapping.filterKeys(item, ('id', 'text', 'help', 'kind'))
        result.Append(**item)
      else :
        item['submenu'] = createMenu(item['submenu'])
        asqMapping.filterKeys(item, ('id', 'text', 'submenu', 'help'))
        result.AppendMenu(**item)
    
    return result


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -