#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
Define a ResBuilder task item used by the building utility. ResBuilder is
used to set the application version.

Since
  Python 2.2

Classes
  class         vrsTaskItem(bool enabled, string directory) extends taskItem
  class         vrsTaskList(bool enabled)                   extends taskList
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.5 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  py_netasq.commonlib.asqDebug            as asqDebug

from    py_netasq.delphi.building.basicTasks    import taskItem, taskList

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if __name__ == '__main__' :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class vrsTaskItem(taskItem) :
    """\
      A versioning task: set application version infos to binary files.
      
      
      constructor vrsTaskItem(bool enabled, string directory)
      
      bool      isValid(void)
      object    toCmdLine()
      void      toFile(string name)
      
      void      expandPaths(void)
      void      completePaths(string base)
      
      property string directory   : start reccursion from directory
      property string modifier    : path to resModifier application
      property string cmdextra    : extra application parameters
      
      property string companyname     : app. info, CompanyName
      property string legalcopyright  : app. info, LegalCopyright
      property string filedescription : app. info, FileDescription
      property string fileversion     : app. info, FileVersion
      property string productversion  : app. info, label (alpha, beta, rc...)
    """
    
    TASKTYPE = "vrsTask"
    
    
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Constructor - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    
    def __init__(self, enabled=True, directory=None) :
        """\
          constructor vrsTaskItem(bool enabled, string directory)
          
          Create a new versioning task. all properties are set to
          'None' value.
          
          param enabled               is task enabled, defaults to True
          param directory             start reccursion from directory,
                                        defaults to None
        """
        super(vrsTaskItem, self).__init__(enabled)
        self._tasktype   = vrsTaskItem.TASKTYPE
        self._compatWith = (vrsTaskItem.TASKTYPE,)
        self._vrsFile    = None
        # new attributes
        self._attributes.append('directory')
        self._attributes.append('modifier')
        self._attributes.append('cmdextra')
        self._attributes.append('companyname')
        self._attributes.append('legalcopyright')
        self._attributes.append('filedescription')
        self._attributes.append('fileversion')
        self._attributes.append('productversion')
        # set new attributes to None
        self.reset(False)
        self.directory = directory


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __str__(self) :
        """\
          string __str__(void)
          
          Called by the str() built-in function and by the print statement
          to compute the "informal" string representation of an object. This
          differs from __repr__() in that it does not have to be a valid
          Python expression: a more convenient or concise representation may
          be used instead. The return value must be a string object
          
          Returns a pseudo command line: only the most important attributes
          will be displayed. If current task is found empty, then an empty
          string will be returned.
          
          return                      pseudo command line string
        """
        if not self.isValid() :
          return ''
        else :
          return 'ResModifier %s' % (self.directory,)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def debug(self, tab=0) :
        """\
          void debug(int tab)
          
          Prints out the main attributes values. Each string to print, will
          be prepended with 'tab' space characters.
          
          param tab                   number of char to prepend,
                                       defaults to 0
        """
        prefix = ' ' * tab
        
        super(vrsTaskItem, self).debug(tab)
        print asqDebug.debugValue('_vrsFile', self._vrsFile, tab)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def normalize(self) :
        """\
          void normalize(void)
          
          Correct values types for some peculiar attributes: "enabled";
          for taskItems, None values do not appear in string
          representation, but that value should ALWAYS be present.
        """
        super(vrsTaskItem, self).normalize()
        if hasattr(self, 'fileversion') and                              \
           isinstance(self.fileversion, basestring) :
          self.fileversion.replace(',', '.')


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def isValid(self) :
        """\
          bool isValid(void)
          
          Tells whether or not the task is "valid", id est, if mandatory
          attributes are set; an empty taskItem is not valid, but an
          invalid taskItem may not be empty ;)
          
          return                      is task valid
        """
        #~ if self.isEmpty() :
        if not self :
          return False
        elif not self._isValidString('modifier') :
          return False
        elif not self._isValidString('directory') :
          return False
        else :
          return True


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def reset(self, force=True) :
        """\
          void reset(bool force)
          
          Set object's attributes to None. If 'force' is set to false, then
          missing attributes will be created, other attributes will remain
          unchanged. 'self.normalize()' is also called.
           
          param force                 reset ALL attributes, defaults to True
        """
        if force or not hasattr(self, '_vrsFile') :
          object.__setattr__(self, '_vrsFile', None)
        super(vrsTaskItem, self).reset(force)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def toCmdLine(self) :
        """\
          object toCmdLine(void)
          
          Returns a command line as an asqCmdLine instance. If current task
          is found empty or invalid, then an empty command is returned.
          
          Returned command should look like :
            resModifier <directory> -v <versionFile>
          
          return                      asqCmdLine instance
        """
        result = super(vrsTaskItem, self).toCmdLine()
        # empty task returns empty command line
        if not self.isValid() : return result
          
        # resModifier.exe
        result.appendQuoted(self.modifier)
        result.appendQuoted(self.directory)
         
        if self._vrsFile is not None :
          result.append('-v')
          result.appendQuoted(self._vrsFile)
        
        if self.cmdextra is not None :
          result.append(self.cmdextra)
        
        return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def toFile(self, name) :
        """\
          void toFile(string name)
          
          write task settings to file, given file 'name' will be used to
          generate the accurate command line (toCmdLine() method).
          If file actually does not exist, then it will be created.
         
          param name                  output file name
        """
        opened = file(name, 'w')
        try :
          if self.fileversion     is not None :
            binary = self.__binaryFileVersion(self.fileversion)
            opened.write('FILEVERSION=%s\n'     % (binary,))
            opened.write('PRODUCTVERSION=%s\n'  % (binary,))
          if self.companyname     is not None :
            opened.write('CompanyName=%s\n'     % (self.companyname,))
          if self.legalcopyright  is not None :
            opened.write('LegalCopyright=%s\n'  % (self.legalcopyright,))
          if self.filedescription is not None :
            opened.write('FileDescription=%s\n' % (self.filedescription,))
          if self.fileversion     is not None :
            opened.write('FileVersion=%s\n'     % (self.fileversion,))
          if self.productversion  is not None :
            opened.write('ProductVersion=%s\n'  % (self.productversion,))
            
          self._vrsFile = name
        finally :
          opened.close()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def expandPaths(self) :
        """\
          void expandPaths(void)
          
          Permanently replace task's file paths with "normalized" versions,
          "expandvars()", "normpath()" and "normcase()" are applied to paths.
        """
        super(vrsTaskItem, self).expandPaths()
        self.modifier  = self._normPath(self.modifier)
        self.directory = self._normPath(self.directory)
        self._vrsFile  = self._normPath(self._vrsFile)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def completePaths(self, base) :
        """\
          void completePaths(string base)
          
          Permanently replace object's paths with absolute versions, using
          'base' value as base dir (thus the name ;).
          
          param base                  base for absolute path
        """
        super(vrsTaskItem, self).completePaths(base)
        self.modifier  = self._completePath(base, self.modifier)
        self.directory = self._completePath(base, self.directory)
        self._vrsFile  = self._completePath(base, self._vrsFile)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __binaryFileVersion(self, version) :
        """\
          string __binaryFileVersion(string version)
          
          Return a binary friendly version from the given one; An exception
          may be raised if given 'version' can't be converted (because of
          invalid format).
          The binary FILEVERSION consists of two 32-bit integers, defined by
          four 16-bit integers; Each field of 'version' must be valid 16-bit
          integer values. Out of range values (e.g. '1042') will be converted
          modulus 256 (e.g. 1042 % 256 => 18).
          
          param version               FileVersion string
          return                      FILEVERSION string
        """
        try :
          chunks = version.split('.')[:4]
          chunks = map(lambda s: str(int(s) % 256), chunks)
        except ValueError :
          raise ValueError, "invalid file version [%s], fields must be numeric values" % (version,)
        else :
          return ','.join(chunks)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # class vrsTaskItem - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


class vrsTaskList(taskList) :
    """\
      A list of vrsTaskItems, the list defines "default" attributes that each
      single item can override; if an item doesn't set an attribute, then the
      list's values will be used.
      
      
      constructor vrsTaskList(bool enabled)
      
      object    appendNewTask(bool enabled, string directory)
      
      property string modifier    : path to resModifier application
      property string cmdextra    : extra application parameters
      
      property string companyname     : app. info, CompanyName
      property string legalcopyright  : app. info, LegalCopyright
      property string filedescription : app. info, FileDescription
      property string fileversion     : app. info, FileVersion
      property string productversion  : app. info, label (alpha, beta, rc...)
    """
    
    
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Constructor - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    
    def __init__(self, enabled=True) :
        """\
          constructor vrsTaskList(bool enabled)
          
          Create an empty list of vrsTaskItems. If a whole taskList is
          disabled, then NONE of its task items will be done.
          
          param enabled               is tasklist enabled, defaults to True
        """
        super(vrsTaskList, self).__init__(enabled)
        self._tasktype   = vrsTaskItem.TASKTYPE
        self._compatWith = (vrsTaskItem.TASKTYPE,)
        # new attributes
        self._attributes.append('modifier')
        self._attributes.append('cmdextra')
        self._attributes.append('companyname')
        self._attributes.append('legalcopyright')
        self._attributes.append('filedescription')
        self._attributes.append('fileversion')
        self._attributes.append('productversion')
        # set new attributes to None
        self.reset(False)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def appendNewTask(self, enabled=False, directory=None) :
        """\
          object appendNewTask(bool enabled, string directory)
          
          Returns a brand new task, which has been added to taskList.
          Somehow, it looks a little bit like a "factory" function.
          BEWARE: If no parameter is set, then an EMPTY AND DISABLED
          taskItem will be yield.
          
          param enabled               is task enabled, defaults to False
          param directory             start reccursion from directory,
                                        defaults to None
          return                      a new task item
        """
        result = vrsTaskItem(enabled, directory)
        self.append(result)
        return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # class vrsTaskList - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -