#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
Introduce the "factory" design pattern: create the accurate
class instance given the tasktype. There is no need to
create a buildFactory instance; buildFactory only defines
class methods.

Since
  Python 2.2

Classes
  class         taskFactory(void)
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.5 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

from    py_netasq.delphi.building.basicTasks    import taskItem, taskList

from    py_netasq.delphi.building.cvsCommand    import cvsCommand
from    py_netasq.delphi.building.dccCommand    import dccCommand
from    py_netasq.delphi.building.delCommand    import delCommand
from    py_netasq.delphi.building.rawCommand    import rawCommand
from    py_netasq.delphi.building.resCommand    import resCommand
from    py_netasq.delphi.building.sf6Command    import sf6Command
from    py_netasq.delphi.building.vrsCommand    import vrsCommand

from    py_netasq.delphi.building.cvsTasks      import cvsTaskItem, cvsTaskList
from    py_netasq.delphi.building.dccTasks      import dccTaskItem, dccTaskList
from    py_netasq.delphi.building.delTasks      import delTaskItem, delTaskList
from    py_netasq.delphi.building.rawTasks      import rawTaskItem, rawTaskList
from    py_netasq.delphi.building.resTasks      import resTaskItem, resTaskList
from    py_netasq.delphi.building.sf6Tasks      import sf6TaskItem, sf6TaskList
from    py_netasq.delphi.building.vrsTasks      import vrsTaskItem, vrsTaskList

from    py_netasq.delphi.building.mailCommand   import mailCommand
from    py_netasq.delphi.building.mailTasks     import mailTaskItem

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if __name__ == '__main__' :
  print __file__
  print __doc__

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class taskFactory(object) :
    """\
      Define a factory to create task items according to a tasktype.
      This class should not be instanciated, classmethods are provided.
      
      object    buildTaskItem(string aType, *args, **kargs)
      object    buildTaskList(string aType, *args, **kargs)
      
      object    buildCommand(string aType, *args, **kargs)
    """
    
    TASKITEM_TYPES      = (
        cvsTaskItem.TASKTYPE,
        dccTaskItem.TASKTYPE,
        resTaskItem.TASKTYPE,
        sf6TaskItem.TASKTYPE,
        vrsTaskItem.TASKTYPE,
        rawTaskItem.TASKTYPE,
        delTaskItem.TASKTYPE,
        
        mailTaskItem.TASKTYPE
      )
    
    TASKLIST_TYPES      = (
        cvsTaskItem.TASKTYPE,
        dccTaskItem.TASKTYPE,
        resTaskItem.TASKTYPE,
        sf6TaskItem.TASKTYPE,
        vrsTaskItem.TASKTYPE,
        rawTaskItem.TASKTYPE,
        delTaskItem.TASKTYPE
      )
    
    ITEM_CLASSES    = {
        cvsTaskItem.TASKTYPE  : cvsTaskItem,
        dccTaskItem.TASKTYPE  : dccTaskItem,
        resTaskItem.TASKTYPE  : resTaskItem,
        sf6TaskItem.TASKTYPE  : sf6TaskItem,
        vrsTaskItem.TASKTYPE  : vrsTaskItem,
        rawTaskItem.TASKTYPE  : rawTaskItem,
        delTaskItem.TASKTYPE  : delTaskItem,
       
        mailTaskItem.TASKTYPE : mailTaskItem
      }
    
    LIST_CLASSES    = {
        cvsTaskItem.TASKTYPE  : cvsTaskList,
        dccTaskItem.TASKTYPE  : dccTaskList,
        resTaskItem.TASKTYPE  : resTaskList,
        sf6TaskItem.TASKTYPE  : sf6TaskList,
        vrsTaskItem.TASKTYPE  : vrsTaskList,
        rawTaskItem.TASKTYPE  : rawTaskList,
        delTaskItem.TASKTYPE  : delTaskList
      }
   
    CMD_CLASSES     = {
        cvsTaskItem.TASKTYPE  : cvsCommand,
        dccTaskItem.TASKTYPE  : dccCommand,
        resTaskItem.TASKTYPE  : resCommand,
        sf6TaskItem.TASKTYPE  : sf6Command,
        vrsTaskItem.TASKTYPE  : vrsCommand,
        rawTaskItem.TASKTYPE  : rawCommand,
        delTaskItem.TASKTYPE  : delCommand,
       
        mailTaskItem.TASKTYPE : mailCommand
      }
   
   
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Class methods - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def buildTaskItem(cls, aType, *args, **kargs) :
        """\
          object buildTaskItem(string aType, *args, **kargs)
          
          Create a task item, according to given task type 'aType'. If task
          type is not recognised then None is returned.
          
          param aType                 task type
          return                      a new task item, or None
        """
        try :
          return cls.ITEM_CLASSES[aType](*args, **kargs)
        except KeyError :
          return None


    def buildTaskList(cls, aType, *args, **kargs) :
        """\
          object buildTaskList(string aType, *args, **kargs)
          
          Create a task list, according to given task type 'aType'. If task
          type is not recognised then None is returned.
          
          param aType                 task type
          return                      a new task list, or None
        """
        try :
          return cls.LIST_CLASSES[aType](*args, **kargs)
        except KeyError :
          return None


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def buildCommand(cls, aType, *args, **kargs) :
        """\
          object buildCommand(string aType, *args, **kargs)
          
          Create a command according to given task type 'aType'. If task
          type is not recognised then None is returned.
          
          param aType                 task type
          return                      a new command, or None
        """
        try :
          return cls.CMD_CLASSES[aType](*args, **kargs)
        except KeyError :
          return None


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    buildTaskItem = classmethod(buildTaskItem)
    buildTaskList = classmethod(buildTaskList)
    
    buildCommand  = classmethod(buildCommand)

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # class taskFactory - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -