#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
Define a Indigo Rose Setup Factory task item used by the building utility.

Since
  Python 2.2

Classes
  class         sf6TaskItem(bool enabled, string projet) extends taskItem
  class         sf6TaskList(bool enabled)                extends taskList
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.5 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  py_netasq.commonlib.asqDebug            as asqDebug
import  py_netasq.commonlib.asqString           as asqString

from    py_netasq.delphi.building.basicTasks    import taskItem, taskList

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if __name__ == '__main__' :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class sf6TaskItem(taskItem) :
    """\
      A setup Factory task: create a "setup" executable for an application.
      
      
      constructor sf6TaskItem(bool enabled, string project)
      
      bool      isValid(void)
      object    toCmdLine()
      
      void      expandPaths(void)
      void      completePaths(string base)
      
      property string project    : setup Factory project (*.sf6)
      property string factory    : path to setup factory design app
      property string cmdextra   : extra application parameters
    """
    
    TASKTYPE = "sf6Task"
    
    
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Constructor - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    
    def __init__(self, enabled=True, project=None) :
        """\
          constructor sf6TaskItem(bool enabled, string project)
          
          Create a new setup Factory task. all properties are set to
          'None' value.
          
          param enabled               is task enabled, defaults to True
          param module                setup factory project (*.sf6)
        """
        super(sf6TaskItem, self).__init__(enabled)
        self._tasktype   = sf6TaskItem.TASKTYPE
        self._compatWith = (sf6TaskItem.TASKTYPE,)
        self._iniFile    = None
        # new attributes
        self._attributes.append('project')
        self._attributes.append('factory')
        self._attributes.append('cmdextra')
        # set new attributes to None
        self.reset(False)
        self.project = project


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __str__(self) :
        """\
          string __str__(void)
          
          Called by the str() built-in function and by the print statement
          to compute the "informal" string representation of an object. This
          differs from __repr__() in that it does not have to be a valid
          Python expression: a more convenient or concise representation may
          be used instead. The return value must be a string object
          
          Returns a pseudo command line: only the most important attributes
          will be displayed. If current task is found empty, then an empty
          string will be returned.
          
          return                      pseudo command line string
        """
        if not self.isValid() :
          return ''
        else :
          return 'SUF60Design %s' % (self.project,)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def debug(self, tab=0) :
        """\
          void debug(int tab)
          
          Prints out the main attributes values. Each string to print, will
          be prepended with 'tab' space characters.
          
          param tab                   number of char to prepend,
                                        defaults to 0
        """
        super(sf6TaskItem, self).debug(tab)
        print asqDebug.debugValue('_iniFile', self._iniFile, tab)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def isValid(self) :
        """\
          bool isValid(void)
          
          Tells whether or not the task is "valid", id est, if mandatory
          attributes are set; an empty taskItem is not valid, but an
          invalid taskItem may not be empty ;)
          
          return                      is task valid
        """
        #~ if self.isEmpty() :
        if not self :
          return False
        elif not self._isValidString('factory') :
          return False
        elif not self._isValidString('project') :
          return False
        else :
          return True


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def reset(self, force=True) :
        """\
          void reset(bool force)
          
          Set object's attributes to None. If 'force' is set to false, then
          missing attributes will be created, other attributes will remain
          unchanged. 'self.normalize()' is also called.
           
          param force                 reset ALL attributes, defaults to True
        """
        if force or not hasattr(self, '_iniFile') :
          object.__setattr__(self, '_iniFile', None)
         
        super(sf6TaskItem, self).reset(force)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def toCmdLine(self) :
        """\
          object toCmdLine(void)
          
          Returns a command line as an asqCmdLine instance. If current task
          is found empty or invalid, then an empty command is returned.
          
          Returned command should look like :
            SUF60Design.exe <project> /B:<inifile> <cmdextra>
          
          return                      asqCmdLine instance
        """
        result = super(sf6TaskItem, self).toCmdLine()
        # empty task returns empty command line
        if not self.isValid() or self._iniFile is None :
          return result
        
        # SUF60Design.exe
        result.appendQuoted(self.factory)
        # command params
        result.appendQuoted(self.project)
        result.appendQuoted('/B:%s' % (self._iniFile,))
        # extra params
        if self.cmdextra is not None :
          result.append(self.cmdextra)
          
        return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def toFile(self, name) :
        """\
          void toFile(string name)
          
          write task settings to file, given file 'name' will be used to
          generate the accurate command line (toCmdLine() method).
          If file actually does not exist, then it will be created.
         
          param name                  output file name
        """
        opened = file(name, 'w')
        try :
          opened.write('[Constants]\n')
          keylst = filter(self.__isSufConstant, self._miscvalues.keys())
          for key in keylst :
            # value MUST NOT BE SURROUNDED by quotes !!
            val = asqString.unQuoteString(self._miscvalues[key], '"', '\\')
            key = self.__toSufConstant(key)
            opened.write('%s = %s\n' % (key, val))
          self._iniFile = name
        finally :
          opened.close()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __isSufConstant(self, key) :
        """\
          bool __isSufConstant(string key)
          
          Tell whether or not given string is recognized as a Setup Factory
          ini constant. Setup Factory constants do begin and end with '#',
          alas, this character is also used to define ini file comment
          lines, so we'll use '$' instead : A constant name must begin and
          end with a '$' char.
          e.g.
            self.__isSufContant('$BUILD$') => True
            self.__isSufContant('$$')      => False (empty name)
            self.__isSufContant('$MODEL')  => False
           
          param key                   string to test
          return                      is a constant name
        """
        return 2 < len(key) and ('$' == key[0] == key[-1])


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __toSufConstant(self, name) :
        """\
          string __toSufConstant(string name)
          
          Replace beginning and trailing '$' character with '#', and
          convert string to uppercase.
          e.g.
            self.__toSufContant('$build$') => #BUILD#
           
          param name                  string to convert
          return                      a sf6 constant name
        """
        return '#%s#' % (name[1:-1].upper(),)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def expandPaths(self) :
        """\
          void expandPaths(void)
          
          Permanently replace task's file paths with "normalized" versions,
          "expandvars()", "normpath()" and "normcase()" are applied to paths.
        """
        super(sf6TaskItem, self).expandPaths()
        self.project  = self._normPath(self.project)
        self.factory  = self._normPath(self.factory)
        self._iniFile = self._normPath(self._iniFile)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def completePaths(self, base) :
        """\
          void completePaths(string base)
          
          Permanently replace object's paths with absolute versions, using
          'base' value as base dir (thus the name ;).
          
          param base                  base for absolute path
        """
        super(sf6TaskItem, self).completePaths(base)
        self.project  = self._completePath(base, self.project)
        self.factory  = self._completePath(base, self.factory)
        self._iniFile = self._completePath(base, self._iniFile)
 

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # class sf6TaskItem - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


class sf6TaskList(taskList) :
    """\
      A list of sf6TaskItems, the list defines "default" attributes that each
      single item can override; if an item doesn't set an attribute, then the
      list's values will be used.
      
      
      constructor sf6TaskList(bool enabled)
      
      object    appendNewTask(bool enabled, string project)
      
      property string factory    : path to setup factory design app
      property string cmdextra   : extra application parameters
    """
    
    
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Constructor - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    
    def __init__(self, enabled=True) :
        """\
          constructor sf6TaskList(bool enabled)
          
          Create an empty list of sf6TaskItems. If a whole taskList is
          disabled, then NONE of its task items will be done.
          
          param enabled               is tasklist enabled, defaults to True
        """
        super(sf6TaskList, self).__init__(enabled)
        self._tasktype   = sf6TaskItem.TASKTYPE
        self._compatWith = (sf6TaskItem.TASKTYPE,)
        # new attributes
        self._attributes.append('factory')
        self._attributes.append('cmdextra')
        # set new attributes to None
        self.reset(False)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def appendNewTask(self, enabled=False, project=None) :
        """\
          object appendNewTask(string project, bool enabled)
          
          Returns a brand new task, which has been added to taskList.
          Somehow, it looks a little bit like a "factory" function.
          BEWARE: If no parameter is set, then an EMPTY AND DISABLED
          taskItem will be yield.
          
          param enabled               is task enabled, defaults to False
          param module                resManager project, defaults to None
          return                      a new task item
        """
        result = sf6TaskItem(enabled, project)
        self.append(result)
        return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # class sf6TaskList - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -