#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
Define a ResManager task item used by the building utility.

Since
  Python 2.2

Classes
  class         resTaskItem(bool enabled, string projet) extends taskItem
  class         resTaskList(bool enabled)                extends taskList
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.5 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

from    py_netasq.delphi.building.basicTasks    import taskItem, taskList

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if __name__ == '__main__' :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class resTaskItem(taskItem) :
    """\
      A ResManager task: compile images ressources files.
      
      
      constructor resTaskItem(bool enabled, string project)
      
      bool      isValid(void)
      object    toCmdLine()
      
      void      expandPaths(void)
      void      completePaths(string base)
      
      property string project    : resManager project (*.rmc)
      property string manager    : path to resManager application
      property string cmdextra   : extra application parameters
      property string inifile    : resManager application options (*.ini)
      property string resdir     : image directory
      property string srcdir     : ????
    """
    
    TASKTYPE = "resTask"
    
    
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Constructor - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    
    def __init__(self, enabled=True, project=None) :
        """\
          constructor resTaskItem(bool enabled, string project)
          
          Create a new resManager task. all properties are set to 
          'None' value.
          
          param enabled               is task enabled, defaults to True
          param module                resManager project, defaults to None
        """
        super(resTaskItem, self).__init__(enabled)
        self._tasktype   = resTaskItem.TASKTYPE
        self._compatWith = (resTaskItem.TASKTYPE,)
        # new attributes
        self._attributes.append('project')
        self._attributes.append('manager')
        self._attributes.append('cmdextra')
        self._attributes.append('inifile')
        self._attributes.append('resdir')
        self._attributes.append('srcdir')
        # set new attributes to None
        self.reset(False)
        self.project = project


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __str__(self) :
        """\
          string __str__(void)
          
          Called by the str() built-in function and by the print statement
          to compute the "informal" string representation of an object. This
          differs from __repr__() in that it does not have to be a valid
          Python expression: a more convenient or concise representation may
          be used instead. The return value must be a string object
          
          Returns a pseudo command line: only the most important attributes
          will be displayed. If current task is found empty, then an empty
          string will be returned.
          
          return                      pseudo command line string
        """
        if not self.isValid() :
          return ''
        else :
          return 'ResManager -p %s' % (self.project,)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def isValid(self) :
        """\
          bool isValid(void)
          
          Tells whether or not the task is "valid", id est, if mandatory
          attributes are set; an empty taskItem is not valid, but an
          invalid taskItem may not be empty ;)
          
          return                      is task valid
        """
        #~ if self.isEmpty() :
        if not self :
          return False
        elif not self._isValidString('manager') :
          return False
        elif not self._isValidString('project') :
          return False
        else :
          return True


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def toCmdLine(self) :
        """\
          object toCmdLine(void)
          
          Returns a command line as an asqCmdLine instance. If current task
          is found empty or invalid, then an empty command is returned.
          
          Returned command should look like :
            ResManager -p <project>
              <cmdextra> -o <inifile> -r <resdir> -s <srcdir>
          
          return                      asqCmdLine instance
        """
        result = super(resTaskItem, self).toCmdLine()
        # empty task returns empty command line
        if not self.isValid() : return result
          
        # resManager.exe
        result.appendQuoted(self.manager)
        result.append('-p')
        result.appendQuoted(self.project)
        
        if self.inifile  is not None :
          result.append('-o')
          result.appendQuoted(self.inifile)
        if self.resdir is not None :
          result.append('-r')
          result.appendQuoted(self.resdir)
        if self.srcdir is not None :
          result.append('-s')
          result.appendQuoted(self.srcdir)
        if self.cmdextra is not None :
          result.append(self.cmdextra)
          
        return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def expandPaths(self) :
        """\
          void expandPaths(void)
          
          Permanently replace task's file paths with "normalized" versions,
          "expandvars()", "normpath()" and "normcase()" are applied to paths.
        """
        super(resTaskItem, self).expandPaths()
        self.manager = self._normPath(self.manager)
        self.project = self._normPath(self.project)
        self.inifile = self._normPath(self.inifile)
        self.srcdir  = self._normPath(self.srcdir)
        self.resdir  = self._normPath(self.resdir)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def completePaths(self, base) :
        """\
          void completePaths(string base)
          
          Permanently replace object's paths with absolute versions, using
          'base' value as base dir (thus the name ;).
          
          param base                  base for absolute path
        """
        super(resTaskItem, self).completePaths(base)
        self.manager = self._completePath(base, self.manager)
        self.project = self._completePath(base, self.project)
        self.inifile = self._completePath(base, self.inifile)
        self.srcdir  = self._completePath(base, self.srcdir)
        self.resdir  = self._completePath(base, self.resdir)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # class resTaskItem - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


class resTaskList(taskList) :
    """\
      A list of resTaskItems, the list defines "default" attributes that each
      single item can override; if an item doesn't set an attribute, then the
      list's values will be used.
      
      
      constructor resTaskList(bool enabled)
      
      object    appendNewTask(bool enabled, string project)
      
      property string manager    : path to resManager application
      property string cmdextra   : extra application parameters
      property string inifile    : resManager application options (*.ini)
      property string resdir     : image directory
      property string srcdir     : ????
    """
   
    
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Constructor - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    
    def __init__(self, enabled=True) :
        """\
          constructor resTaskList(bool enabled)
          
          Create an empty list of resTaskItems. If a whole taskList is
          disabled, then NONE of its task items will be done.
          
          param enabled               is tasklist enabled, defaults to True
        """
        super(resTaskList, self).__init__(enabled)
        self._tasktype   = resTaskItem.TASKTYPE
        self._compatWith = (resTaskItem.TASKTYPE,)
        # new attributes
        self._attributes.append('manager')
        self._attributes.append('cmdextra')
        self._attributes.append('inifile')
        self._attributes.append('resdir')
        self._attributes.append('srcdir')
        # set new attributes to None
        self.reset(False)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def appendNewTask(self, enabled=False, project=None) :
        """\
          object appendNewTask(bool enabled, string project)
          
          Returns a brand new task, which has been added to taskList.
          Somehow, it looks a little bit like a "factory" function.
          BEWARE: If no parameter is set, then an EMPTY AND DISABLED
          taskItem will be yield.
          
          param enabled               is task enabled, defaults to False
          param module                resManager project, defaults to None
          return                      a new task item
        """
        result = resTaskItem(enabled, project)
        self.append(result)
        return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # class resTaskList - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -