#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
A versatile custom task item used by the building utility.

Since
  Python 2.2

Classes
  class         rawTaskItem(bool enabled) extends taskItem
  class         rawTaskList(bool enabled) extends taskList
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.6 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  py_netasq.commonlib.asqDebug            as asqDebug

from    py_netasq.delphi.building.basicTasks    import taskItem, taskList

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if __name__ == '__main__' :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class rawTaskItem(taskItem) :
    """\
      A custom flexible task.
      
      
      constructor rawTaskItem(bool enabled)
      
      bool      isValid(void)
      object    toCmdLine()
      
      property list   arglist    : list of arguments
    """
    
    TASKTYPE        = "rawTask"
    
    KEYWORD_ARGITEM = "arg%u"
    KEYWORD_ARGLEN  = "arglen"
    
    
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Constructor - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    
    def __init__(self, enabled=True) :
        """\
          constructor rawTaskItem(string project, bool enabled)
          
          Create a new "raw" task. all properties are set to 'None' value.
          
          param enabled               is task enabled, defaults to True
        """
        super(rawTaskItem, self).__init__(enabled)
        self._tasktype   = rawTaskItem.TASKTYPE
        self._compatWith = (rawTaskItem.TASKTYPE,)
        self._arglist    = list()
        
        self.reset(False)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Special methods - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __nonzero__(self) :
        """\
          bool __nonzero__(void)
          
          Called to implement truth value testing, and the built-in operation
          bool(); should return False or True, or their integer equivalents
          0 or 1.
          Tells whether or not the task is "empty": an empty task has no
          value defined (None values) for its attributes. Performed right
          after a "task.reset()" this function should return "False".
          
          return                      is task not empty
        """
        return super(rawTaskItem, self).__nonzero__() or bool(self._arglist)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __str__(self) :
        """\
          string __str__(void)
          
          Called by the str() built-in function and by the print statement
          to compute the "informal" string representation of an object. This
          differs from __repr__() in that it does not have to be a valid
          Python expression: a more convenient or concise representation may
          be used instead. The return value must be a string object
          
          Returns a pseudo command line: only the most important attributes
          will be displayed. If current task is found empty, then an empty
          string will be returned.
          
          return                      pseudo command line string
        """
        return str(self.toCmdLine())


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def debug(self, tab=0) :
        """\
          void debug(int tab)
          
          Prints out the main attributes values. Each string to print, will
          be prepended with 'tab' space characters.
          
          param tab                   number of char to prepend,
                                       defaults to 0
        """
        prefix = ' ' * tab
        length = len(self._arglist)
        
        print asqDebug.debugValue('tasktype', self._tasktype, tab, False)
        for key in self._attributes :
          print asqDebug.debugValue(key, getattr(self, key), tab)
         
        print asqDebug.debugValue('arglen', length, tab)
         
        for indx in xrange(length) :
          key = rawTaskItem.KEYWORD_ARGITEM % (indx,)
          val = self._arglist[indx]
          print asqDebug.debugValue(key, val, tab)
         
        if not self._miscvalues : return
          
        print '%s[ TASK misc.  ] - - - - - - - - - - - - - -' % (prefix,)
        for (key, val) in self._miscvalues.iteritems() :
          print asqDebug.debugValue(key, val, tab)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def isValid(self) :
        """\
          bool isValid(void)
          
          Tells whether or not the task is "valid", id est, if mandatory
          attributes are set; an empty taskItem is not valid, but an
          invalid taskItem may not be empty ;)
          
          return                      is task valid
        """
        return bool(self) and bool(self._arglist)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def isEmpty(self) :
        """\
          bool isEmpty(void)
          
          DEPRECATED use bool(self) (see __nonzero__ method)
          
          Tells whether or not the task is "empty": an empty task has no
          value defined (None values) for its attributes. Performed right
          after a "task.reset()" this function should return "True".
          
          return                      is task empty
        """
        return super(rawTaskItem, self).isEmpty() and not self._arglist


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def reset(self, force=True) :
        """\
          void reset(bool force)
          
          Set object's attributes to None. If 'force' is set to false, then
          missing attributes will be created, other attributes will remain
          unchanged. 'self.normalize()' is also called.
           
          param force                 reset ALL attributes, defaults to True
        """
        if not force or not hasattr(self, '_arglist') :
          object.__setattr__(self, '_arglist', list())
        else :
          self._arglist[:] = []
        super(rawTaskItem, self).reset(force)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def fromList(self, aList, listType=None) :
        """\
          void fromList(list aList, string listType)
          
          Reset object's attributes and then overwrite attributes with those
          provided by given list. If there is not "tasktype" value found in
          list, then 'listType' is assumed. If type is not compatible with
          current object, then only the "commonbase" attributes will be
          modified, all others attributes are simply ignored.
          Else, if list is compatible, then unrecognised attributes are added
          to the "miscvalues" list.
          
          e.g. [('logFile', None), ('foo', 'bar'), ('arglen', 1), ('arg0', 'copy')]
            => self.logfile           = None
               self.arglist           = ['copy']
               self.miscvalues['foo'] = 'bar'
           
          param aList                 list of attributes
          param listType              list assumed tasktype, defaults to None
        """
        self.reset(True)
        
        buffer = aList[:]
        # remove tasktype value
        tkType = self._extractTaskType(buffer, listType)
        
        if not self.isCompatibleWith(tkType) :
          self._fromList(buffer, False)
        else :
          # extract args from list
          self._arglist[:] = self._extractArgList(buffer)
          self._fromList(buffer, True)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def toList(self, filtered=False) :
        """\
          list toList(bool filtered)
          
          Return object's attributes, using the following format 
          [(attr, val), (attr, val), ...]. If filtered is set to True, then
          None attributes values won't be added to list.
          
          e.g. self.comments = 'dummy comment'
               self.logFile  = None
               self.miscvalues['dummy'] = None
            
              toList(False)
            => [('comments', 'dummy comment'), ('logFile', None), ('dummy', None)]
              toList(True)
            => [('comments', 'dummy comment')]
            
          param filtered              skip None values
          return                      list of attributes
        """
        result = super(rawTaskItem, self).toList(filtered)
        length = len(self._arglist)
        
        result.append( (rawTaskItem.KEYWORD_ARGLEN, length) )
        for indx in xrange(length) :
          key = rawTaskItem.KEYWORD_ARGITEM % (indx,)
          val = self._arglist[indx]
          result.append( (key, val) )
        if filtered :
          result[:] = filter(lambda x: x[1] is not None, result)
        
        return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def mergeWithList(self, aList, force=False, listType=None) :
        """\
          object mergeWithList(list aList, bool force, string listType)
          
          Return a copy of current object, but with "None" attributes values
          replaced with those provided by 'aList'.
          If 'aList' is not found compatible with current object (i.e. if
          "tasklist" option found in 'aList' does not match), then only
          attributes marked as "commonbase" will be replaced, other ones are
          ignored. If 'force' is not set, then 'enabled' and 'comments'
          attributes won't be replaced, even if they equals "None".
          
          If list is found compatible, items that are not recognised as
          attributes (or commonbase) will be added to the 'miscvalues'
          sequence.
          
          Compatible or not, current arglist will remain unchanged.
           
          param aList                 list of attributes
          param force                 force attributes replacement,
                                        defaults to False
          param listType              list assumed tasktype, defaults to None
          return                      a new task
        """
        buffer = aList[:]
        # remove tasktype value
        tkType = self._extractTaskType(buffer, listType)
        
        if not self.isCompatibleWith(tkType) :
          return self._mergeWithList(buffer, False, force)
        else :
          # args found in list are ignored
          self._extractArgList(buffer)
          return self._mergeWithList(buffer, True, force)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _extractArgList(self, aList) :
        """\
          list _extractArgList(list aList)
          
          Returns the ordered argument list found in 'aList', extracted
          values are removed from given list; the returned list does not
          share 'aList' format
          e.g. aList = [('arglen', 1), ('arg0', 'copy'), ('foo', bar'), ]
               args  = _extractArgList(aList)
               
               args  => ['copy']
               aList => [ ('foo', 'bar') ]
               
          param aList                 list to extract values from
          return                      extracted arg list
        """
        result = list()
        length = 0
        buffer = dict(aList)
        
        if rawTaskItem.KEYWORD_ARGLEN in buffer :
          length = int(buffer[rawTaskItem.KEYWORD_ARGLEN])
          del buffer[rawTaskItem.KEYWORD_ARGLEN]
        for indx in xrange(length) :
          key = rawTaskItem.KEYWORD_ARGITEM % (indx,)
          if key not in buffer :
            result.append('')
          else :
            result.append(buffer[key])
            del buffer[key]
        aList[:] = buffer.items()
        
        return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def toCmdLine(self) :
        """\
          object toCmdLine(void)
          
          Returns a command line as an asqCmdLine instance. If current task
          is found empty or invalid, then an empty command is returned.
          No arguments will be quoted !
          
          Returned command should look like :
            arglist[0] arglist[1] .....
          
          return                      asqCmdLine instance
        """
        result = super(rawTaskItem, self).toCmdLine()
        # empty task returns empty command line
        if self.isValid() :
          result.extend(self._arglist)
        return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _getArgList(self) :
        """\
          list _getArgList(void)
          
          Getter - get task argument list
          
          return                      a list
        """
        return self._arglist

    def _getArgLen(self) :
        """\
          int _getArgLen(void)
          
          Getter - get task argument list length
          
          return                      list length
        """
        return len(self._arglist)

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    arglist = property(
                doc  = "dynamic arguments list",
                fget = _getArgList,
                fset = None,
                fdel = None )

    arglen  = property(
                doc  = "length of dynamic arguments list",
                fget = _getArgLen,
                fset = None,
                fdel = None )

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # class rawTaskItem - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


class rawTaskList(taskList) :
    """\
      A list of rawTaskItems, the list defines "default" attributes that each
      single item can override; if an item doesn't set an attribute, then the
      list's values will be used.
      
      raw task list does only define default values for the basic taskitem's
      attributes: logfile and comment; there is no 'arglist' property.
      
      
      constructor rawTaskList(void)
      
      void debug(int tab)
    """
    
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Constructor - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    
    def __init__(self, enabled=True) :
        """\
          constructor rawTaskList(void)
          
          Create an empty list of rawTaskItems. If a whole taskList is
          disabled, then NONE of its task items will be done.
          
          param enabled               is tasklist enabled, defaults to True
        """
        super(rawTaskList, self).__init__(enabled)
        self._tasktype   = rawTaskItem.TASKTYPE
        self._compatWith = (rawTaskItem.TASKTYPE,)
        
        self.reset(False)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def appendNewTask(self, enabled=False) :
        """\
          object appendNewTask(bool enabled)
          
          Returns a brand new task, which has been added to taskList.
          Somehow, it looks a little bit like a "factory" function.
          BEWARE: If no parameter is set, then an EMPTY AND DISABLED
          taskItem will be yield.
          
          param enabled               is task enabled, defaults to False
          return                      a new task item
        """
        result = rawTaskItem(enabled)
        self.append(result)
        return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # class rawTaskList - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -