#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
Define a "Dcc Task item" list used by the building utility.

Since
  Python 2.2

Classes
  class         dccTaskItem(bool enabled, string project) extends taskItem
  class         dccTaskList(bool enabled)                 extends taskList
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.6 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

from    py_netasq.commonlib.asqCmdLine          import asqCmdLine
from    py_netasq.delphi.building.basicTasks    import taskItem, taskList

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if __name__ == '__main__' :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


class dccTaskItem(taskItem) :
    """\
      A simple task used by the automated building system; this class is
      used to compile delphi projects.
      
      constructor dccTaskItem(bool enabled, string project)
      
      bool      isValid(void)
      
      object    toCmdLine()
      object    patchCommand(string target)
     
      void      expandPaths(void)
      void      completePaths(string base)
      
      property string project   : path to project file
      property string compiler  : path to compiler
      property string cmdextra  : extra compiler params
      property string bindir    : binary (exe/dll) ouput directory
      property string dcudir    : dcu output directory
      property string cfgfile   : path to project configuration file
      
      property string mad_patch   : path to madExceptPatch utility
      property string mad_mesfile : path to .mes file
      property string mad_mapfile : path to .map file
      property string mad_extra   : extra parameters
    """
    
    TASKTYPE = "dccTask"
    
    
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Constructor - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    
    def __init__(self, enabled=True, project=None) :
        """\
          constructor dccTaskItem(bool enabled, string project)
          
          Create a simple record of strings: path settings and comments.
          Except 'project', all properties defaults to 'None'.
          
          param enabled               is task enabled, defaults to True
          param project               Path to project file (*.dpr),
                                        defaults to None
        """
        super(dccTaskItem, self).__init__(enabled)
        self._tasktype   = dccTaskItem.TASKTYPE
        self._compatWith = (dccTaskItem.TASKTYPE,)
        # new attributes
        self._attributes.append('project')
        self._attributes.append('compiler')
        self._attributes.append('cmdextra')
        self._attributes.append('bindir')
        self._attributes.append('dcudir')
        self._attributes.append('cfgfile')
        self._attributes.append('mad_patch')
        self._attributes.append('mad_mesfile')
        self._attributes.append('mad_mapfile')
        self._attributes.append('mad_extra')
        # set new attributes to None
        self.reset(False)
        self.project = project


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __str__(self) :
        """\
          string __str__(void)
          
          Called by the str() built-in function and by the print statement
          to compute the "informal" string representation of an object. This
          differs from __repr__() in that it does not have to be a valid
          Python expression: a more convenient or concise representation may
          be used instead. The return value must be a string object
          
          Returns a pseudo command line: only the most important attributes
          will be displayed. If current task is found empty, then an empty
          string will be returned.
          
          return                      pseudo command line string
        """
        if not self.isValid() :
          return ''
        else :
          return 'dcc32 %s' % (self.project,)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def isValid(self) :
        """\
          bool isValid(void)
          
          Tells whether or not the task is "valid", id est, if mandatory
          attributes are set; an empty taskItem is not valid, but an
          invalid taskItem may not be empty ;)
          
          return                      is task valid
        """
        #~ if self.isEmpty() :
        if not self :
          return False
        elif not self._isValidString('compiler') :
          return False
        elif not self._isValidString('project') :
          return False
        else :
          return True


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def toCmdLine(self) :
        """\
          object toCmdLine(void)
          
          Returns a command line as an asqCmdLine instance. If current task
          is found empty or invalid, then an empty command is returned.
          
          return                      asqCmdLine instance
        """
        result = super(dccTaskItem, self).toCmdLine()
        # empty task returns empty command line
        if not self.isValid() : return result

        # delphi compiler
        result.appendQuoted(self.compiler)
        # project to compile
        result.appendQuoted(self.project)
        
        # binary output directory
        if self.bindir is not None :
          result.appendQuoted('-E%s' % (self.bindir,))
        # dcu output directory
        if self.dcudir is not None :
          result.appendQuoted('-N%s' % (self.dcudir,))
        # extra compiler parameters
        if self.cmdextra is not None :
          result.append(self.cmdextra)
          
        return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def patchCommand(self, target) :
        """\
          object patchCommand(string target)
          
          Returns a command line as an asqCmdLine instance. Apply madExcept
          patch to given 'target' file.
          
          param target                binary to patch, full path
          return                      asqCmdLine instance
        """
        result = asqCmdLine()
        
        if target is None or self.mad_patch is None :
          return result
        
        result.appendQuoted(self.mad_patch)
        result.appendQuoted(target)
        if self.mad_mesfile is not None :
          result.appendQuoted(self.mad_mesfile)
        if self.mad_mapfile is not None :
          result.appendQuoted(self.mad_mapfile)
        if self.mad_extra   is not None :
          result.append(self.mad_extra)
       
        return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def expandPaths(self) :
        """\
          void expandPaths(void)
          
          Permanently replace task's file paths with "normalized" versions,
          "expandvars()", "normpath()" and "normcase()" are applied to paths.
        """
        super(dccTaskItem, self).expandPaths()
        self.compiler  = self._normPath(self.compiler)
        self.project   = self._normPath(self.project)
        self.bindir    = self._normPath(self.bindir)
        self.dcudir    = self._normPath(self.dcudir)
        self.cfgfile   = self._normPath(self.cfgfile)
        
        self.mad_patch   = self._normPath(self.mad_patch)
        self.mad_mesfile = self._normPath(self.mad_mesfile)
        self.mad_mapfile = self._normPath(self.mad_mapfile)



  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def completePaths(self, base) :
        """\
          void completePaths(string base)
          
          Permanently replace object's paths with absolute versions, using
          'base' value as base dir (thus the name ;).
          
          param base                  base for absolute path
        """
        super(dccTaskItem, self).completePaths(base)
        self.compiler  = self._completePath(base, self.compiler)
        self.project   = self._completePath(base, self.project)
        self.bindir    = self._completePath(base, self.bindir)
        self.dcudir    = self._completePath(base, self.dcudir)
        self.cfgfile   = self._completePath(base, self.cfgfile)
        
        self.mad_patch   = self._completePath(base, self.mad_patch)
        self.mad_mesfile = self._completePath(base, self.mad_mesfile)
        self.mad_mapfile = self._completePath(base, self.mad_mapfile)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # class dccTaskItem - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


class dccTaskList(taskList) :
    """\
      A list of dccTaskItems, the list defines "default" attributes that each
      single item can override; if an item doesn't set an attribute, then the
      list's values will be used.
      
      
      constructor dccTaskList(bool enabled)
      
      object    appendNewTask(bool enabled, string project)
      
      property string compiler  : path to compiler
      property string cmdextra  : extra compiler params
      property string bindir    : binary (exe/dll) ouput directory
      property string dcudir    : dcu output directory
      property string cfgfile   : path to project configuration file
      
      property string mad_patch   : path to madExceptPatch utility
      property string mad_mesfile : path to .mes file
      property string mad_mapfile : path to .map file
      property string mad_extra   : extra parameters
    """
    
    
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Constructor - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    
    def __init__(self, enabled=True) :
        """\
          constructor dccTaskList(bool enabled)
          
          Create an empty list of dccTaskItems. If a whole taskList is
          disabled, then NONE of its task items will be done.
          
          param enabled               is tasklist enabled, defaults to True
        """
        super(dccTaskList, self).__init__(enabled)
        self._tasktype   = dccTaskItem.TASKTYPE
        self._compatWith = (dccTaskItem.TASKTYPE,)
        # new attributes
        self._attributes.append('compiler')
        self._attributes.append('cmdextra')
        self._attributes.append('bindir')
        self._attributes.append('dcudir')
        self._attributes.append('cfgfile')
        self._attributes.append('mad_patch')
        self._attributes.append('mad_mesfile')
        self._attributes.append('mad_mapfile')
        self._attributes.append('mad_extra')
        # set new attributes to None
        self.reset(False)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def appendNewTask(self, enabled=False, project=None) :
        """\
          object appendNewTask(bool enabled, string project)
          
          Returns a brand new task, which has been added to taskList.
          Somehow, it looks a little bit like a "factory" function.
          BEWARE: If no parameter is set, then an EMPTY AND DISABLED
          taskItem will be yield.
          
          param enabled               is task enabled,
                                        defaults to False
          param project               Path to project file (*.dpr),
                                        defaults to None
          return                      a new dccTask Item
        """
        result = dccTaskItem(enabled, project)
        self.append(result)
        return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # class dccTaskList - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -