#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
A delphi building utility: Wrapper around delphi compiler, with custom
config file support.

Since
  Python 2.2

Classes
  class         dccCommand(void) extends basicCommand
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.3 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  os
import  shutil

import  py_netasq.commonlib.asqDebug            as asqDebug
import  py_netasq.commonlib.asqPath             as asqPath

from    py_netasq.commonlib.asqWin              import windowsCmd
from    py_netasq.delphi.building.basicCommand  import basicCommand
from    py_netasq.delphi.building.dccTasks      import dccTaskItem

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if __name__ == '__main__' :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

# About *.cfg FILE (taken from "DELPHI in a nutshell") :
#
# The compiler gets options from four places. It checks all four sources,
# in order, so LATER sources can override earlier ones:
#
# - the global configuration file, 'dcc32.cfg', is in Delphi's 'bin'
#  directory. You can modify this file to store options that apply to all
#  projects. For example, you might use -u to list directories where
#  Delphi should look for *.dcu files.
#
# - the local configuration file, 'dcc32.cfg', is in the CURRENT directory.
#
# - the project configuration file resides in the same directory as the
#  project source file. Its filename is the project filename, with the
#  '.cfg' extension.
#  Note that the IDE automatically creates a '.cfg' file for each project,
#  saving the project's options in that configuration file.
#
# - you can override the options in the configuration fileS by specifying
#  additional or different options on the command line. The options can
#  precede or follow the source filename.
#
# If a config file is provided, then we need to back up any local file found.

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


class dccCommand(basicCommand) :
    """\
      Wrapper around delphi compiler, with custom config file support.
      Settings have to be set through object's "task" (class dccTaskItem)
      attribute.
      
      
      constructor dccCommand(void)
      
      void      debug(int tab)
      var       run(object task, string codepage)
      
      property object  task
    """
    
    # format string : "<project>.cfg"
    CONFIG_FILE = '%s.cfg'
    # format string : "_<project>_cfg.bak"
    BACKUP_FILE = '_%s_cfg.bak'
    # extension resolution order for binary file, first 'exe', then 'dll' ...
    BIN_EXTS    = ('.exe', '.dll', '.bpl')

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Constructor - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __init__(self) :
        """\
          constructor dccCommand(void)
          
          Create a new dccCommand object.
        """
        super(dccCommand, self).__init__()
        self._task     = dccTaskItem(False, None)
        # to be certain that properties are available in object's scope
        self.__initCwd = None
        self.__prjPath = None
        self.__prjName = None
        self.__cfgCopy = None
        self.__cfgBack = None


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Special methods - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def debug(self, tab=0) :
        """\
          void debug(int tab)
          
          Prints out the main attributes values. Each string to print, will
          be prepended with 'tab' space characters.
          
          param tab                   number of char to prepend,
                                       defaults to 0
        """
        prefix  = ' ' * tab
        
        print asqDebug.debugValue('currCwd', os.getcwd()   , tab)
        print asqDebug.debugValue('initCwd', self.__initCwd, tab)
        print asqDebug.debugValue('prjPath', self.__prjPath, tab)
        print asqDebug.debugValue('cfgCopy', self.__cfgCopy, tab)
        print asqDebug.debugValue('cfgBack', self.__cfgBack, tab)
        
        super(dccCommand, self).debug(tab)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _setUp(self) :
        """\
          void _setUp(void)
          
          Since the delphi compiler has some nasty limitations, a few things
          have to be set up to overcome those pitfall :
          
          - Paths found in project's dpr are RELATIVE to that peculiar file,
           we have to execute the build command FROM THE PROJECT'S DIRECTORY.
          - If a custom config file (cfgFile) is supplied it must be placed
           as '<project>.cfg' in project's directory, overridding the
           project's one (if existing), and any 'dcc32.cfg' file.
          - append task's log file to listeners list
        """
        self._resolvePaths()
        
        if self.__prjPath is not None :
          # backup path to current working directory
          self.__initCwd = os.getcwd()
          # go to project's dir, project's unit paths should resolve
          os.chdir(self.__prjPath)
        
        # maybe given custom cfg file is actually local cfg file !!
        if self._task.cfgfile is not None                                \
           and self._task.cfgfile != self.__cfgCopy :
          self._backupConfigFile()
         
        if self._task.bindir is not None :
          asqPath.makeDir(self._task.bindir)
        if self._task.dcudir is not None :
          asqPath.makeDir(self._task.dcudir)
         
        super(dccCommand, self)._setUp()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  
    def _run(self, command, codepage) :
        """\
          var _run(string command, string codepage)
          
          Execute given 'command' and return shell status.
          
          param command               command to execute
          param codepage              command output codepage
          return                      shell status
        """
        result = super(dccCommand, self)._run(command, codepage)
        if result is None :
          return self._applyMadPatch(codepage)
        else :
          return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _tearDown(self) :
        """\
          void _tearDown(void)
          
          Clean up the mess caused by the '_setUp' procedure;
          - delete copied custom config file (if any)
          - restore local config file (if any)
          - change back working directory.
          - remove task's log file from listeners list
        """
        if self.__prjPath is not None :
          # restore initial directory path
          os.chdir(self.__initCwd)
         
        # maybe given custom cfg file is actually local cfg file !!
        if (self._task.cfgfile is not None)                              \
           and (self._task.cfgfile != self.__cfgCopy) :
          self._restoreConfigFile()
         
        super(dccCommand, self)._tearDown()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _applyMadPatch(self, codepage) :
        """\
          var _applyMadPatch(string codepage)
          
          Apply madexcept patch to the newly created binary file; Since we
          do not know the binary extension, we try to guess it: resolution
          order is defined by the dccCommand.BIN_EXTS constant
          
          param codepage              command output codepage
          return                      shell status
        """
        binfile = self._findBinaryName(dccCommand.BIN_EXTS)
        command = self._task.patchCommand(binfile)
        
        if 0 < len(command) :
          return windowsCmd.run(self, command, codepage)
        else :
          return None


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _backupConfigFile(self) :
        """\
          void _setupConfigFile(void)
          
          If a custom config file is provided, it must be copied to project's
          directory; To avoid file over writting, existing local config file
          is renamed to dccCommand.BACKUP_FILE.
        """
        if os.path.isfile(self.__cfgCopy) :
          os.rename(self.__cfgCopy, self.__cfgBack)
          
        shutil.copy(self._task.cfgfile, self.__cfgCopy)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _restoreConfigFile(self) :
        """\
          void _restoreConfigFile(void)
          
          If a custom config file is provided, then it might have been copied
          to project's directory; It will be deleted, and backed up local 
          config (if any) will be restored.
        """
        if os.path.isfile(self.__cfgCopy) :
          os.remove(self.__cfgCopy)
        if os.path.isfile(self.__cfgBack) :
          os.rename(self.__cfgBack, self.__cfgCopy)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _findBinaryName(self, exts) :
        """\
          var _findBinaryName(list exts)
          
          Try to guess binary file extension; each extensions of the given
          list will be used until a matchin file is found. A fully qualified
          path is then returned (using self.task.bindir and self.__prjName),
          else None value is returned.
          
          param exts                  file extensions to check
          return                      binary full path, or None
        """
        for item in exts :
          result = r'%s\%s%s' % (self._task.bindir, self.__prjName, item)
          if os.path.isfile(result) : return result
        else :
          return None


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _resolvePaths(self) :
        """\
          void _resolvePaths(void)
          
          Set some useful attributes, such as __prjPath, __cfgCopy
          and __cfgBack.
        """
        project = self._task.project
        
        if not isinstance(project, basestring) :
          self.__prjPath = None
          self.__cfgCopy = None
          self.__cfgBack = None
        else :
          prjPath, prjName = os.path.split(project)
          # config and project files must share the same
          # name (minus extensions)
          prjName = os.path.splitext(prjName)[0]
          # cfgCopy = path/to/project/<project>.cfg
          # cfgBack = path/to/project/_<project>_cfg.bak
          self.__cfgCopy = dccCommand.CONFIG_FILE % prjName
          self.__cfgBack = dccCommand.BACKUP_FILE % prjName
          self.__prjPath = prjPath
          self.__prjName = prjName
          self.__cfgCopy = os.path.join(prjPath, self.__cfgCopy)
          self.__cfgBack = os.path.join(prjPath, self.__cfgBack)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # class dccCommand  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -