#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
Define a "CVS Task item" list used by the building utility.

Since
  Python 2.2

Classes
  class         cvsTaskItem(bool enabled, string module) extends taskItem
  class         cvsTaskList(bool enabled)                extends taskList
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.10 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

from    py_netasq.delphi.building.basicTasks    import taskItem, taskList

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if __name__ == '__main__' :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


class cvsTaskItem(taskItem) :
    """\
      A CVS task: a cvs command, which can be either 'checkout', 'export'
      or 'update'.
      
      
      constructor cvsTaskItem(bool enabled, string module)
      
      bool      isValid(void)
      object    toCmdLine()
      
      void      expandPaths(void)
      void      completePaths(string base)
      
      property string module     : module in cvs tree
      property string client     : cvs client (exe) to use
      property string command    : cvs command to perform
      property string cmdextra   : extra cvs CLIENT parameters
      property string cvsroot    : CVSROOT
      property string cvsrsh     : CVS_RSH environement value
      property string destdir    : retrieved files destination
      property string notafter   : revision date limit
      property string tag        : module tag
    """
    
    TASKTYPE = "cvsTask"
    
    # CVS tasks can only RETRIEVE files
    ALLOWED_COMMANDS = ('checkout', 'export', 'update')
    # checkout or export need tag/branch
    DEFAULT_CVS_TAG  = 'HEAD'
    
    
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Constructor - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    
    def __init__(self, enabled=True, module=None) :
        """\
          constructor cvsTaskItem(bool enabled, string module)
          
          Create a new cvs task. all properties are set to 'None' value.
          
          param enabled               is task enabled, defaults to True
          param module                module in cvs tree, defaults to None
        """
        super(cvsTaskItem, self).__init__(enabled)
        self._tasktype   = cvsTaskItem.TASKTYPE
        self._compatWith = (cvsTaskItem.TASKTYPE,)
        # new attributes
        self._attributes.append('module')
        self._attributes.append('client')
        self._attributes.append('command')
        self._attributes.append('cmdextra')
        self._attributes.append('cvsroot')
        self._attributes.append('cvsrsh')
        self._attributes.append('destdir')
        self._attributes.append('notafter')
        self._attributes.append('tag')
        # set new attributes to None
        self.reset(False)
        self.module = module


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __str__(self) :
        """\
          string __str__(void)
          
          Called by the str() built-in function and by the print statement
          to compute the "informal" string representation of an object. This
          differs from __repr__() in that it does not have to be a valid
          Python expression: a more convenient or concise representation may
          be used instead. The return value must be a string object
          
          Returns a pseudo command line: only the most important attributes
          will be displayed. If current task is found empty, then an empty
          string will be returned.
          
          return                      pseudo command line string
        """
        if not self.isValid() :
          return ''
        elif self.tag is None :
          return 'cvs %s %s'    % (self.command, self.module)
        else :
          return 'cvs %s %s %s' % (self.command, self.module, self.tag)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def isValid(self) :
        """\
          bool isValid(void)
          
          Tells whether or not the task is "valid", id est, if mandatory
          attributes are set; an empty taskItem is not valid, but an
          invalid taskItem may not be empty ;)
          
          return                      is task valid
        """
        #~ if self.isEmpty() :
        if not self : # empty task
          return False
        elif not self._isValidString('client') :
          return False
        elif not self._isValidString('command') :
          return False
        elif not self._isValidString('module') :
          return False
        else :
          return True


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def toCmdLine(self) :
        """\
          object toCmdLine(void)
          
          Returns a command line as an asqCmdLine instance. If current task
          is found empty or invalid, then an empty command is returned.
          
          return                      asqCmdLine instance
        """
        result = super(cvsTaskItem, self).toCmdLine()
        # empty task returns empty command line
        if not self.isValid() : return result
        
        # cvs executable
        result.appendQuoted(self.client)
        # fully qualified cvs root
        if self.cvsroot is not None :
          result.append('-d')
          result.appendQuoted(self.cvsroot)
        # cvs extra parameters
        if self.cmdextra is not None :
          result.append(self.cmdextra)
        # cvs command
        result.append(self.command)
        # retrieve revisions upto 'notAfter' date (not after)
        if self.notafter is not None :
          result.append('-D')
          result.appendQuoted(self.notafter)
          
        if bool(self.tag) : 
          result.append('-r')
          result.append(self.tag)
        elif self.command in ('update', 'checkout') :
          # reset sticky tag
          result.append('-A')
        
        if 'update' == self.command :
          # create directory but skip empty ones
          # module name should not be present
          result.append('-dP')
        else :
          result.appendQuoted(self.module)
          
        return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def expandPaths(self) :
        """\
          void expandPaths(void)
          
          Permanently replace task's file paths with "normalized" versions,
          "expandvars()", "normpath()" and "normcase()" are applied to paths.
        """
        super(cvsTaskItem, self).expandPaths()
        self.client   = self._normPath(self.client)
        self.destdir  = self._normPath(self.destdir)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def completePaths(self, base) :
        """\
          void completePaths(string base)
          
          Permanently replace object's paths with absolute versions, using
          'base' value as base dir (thus the name ;).
          
          param base                  base for absolute path
        """
        super(cvsTaskItem, self).completePaths(base)
        self.client   = self._completePath(base, self.client)
        self.destdir  = self._completePath(base, self.destdir)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # class cvsTaskItem - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


class cvsTaskList(taskList) :
    """\
      A list of cvsTaskItems, the list defines "default" attributes that each
      single item can override; if an item doesn't set an attribute, then the
      list's values will be used.
      
      
      constructor cvsTaskList(bool enabled)
      
      object    appendNewTask(bool enabled, string module)
      
      property string client    : cvs client (exe) to use
      property string command   : cvs command to perform
      property string cmdextra  : extra cvs CLIENT parameters
      property string cvsroot   : CVSROOT
      property string cvsrsh    : CVS_RSH environement value
      property string destdir   : retrieved files destination
      property string notafter  : revision date limit
      property string tag       : module tag
    """
   
   
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Constructor - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    
    def __init__(self, enabled=True) :
        """\
          constructor cvsTaskList(bool enabled)
          
          Create an empty list of cvsTaskItems. If a whole taskList is
          disabled, then NONE of its task items will be done.
          
          param enabled               is tasklist enabled, defaults to True
        """
        super(cvsTaskList, self).__init__(enabled)
        self._tasktype   = cvsTaskItem.TASKTYPE
        self._compatWith = (cvsTaskItem.TASKTYPE,)
        # new attributes
        self._attributes.append('client')
        self._attributes.append('command')
        self._attributes.append('cmdextra')
        self._attributes.append('cvsroot')
        self._attributes.append('cvsrsh')
        self._attributes.append('destdir')
        self._attributes.append('notafter')
        self._attributes.append('tag')
        # set new attributes to None
        self.reset(False)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def appendNewTask(self, enabled=False, module=None) :
        """\
          object appendNewTask(bool enabled, string module)
          
          Returns a brand new task, which has been added to taskList.
          Somehow, it looks a little bit like a "factory" function.
          BEWARE: If no parameter is set, then an EMPTY AND DISABLED
          taskItem will be yield.
          
          param enabled               is task enabled, defaults to False
          param module                module in cvs tree, defaults to None
          return                      a new cvsTask Item
        """
        result = cvsTaskItem(enabled, module)
        self.append(result)
        return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # class cvsTaskList - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -