#!/usr/bin/env python

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
A delphi building utility: Compiler options list wrapper.

Since
  Python 2.2

Classes
  class dccOptions(void)
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.1 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import re
import string
import types

# import needed classes from units.
from netasq.delphi.building.dccDirectives import dccDirectives

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if __name__ == '__main__' :
  print __file__
  print __doc__

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

# Delphi compiler options list
# 
#  With no value (switches) :
#   B                     Build all units
#   H                     Output hint messages
#   M                     Make modified units
#   P                     Look for 8.3 file names also
#   Q                     Quiet compile
#   W                     Output warning messages
#   Z                     Disable implicit compilation option
# 
#  Single value (last setting override others)
#   E<path>               EXE output directory
#   F<offset>             Find errorneous statement at offset
#   K<addr>               Set image base addr
#   N<path>               DCU output directory
#   TX<ext>               change compiled file extension
# 
#  Multiple values (act as switches)
#   C[C|G]                application type ('C' : console, 'G' : GUI)
#   G[D|P|S]              output map file
#   J[P]                  Generate .obj file
#   V[N|R]                Debug
#  
#  Lists
#   A(<unit>=<alias>;)*   Set unit alias
#   D(<conditional>;)*    Define conditionals
#   I<path>               Include directories
#   LE<path>              BPL output directory (package files)
#   LN<path>              DCP output directory (package sources)
#   LU<package>           Link using runtime package
#   O<path>               Object files (.obj) directories
#   R<path>               Resource directories
#   U<path>               Unit directories
#   
#  Directives
#   $<directive>          Compiler directives
# 

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


class dccOptions(object) :
    """\
      Compiler options list wrapper. Set all needed path for a project.
      Beware: this is a new-style class (derivated from 'object').
      
      constructor dccOptions(void)
      
      void fromLines(list lines)
      list toLines(void)
      
      void reset(void)
      void debug(void)
    """

    List = (
        '$'  , # Compiler directive (switch)
        'A'  , # Set unit alias
        'B'  , # Build all units
        'M'  , # Make modified units
        'CC' , # Target type Console
        'CG' , # Target type GUI
        'H'  , # Output hint messages
        'D'  , # Define conditionals
        'P'  , # look for 8.3 file names
        'GD' , # Detailed map file
        'GP' , # Map file with publics
        'GS' , # Map file with segments
        'JP' , # Generate C++ .obj file
        'J'  , # Generate C .obj file
        'SK' , # (little hack) Stack size : $M<min>[,<max>]
        'K'  , # Set image base addr
        'E'  , # EXE output directory
        'N'  , # DCU output directory
        'U'  , # Unit directories
        'LE' , # BPL output directory
        'LN' , # DCP output directory
        'LU' , # Link using runtime package
        'O'  , # Object directories
        'I'  , # Include directories
        'R'  , # Resource directories
        'TX' , # Change compiled file extension
        'VN' , # Generate C++ debug info
        'VR' , # Generate remote debug (RSM) file
        'V'  , # Debug information
        'Q'  , # Quiet compile
        'W'  , # Output warning messages
        'F'  , # Find error
        'Z'  , # Output 'never build' DCPs
      )

    # default options config
    DEFAULT = [
        '-$'+ dccDirectives.DEFAULT ,
        '-CG',
        '-AWinTypes=Windows;WinProcs=Windows;DbiTypes=BDE;DbiProcs=BDE;DbiErrs=BDE;',
        '-M',
        '-Q',
        '-H',
        '-K$00400000',
        '-$M$00004000,$00100000',
        '-W',
        '-W-UNSAFE_TYPE',
        '-W-UNSAFE_CODE',
        '-W-UNSAFE_CAST'
      ]

    # specific options config: Debug
    DEBUG   = [
        '-$'+ dccDirectives.DEBUG ,
        '-CG',
        '-AWinTypes=Windows;WinProcs=Windows;DbiTypes=BDE;DbiProcs=BDE;DbiErrs=BDE;',
        '-M',
        '-Q',
        '-H',
        '-K$00400000',
        '-$M$00004000,$00100000',
        '-GD',
        '-W',
        '-W-UNSAFE_TYPE',
        '-W-UNSAFE_CODE',
        '-W-UNSAFE_CAST'
      ]

    # specific options config: Build
    BUILD   = [
        '-$'+ dccDirectives.BUILD ,
        '-CG',
        '-AWinTypes=Windows;WinProcs=Windows;DbiTypes=BDE;DbiProcs=BDE;DbiErrs=BDE;',
        '-B',
        '-Q',
        '-H',
        '-K$00400000',
        '-$M$00004000,$00100000',
        '-W',
        '-W-UNSAFE_TYPE',
        '-W-UNSAFE_CODE',
        '-W-UNSAFE_CAST'
      ]


    # option line prefixes
    Prefixes  = ('-', '/')

    # A type check is made upon each attribute setting
    # otions : booleans, flags
    _BoolAttr  = (
        'CC', 'CG', 'GD', 'GP', 'GS', 'JP', 'VN', 'VR',
        'B' , 'H' , 'J' , 'M' , 'P' , 'Q' , 'V' , 'Z'
      )
    # options : lists
    _ListAttr  = ( 'LU', 'A', 'D', 'I', 'U', 'O', 'R', 'SK', 'W' )
    # others options are just handled as strings...

    # -JP allow two modifiers, that can be combined
    _JP_MODIFIERS = ('N', 'E', 'NE', 'EN')


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Constructor - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __init__(self) :
        """\
          constructor dccOptions(void)
          
          Create a new wrapper around Delphi compiler options list. This
          class does guarantee option values consistency.
        """
        self.directives = dccDirectives()
        self.reset()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Special methods - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __setattr__(self, name, value) :
        """\
          void __setattr__(string name, var value)
          
          Called when an attribute assignment is attempted. This is called
          instead of the normal mechanism (i.e. store the value in the
          instance dictionary). name is the attribute name, value is the
          value to be assigned to it.
          WARNING : this function is called whether attribute actually
          exists or not !
          
          param  name                 Attribute name
          param  value                Attribute value
        """
        if name in dccOptions.List :
          self._setOption(name, value)
        else :
          object.__setattr__(self, name, value)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def fromList(self, strlist) :
        """\
          void fromList(list strlist)
          
          If a compiler option is recognized among list, then according
          attribute is set.
          e.g. fromLines(['-CC', '-H', '-$A8,C+'])
           => self.CC = True (self.CG= False)
              self.H  = True
              self.directives.A = 8
              self.directives.C = True
          
          param strlist               list of strings to parse
        """
        buffer = ''
        value  = None

        for line in self._filterLines(strlist) :
          buffer = line.upper()
          for item in dccOptions.List :
            if buffer.startswith(item) :
              value = line[len(item):]
              if 0 == cmp('$', item) :
                self._registerDirective(value)
              elif item in dccOptions._ListAttr :
                getattr(self, item).append(value)
              # take care of 'JP' option !
              # item in dccOptions._BoolAttr and 0 < len(value)
              # => bool(value) = True
              elif 0 == len(value) and item in dccOptions._BoolAttr :
                self._setOption(item, True)
              else :
                self._setOption(item, value)
              break


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def toList(self) :
        """\
          list toList(void)
          
          Generate compiler options configuration (a kind of representation
          of current dccOptions instance) as a list of strings.
          
          return                      string list "representation" of object
        """
        result = list()
        for name in dccOptions.List :
          if 0 == cmp('$', name) :
            self._appendDirectives(result)
          elif 0 == cmp('SK', name) :
            self._appendStackSize(result)
          elif 0 == cmp('JP', name) :
            self._appendValue(result, name)
          elif name in dccOptions._BoolAttr :
            self._appendBool(result, name)
          elif name in dccOptions._ListAttr :
            self._appendList(result, name)
          else :
            self._appendValue(result, name)
        return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def reset(self) :
        """\
          void reset(void)
          
          Set compiler options to empty/default values. For each attribute
          'None' will be converted to a logical default value.
            booleans : None  => False
            Lists    : None  => []
            Others   : None  => None (compiler flag will not appear)
        """
        self.directives.reset()
        for item in dccOptions.List :
          setattr(self, item, None)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def debug(self) :
        """\
          void debug(void)
          
          Debug utility, prints out option configuration.
        """
        buffer = self.toLines()
        for item in buffer : print item


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _registerDirective(self, S) :
        """\
          void _extractDirective(string S)
          
          Extract compiler directive values, OR memory stack size settings;
          "Runtime type info" and stack size do share the same prefix '-$M',
          we have to look at the end of the string to tell. The according
          object attribute will be then set.
          If given string is neither a stack definition, nor a valid directive
          list, then an exception may be raised.
          
          param S                     subject string
        """
        pattern = re.compile(r'^M(\$?\d+)(?:,(\$?\d+))?', re.I)
        if not pattern.match(S) :
          self.directives.fromString(S)
        else :
          # re.findall will return [ (<first group>, <second group>) ]
          for match in pattern.findall(S) :
            self.SK = list( filter(lambda str : 0 < len(str), match) )


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _appendDirectives(self, list) :
        """\
          void _appendDirectives(list list)
          
          Append compiler directives to list, directives are append as a
          single string.
          
          param list                  list of strings
        """
        assert list is not None, "list not assigned"
        list.append( '-$%s' % str(self.directives) )


    def _appendStackSize(self, list) :
        """\
          void _appendStackSize(list list)
          
          Append stack size defintion to list; this compiler option may not
          be confused with the "-$M" directive switch.
          
          param list                  list of strings
        """
        assert list is not None, "list not assigned"
        if 0 < len(self.SK) :
          list.append( '-$M%s' % string.join(self.SK, ',') )


    def _appendBool(self, list, name) :
        """\
          void _appendBool(list list, string name)
          
          Append option string representation to list, for boolean options
          (switches), only those which are set to True must appear.
          
          param list                  list of strings
          param name                  compiler option name
        """
        assert list is not None, "list not assigned"
        if getattr(self, name) :
          list.append( '-%s' % (name,) )


    def _appendList(self, list, name) :
        """\
          void _appendList(list list, string name)
          
          Append option string representation to list, for lists options
          (generally search paths), each item of the list will be added to
          list as a single item.
          
          param list                  list of strings
          param name                  compiler option name
        """
        assert list is not None, "list not assigned"
        for item in getattr(self, name) :
          list.append( '-%s%s' % (name, str(item)) )


    def _appendValue(self, list, name) :
        """\
          void _appendValue(list list, string name)
          
          Append option string representation to list, "others" options
          are added to list if their value differ from 'None'.
          
          param list                  list of strings
          param name                  compiler option name
        """
        assert list is not None, "list not assigned"
        value  = getattr(self, name)
        if None != value :
          list.append( '-%s%s' % (name, str(value)) )


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _stripPrefix(self, S) :
        """\
          string _stripPrefix(string S)
          
          Strip option prefix from beginning of given string. An empty string
          will be returned if no prefixes do match. Hopefully prefixes are
          case insensitive.
          
          param S                     subject string
          return                      "prefix-free" string, or empty string
        """
        for prefix in dccOptions.Prefixes :
          if S.startswith(prefix) :
            return S[len(prefix):]
        else :
          return ''


    def _filterLines(self, lines) :
        """\
          list _filterLines(list lines)
          
          Filter given list of string: remove non prefixed or empty strings
          from list.
          
          param lines                 string list to filter
          return                      filtered string list
        """
        result = map(self._stripPrefix, lines)
        result = filter(lambda item : 0 < len(item), result)
        return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _setOptionSwitch(self, name, value) :
        """\
          void _setOptionSwitch(string name, var value)
          
          Some peculiar compiler option switch are mutually exclusive: if
          one is set to True, then the other must be set to False. If one
          is set to False, then the other remains unmodified: both can be
          set to False, and thus any of them won't appear in the list
          returned by 'self.toLines()'
          
          param name                  compiler option name
          param value                 compiler option value
        """
        if not value :
          pass
        elif 0 == cmp( 'B', name) :
          object.__setattr__(self,  'M', False)
        elif 0 == cmp( 'M', name) :
          object.__setattr__(self,  'B', False)
        elif 0 == cmp('CC', name) :
          object.__setattr__(self, 'CG', False)
        elif 0 == cmp('CG', name) :
          object.__setattr__(self, 'CC', False)
        elif 0 == cmp('JP', name) :
          object.__setattr__(self,  'J', False)
        elif 0 == cmp( 'J', name) :
          # JP IS NOT A FRIGGIN BOOLEAN FLAG
          object.__setattr__(self, 'JP', None)
        elif 0 == cmp('GD', name) :
          object.__setattr__(self, 'GP', False)
          object.__setattr__(self, 'GS', False)
        elif 0 == cmp('GP', name) :
          object.__setattr__(self, 'GD', False)
          object.__setattr__(self, 'GS', False)
        elif 0 == cmp('GS', name) :
          object.__setattr__(self, 'GD', False)
          object.__setattr__(self, 'GP', False)


    def _setOptionJP(self, value) :
        """\
          var _setOptionJP(var value)
          
          'JP' option is somewhat a little more special, basically it is
          handled as a switch: if set then 'J' is disabled; Alas 'JP' allows
          two options : 'N' and/or 'E' (and thus must be handled as a
          'string' option).
          
          self.J  = True and self.JP = ''    => self.J = True  , self.JP = None
          self.J  = True and self.JP = True  => self.J = False , self.JP = ''
          self.JP = True and self.J  = True  => self.J = True  , self.JP = None
          self.J  = True and self.JP = 'N'   => self.J = False , self.JP = 'N'
          
          self.JP = 'X'                      => self.JP = False
          
          param value                 compiler option value
        """
        if value is True :
          self._setOptionSwitch('JP', True)
          return ''
        elif value in dccOptions._JP_MODIFIERS :
          self._setOptionSwitch('JP', True)
          return value
        else :
          return None


    def _setOption(self, name, value) :
        """\
          void _setOption(string name, var value)
          
          Set option value, the given 'value' type is checked to avoid type
          mismatch.
          
          param name                  compiler option name
          param value                 compiler option value
        """
        valType = type(value)
        # dirty little hack to handle JP special attributes (curse !)
        if 0 == cmp('JP', name) :
          value = self._setOptionJP(value)
        elif name in dccOptions._BoolAttr :
          value = bool(value)
          self._setOptionSwitch(name, value)
        elif name in dccOptions._ListAttr :
          if valType is types.ListType :
            pass
          elif valType is types.NoneType :
            value = list()
          else :
            raise TypeError, "Type mismatch : attribute '%s' is list" % (name,)
         
        object.__setattr__(self, name, value)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # class dccOptions  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -