#!/usr/bin/env python

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
A delphi building utility: Compiler directives list wrapper.

Since
  Python 2.2

Classes
  class dccDirectives(void)
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.1 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import types
import string

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if __name__ == '__main__' :
  print __file__
  print __doc__

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

# About delphi compiler directives
#
# Be aware that infos about Delphi compiler directives are scarce and
# sometimes hardly accurate. Often command line switches are confused with
# infile switches: but for some special case, those do not mean the same
# thing
#

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class dccDirectives(object) :
    """\
      Compiler directives list wrapper. All compiler directives can be seen
      as simple switches.
      
      Runtime type info directive and stack size definition seem to use the
      same identifier, respectively $M[+|-] and $M[minStackSize, maxStackSize]
      Stack size definition is currently not supported.
      
      This class implements the __str__ function.
      Beware: this is a new-style class (derivated from 'object')
      
      constructor dccDirectives(void)
      
      void fromString(string S)
      string toString(void)
      
      void reset(void)
    """

    List = (
        'A' , # Aligned record fields; defaults : 8 or True
        'B' , # Full boolean Evaluation; defaults : False
        'C' , # Evaluate assertions at runtime; defaults : True
        'D' , # Debug information; defaults : True
        'E' , # unknow; defaults : False
        'F' , # Force far procedure calls; defaults : False
        'G' , # Use imported data references; defaults : True
        'H' , # Use long strings by default; defaults : True
        'I' , # I/O checking; defaults : True
        'J' , # Writeable structured consts; defaults : False
        'K' , # Use smart callbacks; defaults : False
        'L' , # Local debug symbols; defaults : True
        'M' , # Runtime type info; defaults : False
        'N' , # Numeric coprocessor support; defaults : True
        'O' , # Optimization; defaults : True
        'P' , # Open string params; defaults : True
        'Q' , # Integer overflow checking; defaults : False
        'R' , # Range checking; defaults : False
        'S' , # Generate stack-overflow checking code; defaults : False
        'T' , # Typed @ operator; defaults : False
        'U' , # Pentium(tm)-safe divide; defaults : False
        'V' , # Strict var-strings; defaults : True
        'W' , # Generate stack frames; defaults : False
        'X' , # Extended syntax; defaults : True
        'Y' , # Symbol reference info; defaults : True
        'Z' , # Minimum size of enum types; defaults : 1 or False
      )

    # default directives values
    DEFAULT = 'A8,B-,C+,D+,E-,F-,G+,H+,I+,' \
              'J-,K-,L+,M-,N+,O+,P+,Q-,R-,' \
              'S-,T-,U-,V+,W-,X+,Y+,Z1'

    # debug directives
    DEBUG   = 'A8,B-,C+,D+,E-,F-,G+,H+,I+,' \
              'J-,K-,L+,M+,N+,O-,P+,Q+,R+,' \
              'S+,T-,U-,V+,W+,X+,Y+,Z1'

    # build release directives
    BUILD   = 'A8,B-,C-,D-,E-,F-,G+,H+,I+,' \
              'J-,K-,L-,M-,N+,O+,P+,Q-,R-,' \
              'S-,T-,U-,V+,W-,X+,Y-,Z1'

    # value separator
    Separator = ','

    # make conversion a little easier
    _toString  = { False : '-', True : '+' }
    _toBoolean = { '-' : False, '+' : True }

    # Mixed 'type' directives :
    # Aligned record fields
    _A_Values  = { '-' : 1, '+' : 8, '1' : 1, '2' : 2, '4' : 4, '8' : 8 }
    # Minimum size of enum types
    _Z_Values  = { '-' : 1, '+' : 4, '1' : 1, '2' : 2, '4' : 4 }


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Constructor - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __init__(self) :
        """\
          constructor cfgSwitches(void)
          
          Create a new wrapper around Delphi Compiler directives list. This
          class provides easier access to directive's values. Directives are
          set accordingly to dccDirectives.DEFAULT.
        """
        self.fromString(dccDirectives.DEFAULT)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Special methods - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __setattr__(self, name, value) :
        """\
          void __setattr__(string name, var value)
          
          Called when an attribute assignment is attempted. This is called
          instead of the normal mechanism (i.e. store the value in the
          instance dictionary). name is the attribute name, value is the
          value to be assigned to it.
          WARNING : this function is called whether attribute actually
          exists or not !
          
          param  name                 Attribute name
          param  value                Attribute value
        """
        if name in dccDirectives.List :
          self._setDirective(name, value)
        else :
          object.__setattr__(self, name, value)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __str__(self) :
        """\
          string __str__(void)
          
          Called by the 'str()' built-in function and by the 'print' statement
          to compute the "informal" string representation of an object. This
          differs from __repr__() in that it does not have to be a valid
          Python expression: a more convenient or concise representation may
          be used instead. The return value must be a string object.
          
          Return directives representation, as found in configuration file
          e.g. "A8,B-,C+,D+,G+,H+,I+,J-,L+,M-,O+,Q-,R-,T-,U-,V+,W-,X+,Y+,Z1"
          
          return                      string representation
        """
        strlst = list()
        for item in dccDirectives.List :
          strval = self._ConvertToString(getattr(self, item))
          strlst.append( '%s%s' % (item, strval) )
        return string.join(strlst, dccDirectives.Separator)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def fromString(self, S) :
        """\
          void fromString(string S)
          
          Set attributes, directive values are extracted from given string.
          If a directive is recognized then according attribute is set.
          e.g. self.fromString('A8,B-,C+,D+,G+')
          => self.A = 8 , self.B = False, self.C = True, ...
          
          param S                     string to extract directive values from
        """
        strlst = string.split(S, dccDirectives.Separator)
        strlst = map(lambda str : str.strip(), strlst)
        strlst = filter(lambda str : 0 < len(str), strlst)
        for item in strlst :
          self._registerDirective(item)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def toString(self) :
        """\
          string toString(void)
          
          __str__ alias: Return directives representation, as found in
          configuration file. There is no peculiar need for that alias, I
          just wanted to have some kind of consistency with 'fromString'.
          
          e.g. "A8,B-,C+,D+,G+,H+,I+,J-,L+,M-,O+,Q-,R-,T-,U-,V+,W-,X+,Y+,Z1"
          
          return                      string representation
        """
        return self.__str__


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def reset(self) :
        """\
          void reset(void)
          
          Set all directives to their default values; Directives are set
          accordingly to dccDirectives.DEFAULT.
        """
        self.fromString(dccDirectives.DEFAULT)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _registerDirective(self, S) :
        """\
          void _extractSwitch(string S)
          
          If a directive is recognized then according attribute is set.
          Directive are only matched at string beginning.
          
          param S                     string to extract directive values from
        """
        buffer = S.upper()
        for item in dccDirectives.List :
          if buffer.startswith(item) :
            self._setDirective(item, S[len(item):])


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _ConvertToBoolean(self, value) :
        """\
          bool _ConvertToBoolean(var value)
          
          Convert given value to boolean; If value is not a boolean, or one
          of ('-', '+'), then a TypeError exception is raised.
          
          param  value                value to convert
          return                      boolean value
        """
        if type(value) is types.BooleanType :
          return value
        elif dccDirectives._toBoolean.has_key(value) :
          return dccDirectives._toBoolean[value]
        else :
          raise TypeError, "cannot convert [%s] to boolean" % (str(value),)


    def _ConvertToString(self, value) :
        """\
          string _ConvertToString(var value)
          
          Convert given value to boolean; If value is boolean then it is
          converted to '-' or '+' (false => '-', true => '+'), else the
          string representation, str(value), is returned.
          
          param  value                value to convert
          return                      string value
        """
        if type(value) is types.BooleanType :
          return dccDirectives._toString[value]
        else :
          return str(value)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _setDirective(self, name, value) :
        """\
          void _setDirective(string name, var value)
          
          Setter - Generic compiler directive; Given 'name' parameter must be
          one of dccDirectives.List, if the directive is a "simple" switch
          then the given 'value' has to be convertible to boolean (using
          _ConvertToBoolean).
          
          param name                  directive name
          param value                 directive value
        """
        if 0 == cmp('A', name) :
          self._setA(value)
        elif 0 == cmp('Z', name) :
          self._setZ(value)
        else :
          object.__setattr__(self, name, self._ConvertToBoolean(value))


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _setA(self, value) :
        """\
          void _setA(var value)
          
          Setter - set aligned record fields; Given 'value' must be either a
          boolean, or one of the following values: (+, -, 1, 2, 4, 8), or
          else a TypeError excpetion will be raised.
          
          In older versions of Delphi, this option could be checked on or off.
          Choosing 1 is the equivalent of off and 8 is the equivalent of on.
          
          param value                 Aligned record fields
        """
        # since True == 1, we can't mix up boolean and interger as dict key
        strval = self._ConvertToString(value)
        if dccDirectives._A_Values.has_key(strval) :
          object.__setattr__(self, 'A', dccDirectives._A_Values[strval])
        else :
          raise TypeError, "Type mismatch : attribute A"


    def _setZ(self, value) :
        """\
          void _setZ(var value)
          
          Setter - set minimum size of enum types; Given 'value' must be
          either a boolean, or one of the following values: (+, -, 1, 2, 4),
          or else a TypeError excpetion will be raised.
          
          In older versions of Delphi, this option could be checked on or off.
          Choosing 1 is the equivalent of off and 4 is the equivalent of on.
          
          param value                 Aligned record fields
        """
        # since True == 1, we can't mix up boolean and interger as dict key
        strval = self._ConvertToString(value)
        if dccDirectives._Z_Values.has_key(strval) :
          object.__setattr__(self, 'Z', dccDirectives._Z_Values[strval])
        else :
          raise TypeError, "Type mismatch : attribute Z"


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # class dccDirectives - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -