#!/usr/bin/env python

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
A delphi compiler configuration file wrapper: a configuration file can be
easily build from scratch with a few attributes sets.

Since
  Python 2.2

Classes
  class cfgFile(void)
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.1 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import os

# import needed classes from units.
from netasq.delphi.building.dccDirectives import dccDirectives
from netasq.delphi.building.dccOptions    import dccOptions

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if __name__ == '__main__' :
  print __file__
  print __doc__

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

# About Python "Universal Newline"
#
# The three major operating systems used today are Microsoft Windows,
# Apple's Macintosh OS, and the various Unix derivatives. A minor irritation
# of cross-platform work is that these three platforms all use different
# characters to mark the ends of lines in text files. Unix uses the linefeed
# (ASCII character 10), MacOS uses the carriage return (ASCII character 13),
# and Windows uses a two-character sequence of a carriage return plus a
# newline.
#
# Python's file objects can now support end of line conventions other than
# the one followed by the platform on which Python is running. Opening a
# file with the mode 'U' or 'rU' will open a file for reading in universal
# newline mode. All three line ending conventions will be translated to a
# "\n" in the strings returned by the various file methods such as read()
# and readline().

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


class cfgFile(object) :
    """\
      A delphi compiler configuration file wrapper: a configuration file can
      be easily build from scratch with a few attributes sets.
      dccOptions class defines some "profiles": DEFAULT, DEBUG, BUILD;
      simply use
       self.options.fromList( dccOptions.<profile> )
      and you're done.
      
      Beware: this is a new-style class (derivated from 'object')
      
      constructor cfgFile(void)
      
      void loadFile(string filename)
      void saveFile(string filename)
     
      void reset(void)
      
      property object options
    """

    # some words of advice to write on top of file
    Banner = [
        "; Delphi compiler (dcc32.exe) configuration file.",
        "; Automatically generated file, please to not edit"
      ]


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Constructor - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __init__(self) :
        """\
          constructor cfgFile(void)
          
          Create a delphi compiler configuration file wrapper: load from
          file, change some options and then save to a file; simple.
        """
        self.__options = dccOptions()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def loadFile(self, filename) :
        """\
          void loadFile(string filename)
          
          Load given config file into object's option list. An exception
          may be raised if there is no file found at given path.
          
          param filename              config file path
        """
        if not os.path.isfile(filename) :
          raise Exception, "couldn't read file '%s'" % (filename,)
        else :
          strlst = self._getLines(filename)
          self.__options.fromList(strlst)


    def saveFile(self, filename) :
        """\
          void saveFile(string filename)
          
          Save compiler configuration to given file; if file does actually
          not exist, then it will be created.
          
          param filename              config file path
        """
        buffer = cfgFile.Banner + self.__options.toList()
        self._setLines(filename, buffer)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def reset(self) :
        """\
          void reset(void)
          
          Set compiler options and directives to empty/default values.
        """
        self.__options.reset()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _getLines(self, filename) :
        """\
          list _getLines(string filename)
          
          Read file content until EOF, and return a list containing the lines
          thus read. Leading and trailing whitespace characters  are removed
          from each line. An Exception may be thrown if file cannot be read.
          
          Note : file is read using python "Universal Newline".
          
          param filename              config file path
        """
        target = file(filename, 'rU')
        try :
          result = map(lambda s : s.strip(), target.readlines())
        finally :
          target.close()
        return result


    def _setLines(self, filename, lines) :
        """\
          void _setLines(string filename, list lines)
          
          Write a sequence of strings to the file. For each item of the string
          list, a trailing newline character is added. An Exception may be
          thrown if file cannot be written.
          
          Note : file is written using python "Universal Newline".
          
          param filename              config file path
        """
        target = file(filename, 'wU')
        try :
          target.writelines( map(lambda s : s +'\n', lines) )
        finally :
          target.close()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _getOptions(self) :
        """\
          object _getOptions(void)
          
          Getter - get delphi compiler options wrapper.
          
          return                      an instance of dccOptions
        """
        return self.__options


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    # read only
    options = property(
                doc  = "Delphi compiler options and directives",
                fget = _getOptions,
                fset = None,
                fdel = None )


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # class cfgFile - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -