#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
A build file syntax validator.

Since
  Python 2.3

Classes
  class         buildFileValidator(void) extends buildFileParser
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.2 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  ConfigParser
import  os

import  py_netasq.commonlib.asqPath             as asqPath

from    py_netasq.commonlib.asqIniParser        import asqIniParser
from    py_netasq.delphi.building.buildFileParser               \
        import Error, sectionError, taskItemParseError, buildFileParser

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if __name__ == '__main__' :
  print __file__
  print __doc__

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


class buildFileValidator(buildFileParser) :
    """\
      A build file validator, derivated from buildFileParser. This class
      should not be instanciated, a class method "checkFile" is provided to
      perform file validation.
      
      
      constructor buildFileValidator(void)
      
      void      checkFile(string filename, object out)
    """
    
    
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Class methods - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
   
    def checkFile(cls, filename, out) :
        """\
          void checkFile(string filename, object out)
          
          Open and parse given file, but do not load dataset; This function
          is intended to ease the location of build file (*.ini) errors.
          'out' must behave like a file/stream object. If file does not
          exist then an exception will be raised.
          
          param filename              config file path
          param out                   output stream
        """
        filename = asqPath.normalize(os.path.abspath(filename))
        aParser  = buildFileValidator()
        aParser.readFile(filename)
        
        aParser._checkFileVersion(out)
        aParser._checkMailConfig(out)
        for aType in buildFileParser.TASKS_ORDER :
          aParser._checkTaskList(aType, out)
   
   
    # define class methods
    checkFile = classmethod(checkFile)
    
    
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Constructor - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
   
    def __init__(self) :
        """\
          constructor buildFileValidator(void)
          
          Create a new build file validator.
        """
        # file version MUST not be set
        asqIniParser.__init__(self, None)
   
   
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
   
    def _checkFileVersion(self, out) :
        """\
          void _checkFileVersion(object out)
          
          Parses file's "FileVersion" section, validation results are written
          to 'out' object. If section or option is not found, then current
          FILEVERSION is assumed.
          The 'out' object must behave like a file descriptor: at least the
          "write" method must be present (sys.stdout can do the trick too).
          
          param out                   output stream/file descriptor
        """
        header = buildFileParser.SECT_FILEVERSION
        option = buildFileParser.TOKN_VERSION
        defval = '.'.join(buildFileParser.FILEVERSION)
        
        out.write(self._CheckingSectionString(header))
        try :
          result = buildFileParser.get(self, header, option, False)
        except ConfigParser.NoSectionError :
          out.write('-Missing\n')
          out.write('>> Missing value %s - assuming %s\n' % (option, defval))
        except ConfigParser.NoOptionError :
          out.write('-Warning\n')
          out.write('>> Missing value %s - assuming %s\n' % (option, defval))
        except Exception, e :
          out.write('-Failed\n')
          out.write('%s\n' % (str(e).rstrip(),))
        else :
          out.write('-Ok\n')
   
   
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
   
    def _checkMailConfig(self, out) :
        """\
          void _checkMailConfig(object out)
          
          Parses file's "Mailing" section, validation results are written
          to 'out' object.
          The 'out' object must behave like a file descriptor: at least the
          "write" method must be present (sys.stdout can do the trick too).
          
          param out                   output stream/file descriptor
        """
        header = buildFileParser.SECT_MAILING
        
        out.write(self._CheckingSectionString(header))
        try :
          values = self.items(header, False)
        except ConfigParser.NoSectionError :
          out.write('-Missing\n')
        except Exception, e :
          out.write('-Failed\n')
          out.write('%s\n' % (str(sectionError(e, header)),))
        else :
          out.write('-Ok\n')
   
   
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
   
    def _checkTaskList(self, listType, out) :
        """\
          void _checkTaskList(string listType, object out)
          
          Parses file's task list, validation results are written to 'out'
          object. The checking process tries to gather as many errors as
          possible; it does not stop on first error.
          The 'out' object must behave like a file descriptor: at least the
          "write" method must be present (sys.stdout can do the trick too).
          
          param listType              tasklist type
          param out                   output stream/file descriptor
        """
        header = buildFileParser.TASKS_SECTIONS[listType]
        option = buildFileParser.TOKN_COUNT
        errors = list()
        length = 0
        
        out.write(self._CheckingSectionString(header))
        try :
          length = self.getint(header, option)
          self._getTaskListHead(listType, False)
        except ConfigParser.NoSectionError :
          out.write('-Missing\n')
          return
        except ConfigParser.NoOptionError :
          # _getTaskListHead can not throw that exception
          out.write('-Warning\n')
          out.write('>> Missing value %s - assuming %d\n' % (option, length))
          return
        except Exception, e :
          errors.append(taskListParseError(e, header))
          
        errors.extend(self._checkTaskListItems(listType, length))
        if not errors :
          out.write('-Ok\n')
        else :
          out.write('-Failed\n')
          for item in errors :
            out.write('%s\n' % (str(item).rstrip(),))
   
   
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
   
    def _checkTaskListItems(self, listType, length) :
        """\
          list _checkTaskListItems(string listType)
          
          Check task list's items syntax, each task is simply read with string
          interpolation from dataset. Each time an exception is encountered
          it is stored in resulting list.
          
          param listType              tasklist type
          param length                tasklist length
          return                      encountered parser exceptions
        """
        result = list()
        header = buildFileParser.TASKS_SECTIONS[listType]
        
        for indx in xrange(length) :
          try :
            self._getTaskItem(listType, indx, False)
          except Exception, e :
            result.append(taskItemParseError(e, header, indx))
        
        return result
   
   
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
   
    def _CheckingSectionString(self, header) :
        """\
        """
        result = "Section %s" % (header,)
        return result.ljust(22)
   
   
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # class buildFileValidator  - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -