#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
A build file parser.

Since
  Python 2.3

Classes
  class         Error(void)
  class         sectionError(object exception, string section)
  class         taskItemParseError(object exception, string section, int index)

  class         buildFileParser(dict defaults) extends asqIniParser
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.4 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  ConfigParser

import  py_netasq.commonlib.asqSequence         as asqSequence

from    py_netasq.commonlib.asqIniParser        import asqIniParser
from    py_netasq.delphi.building.mailTasks     import mailTaskItem
from    py_netasq.delphi.building.taskFactory   import taskFactory

from    py_netasq.delphi.building.cvsTasks      import cvsTaskItem, cvsTaskList
from    py_netasq.delphi.building.dccTasks      import dccTaskItem, dccTaskList
from    py_netasq.delphi.building.rawTasks      import rawTaskItem, rawTaskList
from    py_netasq.delphi.building.resTasks      import resTaskItem, resTaskList
from    py_netasq.delphi.building.sf6Tasks      import sf6TaskItem, sf6TaskList
from    py_netasq.delphi.building.vrsTasks      import vrsTaskItem, vrsTaskList

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if __name__ == '__main__' :
  print __file__
  print __doc__

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

# SF Bug #997050 : ConfigParser behavior change
#
# https://sourceforge.net/tracker/?func=detail&atid=105470&aid=997050&group_id=5470
#
# ConfigParser.set() doesn't allow non-string arguments for 'value'
# any more. Actually, this restriction should only be encountered with
# SafeConfigParser, check the cvs revision :
#
# cvs: python/python/dist/src/Lib/ConfigParser.py
# Revision: 1.68, Sun Oct 3 15:55:09 2004 UTC by goodger
#  Document, test, & check for non-string values in ConfigParser.
#  Moved the new string-only restriction added in rev. 1.65 to the
#  SafeConfigParser class, leaving existing ConfigParser & RawConfigParser
#  behavior alone, and documented the conditions under which non-string
#  values work.
#

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class Error(Exception) :
    """\
      Base class for buildFileParser exceptions.
    """
   
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
   
    def __repr__(self) :
        result = filter(None, self._buildRepr())
        return ' '.join(result).rstrip()
      
    __str__ = __repr__
   
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
   
    def _buildRepr(self) :
        return list()
   
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # class Error - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class sectionError(Error) :
    """\
      Happens when an exception is raised concerning a section.
    """
   
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    
    def __init__(self, exception, section) :
        self._error   = exception
        self._section = section
   
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
   
    def _buildRepr(self) :
        result = Error._buildRepr(self)
        buffer = '%s section' % (self._section,)
        
        result.append('%s:' % (buffer.ljust(20),))
        result.append(str(self._error))
        return result
   
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # class sectionError  - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class taskItemParseError(sectionError) :
    """\
      Happens when an exception is raised as a task list item is parsed.
    """
    
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    
    def __init__(self, exception, section, index) :
        sectionError.__init__(self, exception, section)
        self._index = index
   
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
   
    def _buildRepr(self) :
        result = list()
        buffer = '%s.items[%02d]' % (self._section, self._index)
        
        result.append('%s:' % (buffer.ljust(20),))
        result.append(str(self._error))
        return result
   
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # class taskItemParseError  - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


class buildFileParser(asqIniParser) :
    """\
      A build file parser, derivated from asqIniParser.
      
      
      constructor buildFileParser(void)
      
      var       getTypeFromSection(string aSection)
      var       getSectionFromType(string aType)
      
      void      write(object file)
      
      void      extractTaskList(object aList, bool raw, string listType)
      void      extractMailConfig(object aTask, bool raw)
      
      tuple     getFileVersion(void)
      object    getMailConfig(bool raw)
      
      object    getTaskItem(string listType, int index, bool raw)
      object    getTaskList(string listType, bool raw)
      
      void      setFileVersion(tuple version)
      void      setMailConfig(object aTask)
      
      void      setTaskItem(object aItem, string listType, int index)
      void      setTaskList(object aList, string aType)
    """
    
    # default file version
    FILEVERSION       = ('1', '0')
    
    SECT_DEFAULT      = "DEFAULT"
    SECT_FILEVERSION  = "FileVersion"
    SECT_MAILING      = "Mailing"
    
    TOKN_COUNT        = "count"
    TOKN_VERSION      = "version"
    TOKN_TASKTYPE     = "tasktype"
   
    TASKS_SECTIONS    = {
        cvsTaskItem.TASKTYPE : "CvsTasks",
        dccTaskItem.TASKTYPE : "DccTasks",
        resTaskItem.TASKTYPE : "ResTasks",
        sf6TaskItem.TASKTYPE : "Sf6Tasks",
        vrsTaskItem.TASKTYPE : "VrsTasks"
      }
    
    TASKS_SUBSECTIONS = {
        cvsTaskItem.TASKTYPE : "CvsTask_%u",
        dccTaskItem.TASKTYPE : "DccTask_%u",
        resTaskItem.TASKTYPE : "ResTask_%u",
        sf6TaskItem.TASKTYPE : "Sf6Task_%u",
        vrsTaskItem.TASKTYPE : "VrsTask_%u"
      }
    
    TASKS_ORDER       = (
        cvsTaskItem.TASKTYPE,
        vrsTaskItem.TASKTYPE,
        resTaskItem.TASKTYPE,
        dccTaskItem.TASKTYPE,
        sf6TaskItem.TASKTYPE
      )
   
    # invert TASKS_SECTIONS dictionary
    SECTIONS_TYPES    =                                                 \
      dict([ (val.lower(), key) for key, val in TASKS_SECTIONS.iteritems() ])
    SECTIONS_ORDER    = [ TASKS_SECTIONS[i] for i in TASKS_ORDER ]
    
    
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Class methods - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def getTypeFromSection(cls, *args) :
        """\
          var getTypeFromSection(...)
          
          Returns the task type for each given section name; name matching
          is case insensitive. If there is no matching tasktype for a section
          then None is yield.
          
          param ...                   section name or list of names
          return                      task type or list of types
        """
        if not args :
          return None
        elif 1 < len(args) :
          return [ cls.getTypeFromSection(s) for s in args ]
        else :
          try :
            return cls.SECTIONS_TYPES[args[0].lower()]
          except :
            return None
   
   
    def getSectionFromType(cls, *args) :
        """\
          var getSectionFromType(...)
          
          Returns the section name for each given task type. If there is no
          matching section name for a tasktype then None is yield.
          
          param ...                   task type  or list of types
          return                      section name or list of names
        """
        if not args :
          return None
        elif 1 < len(args) :
          return [ cls.getSectionFromType(s) for s in args ]
        else :
          try :
            return cls.TASKS_SECTIONS[aType]
          except :
            return None
   
   
    # define class methods
    getTypeFromSection = classmethod(getTypeFromSection)
    getSectionFromType = classmethod(getSectionFromType)

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Constructor - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __init__(self, defaults=None) :
        """\
          constructor buildFileParser(dict defaults)
          
          Create a new build file parser.
          
          param defaults              parser DEFAULT options,
                                        defaults to None
        """
        asqIniParser.__init__(self, defaults)
        self.setFileVersion(buildFileParser.FILEVERSION)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def write(self, file) :
        """\
          void write(object file)
          
          Write a representation of the configuration to the specified file
          object. This representation can be parsed by a future read() call.
          
          Because ConfigParser does not give a friggin damn about sections
          order, we have to do the dirty work by ourselves...
          WARNING : Any change made to ConfigParser may break this function !
          
          param file                  file descriptor
        """
        # here we rely on ConfigParser._sections dictionary :(
        keylst = self._sections.keys()
       
        # Start with file version info ...
        self._writeFileVersion(file)
        if buildFileParser.SECT_FILEVERSION in keylst :
          keylst.remove(buildFileParser.SECT_FILEVERSION)
        # ... then DEFAULT values, ...
        # no need to remove DEFAULT header
        self._writeDefaults(file, True)
        # ... Mail config, ...
        self._writeMailConfig(file)
        if buildFileParser.SECT_MAILING in keylst :
          keylst.remove(buildFileParser.SECT_MAILING)
        # ... task lists ...
        for aType in buildFileParser.TASKS_ORDER :
          if buildFileParser.TASKS_SECTIONS[aType] in keylst :
            writtn = self._flushTaskList(aType, file)
            keylst = asqSequence.difference(keylst, writtn)
        # ... and remaining unrecognised sections
        for name in keylst :
          self._writeSection(name, self._sections[name], file)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def extractTaskList(self, aList, raw=False, listType=None) :
        """\
          void extractTaskList(object aList, bool raw, string listType)
          
          Updates given task list 'aList' to reflect current dataset; if
          'listType' is given, then it overrides aList.tasktype. If raw is
          set to True, then no string interpolation is done for section's
          items. If no matching section is found, then given task list will
          only be reset.
          
          param aList                 task list to update
          param raw                   extract with no interpolation,
                                        defaults to False
          param listType              tasklist type, defaults to None
        """
        assert aList is not None, "no valid task list supplied"
        
        if listType is None :
          listType = aList.tasktype
        aList.reset()
        buffer = self.getTaskList(listType, raw)
        if buffer is not None :
          aList.assign(buffer)
          aList[:] = buffer[:]


    def extractMailConfig(self, aTask, raw=False) :
        """\
          void extractMailConfig(object aTask, bool raw)
          
          Updates given mail task 'aTask' to reflect current dataset. If
          'raw' is set to True, no string interpolation is done upon
          section's items. If no matching section is found, then given mail
          task will only be reset.
          
          param aTask                 mail task to update
          param raw                   extract with no interpolation,
                                        defaults to False
        """
        assert aTask is not None, "no valid mail task given"
        
        aTask.reset()
        buffer = self.getMailConfig(raw)
        if buffer is not None :
          aTask.assign(buffer)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _flushTaskList(self, aType, out) :
        """\
          list _flushTaskList(string aType, object out)
          
          Write a taskList, whose tasktype attribute matches 'aType', to
          'out' object. The 'out' object must behave like a file descriptor :
          at least the "write" method must be present (sys.stdout can do the
          trick too). The list of written sections is returned.
          
          param out                   output stream/file descriptor
          return                      list of written sections
        """
        try :
          aList  = self.getTaskList(aType, True)
          result = self._writeTaskList(aList, aType, out)
        except AssertionError :
          return list()
        else :
          return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _writeFileVersion(self, out) :
        """\
          void _writeFileVersion(object out)
          
          Write "FileVersion" section; The 'out' object must behave like a
          file descriptor: at least the "write" method must be present
          (sys.stdout can do the trick too).
          
          param out                   output stream/file descriptor
        """
        version = '.'.join(self.getFileVersion())
        
        out.write('[%s]\n'      % (buildFileParser.SECT_FILEVERSION,))
        self._writeOption(buildFileParser.TOKN_VERSION, version, out)
        out.write('\n\n')


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _writeMailConfig(self, out) :
        """\
          var _writeMailConfig(object out)
          
          Write mailing config set in current parser dataset to 'out' object.
          The 'out' object must behave like a file descriptor: at least the
          "write" method must be present (sys.stdout can do the trick too).
          
          param out                   output stream/file descriptor
          return                      section header, or None
        """
        aTask = self.getMailConfig(True)
        
        if aTask is None :
          return None
        elif aTask.isEmpty() :
          return None
        else :
          result = buildFileParser.SECT_MAILING
          buffer = aTask.toList(True)
          if mailTaskItem.TASKTYPE == aTask.tasktype :
            self._extractOption(buildFileParser.TOKN_TASKTYPE, buffer)
          self._writeSection(result, buffer, out, '')
          
          return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _writeTaskItem(self, aItem, listType, index, out) :
        """\
          var _writeTaskItem(
                  object aItem, string listType, int index, object out)
          
          Write given task item 'aItem' to 'out' object. The 'out' object
          must behave like a file descriptor: at least the "write" method
          must be present (sys.stdout can do the trick too).
         
          Return section header string, or None if nothing was actually
          written.
          
          param aItem                 item to write
          param listType              owner type
          param index                 item index
          param out                   output stream/file descriptor
          return                      section header, or None
        """
        assert aItem is not None, "no valid task item supplied"
        
        try :
          result = buildFileParser.TASKS_SUBSECTIONS[listType] % (index,)
          buffer = aItem.toList(True)
          if listType == aItem.tasktype :
            self._extractOption(buildFileParser.TOKN_TASKTYPE, buffer)
          self._writeSection(result, buffer, out, '')
        except :
          return None
        else :
          return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _writeTaskList(self, aList, listType, out) :
        """\
          list _writeTaskList(object aList, string listType, object out)
          
          Write given 'aList' to 'out' object. The 'out' object must behave
          like a file descriptor: at least the "write" method must be present
          (sys.stdout can do the trick too). The list of written sections is
          returned (as list of section headers).
          
          param aList                 list to write
          param listType              list type
          param out                   output stream/file descriptor
          return                      list of written sections
        """
        assert aList is not None, "no valid task list supplied"
        
        try :
          result = list()
          header = buildFileParser.TASKS_SECTIONS[listType]
          length = len(aList)
          buffer = aList.toList(False)
        
          buffer.insert(0, (buildFileParser.TOKN_COUNT, length))
          self._writeSection(header, buffer, out, '')
          result.append(header)
          
          for indx in xrange(length) :
            header = self._writeTaskItem(aList[indx], listType, indx, out)
            result.append(header)
          out.write('\n')
        except :
          return list()
        else :
          # strip out any None values
          return filter(None, result)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  
    def _getTaskItem(self, listType, index, raw) :
        """\
          object _getTaskItem(string listType, int index, bool raw)
          
          Return a task item build from current parser dataset, 'listType'
          must be the owner tasktype. If raw is set to True, then no string
          interpolation is done for section's items.
          Exception may be raised if tasktype is not recognized, or if there
          is no matching section in dataset.
          None can be returned if taskFactory does not know given 'listType'.
          
          param listType              owner type
          param index                 task item index
          param raw                   extract with no interpolation
          return                      a task item, or None
        """
        header = buildFileParser.TASKS_SUBSECTIONS[listType] % (index,)
        values = self.items(header, raw)
        # remove "DEFAULT" values from list
        values = asqSequence.difference(values, self.defaults().items())
        
        # listType is assumed if tasktype is not found
        tkType = self.getOption(header, buildFileParser.TOKN_TASKTYPE, listType)
        result = taskFactory.buildTaskItem(tkType)
        
        if result is not None :
          result.fromList(values, listType)
        return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _getTaskListHead(self, listType, raw) :
        """\
          object _getTaskListHead(string listType, bool raw)
          
          Return a task list build from current parser dataset; the task
          list is filled with None values as list items to match the length
          found in dataset. If raw is set to True, then no string
          interpolation is done for section's items.
          This function does NOT extract list items from current dataset.
          
          Exception may be raised if tasktype is not recognized, or if
          there is no matching section in dataset.
          
          param listType              tasklist type
          param raw                   get task list with no interpolation
          return                      a task list, or None
        """
        header = buildFileParser.TASKS_SECTIONS[listType]
        values = self.items(header, raw)
        # remove "DEFAULT" values from list
        values = asqSequence.difference(values, self.defaults().items())
        # listType is assumed if tasktype is not found
        tkType = self.getOption(header, buildFileParser.TOKN_TASKTYPE, listType)
        result = taskFactory.buildTaskList(tkType)
        length = self._extractIntOption(buildFileParser.TOKN_COUNT, values, 0)
        
        if result is not None :
          result.fromList(values, listType)
          result[:] = [None] * length
        return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _getTaskListItems(self, listType, length, raw) :
        """\
          list _getTaskListItems(string listType, int length, bool raw)
          
          Return a list of items for the given task type, 'length' is the
          expected list length; if 'raw' is set to True, then no string
          interpolation will be done upon item values.
          
          param listType              tasklist type
          param length                expectd tasklist length
          param raw                   get task items with no interpolation
          return                      a task list, or None
        """
        result = list()
        for indx in xrange(length) :
          result.append(self.getTaskItem(listType, indx, raw))
        return filter(None, result)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _setTaskItem(self, aItem, section, filtered) :
        """\
          void _setTaskItem(object aItem, string section, bool filtered)
          
          Add/Update task item in current dataset. If 'filtered' is set,
          then item attributes that equals None are skipped, they are not
          set in current dataset.
          
          param aItem                 a TaskItem
          param section               task section
          param filtered              skip None values
        """
        self.addSection(section)
        self.setItems(section, aItem.toList(filtered))


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _setTaskListHead(self, aList, section, filtered) :
        """\
          void _setTaskListHead(object aList, string section, bool filtered)
          
          Add/Update task list in current dataset. If 'filtered' is set,
          then list attributes that equals None are skipped, they are not
          set in current dataset.
          
          param aList                 a TaskList
          param section               task section
          param filtered              skip None values
        """
        length = len(aList)
        self._setTaskItem(aList, section, filtered)
        self.setOption(section, buildFileParser.TOKN_COUNT, length, 0)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def getFileVersion(self) :
        """\
          tuple getFileVersion(void)
          
          Extract FileVersion from current file. The version is returned as
          a tuple; An assertion except will be thrown if there is no
          accurate section found in parser dataset.
          
          e.g. "version = 0.0.5" => ('0', '0', '5')
          
          return                      FileVersion
        """
        self.assertSection(buildFileParser.SECT_FILEVERSION)
        result = self.getOption(
                  buildFileParser.SECT_FILEVERSION,
                  buildFileParser.TOKN_VERSION,
                  buildFileParser.FILEVERSION )
        return tuple(result.split('.'))


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def getMailConfig(self, raw=False) :
        """\
          object getMailConfig(bool raw)
          
          Return a mail task build from current parser dataset. If raw is
          set to True, then no string interpolation is done for section's
          items.
          
          param raw                   extract with no interpolation,
                                        defaults to False
          return                      mail task, or None
        """
        try :
          header = buildFileParser.SECT_MAILING
          values = self.items(header, raw)
          # mailTaskItem.TASKTYPE is assumed if tasktype is not found
          tkType = mailTaskItem.TASKTYPE
          tkType = self.getOption(header, buildFileParser.TOKN_TASKTYPE, tkType)
        except ConfigParser.NoSectionError :
          return None
        else :
          # build right task according to tkType
          result = taskFactory.buildTaskItem(tkType)
          if result is not None :
            result.fromList(values, mailTaskItem.TASKTYPE)
          return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def getTaskItem(self, listType, index, raw=False) :
        """\
          object getTaskItem(string listType, int index, bool raw)
          
          Return a task item build from current parser dataset, 'listType'
          must be the owner tasktype. If raw is set to True, then no string
          interpolation is done for section's items.
          None can be returned if taskFactory does not know given 'listType'.
          
          param listType              taskitem type
          param index                 task item index
          param raw                   extract with no interpolation,
                                        defaults to False
          return                      a task item, or None
        """
        try :
          header = buildFileParser.TASKS_SECTIONS[listType]
          result = self._getTaskItem(listType, index, raw)
        except (KeyError, ConfigParser.NoSectionError) :
          return None
        except Exception, e :
          raise taskItemParseError(e, header, index)
        else :
          return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def getTaskList(self, listType, raw=False) :
        """\
          object getTaskList(string listType, bool raw)
          
          Return a task list build from current parser dataset; If raw is
          set to True, then no string interpolation is done for section's
          items. 
          
          param listType              tasklist type
          param raw                   extract with no interpolation,
                                        defaults to False
          return                      a task list, or None
        """
        try :
          header = buildFileParser.TASKS_SECTIONS[listType]
          result = self._getTaskListHead(listType, raw)
        except (KeyError, ConfigParser.NoSectionError) :
          return None
        except Exception, e :
          raise sectionError(e, header)
        else :
          result[:] = self._getTaskListItems(listType, len(result), raw)
          return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def setFileVersion(self, version) :
        """\
          void setFileVersion(tuple version)
          
          Set current file FileVersion. The version should be a tuple;
          e.g. setFileVersion(('0', '0', '5')) => "version = 0.0.5"
          
          param version               FileVersion
        """
        header = buildFileParser.SECT_FILEVERSION
        option = buildFileParser.TOKN_VERSION
        value  = '.'.join(version)
        
        self.addSection(header)
        self.set(header, option, value)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def setMailConfig(self, aTask) :
        """\
          void setMailConfig(object aTask)
          
          Update mail config using given 'aTask' mailing task.
          
          param aTask                 mail task item
        """
        if not aTask.isEmpty() :
          self._setTaskItem(aTask, buildFileParser.SECT_MAILING, True)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def setTaskItem(self, aItem, listType, index) :
        """\
          void setTaskItem(object aItem, string listType, int index)
          
          Update taskitem at given 'index'. If a task's attribute equals
          None, then it is skipped and the previous option value remains;
          other attribute values will overwrite section's option.
          e.g.
            in parser dataset : "logfile = somefile.log"
           
            task.logFile = None
            setTaskItem(task, index)
            in parser dataset : "logfile = somefile.log"
            
            task.logFile = ''
            setTaskItem(task, index)
            in parser dataset : "logfile = "
          
          param aItem                 a task item
          param listType              task owner type
          param index                 task index
        """
        if aItem.isEmpty() :
          return
        try :
          header = buildFileParser.TASKS_SUBSECTIONS[listType] % (index,)
        except KeyError :
          pass
        else :
          self._setTaskItem(aItem, header, True)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def setTaskList(self, aList, listType=None) :
        """\
          void setTaskList(object aList, string aType)
          
          Update tasklist in dataset. If listType is not set, then
          'aList.tasktype' will be used instead. If a task's attribute
          equals None, then it is skipped and the previous option value
          remains; other attribute values will overwrite section's option.
          
          param aList                 a task list object
          param listType              task list type, defaults to None
        """
        if aList.isEmpty() :
          return
        try :
          if listType is None :
            listType = aList.tasktype
          header = buildFileParser.TASKS_SECTIONS[listType]
        except KeyError :
          pass
        else :
          self._setTaskListHead(aList, header, True)
          for indx in xrange(len(aList)) :
            self.setTaskItem(aList[indx], listType, indx)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # class buildFileParser - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -