#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
A build.ini file wrapper. Provide an easy way to handle building task.

Since
  Python 2.2

Classes
  class         taskItemRunError(string exception, string section, int index)

  class         buildFileExec(void) extends buildFileParser
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.12 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  os

import  py_netasq.commonlib.asqPath             as asqPath

from    py_netasq.delphi.building.buildFileParser               \
        import Error, sectionError, buildFileParser
from    py_netasq.delphi.building.buildFileEditor               \
        import buildFileEditor

# taskFactory came to the rescue :)
#~ from    py_netasq.delphi.building.cvsTasks      import cvsTaskItem
#~ from    py_netasq.delphi.building.dccTasks      import dccTaskItem
#~ from    py_netasq.delphi.building.rawTasks      import rawTaskItem
#~ from    py_netasq.delphi.building.resTasks      import resTaskItem
#~ from    py_netasq.delphi.building.sf6Tasks      import sf6TaskItem
#~ from    py_netasq.delphi.building.vrsTasks      import vrsTaskItem

#~ from    py_netasq.delphi.building.mailTasks     import mailTaskItem

from    py_netasq.delphi.building.taskFactory   import taskFactory

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if __name__ == '__main__' :
  print __file__
  print __doc__

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class taskItemRunError(sectionError) :
    """\
      Happens when an exception is raised as a task list item is parsed.
    """
    
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    
    def __init__(self, exception, section, index) :
        sectionError.__init__(self, exception, section)
        self._index  = index
   
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
   
    def _buildRepr(self) :
        result = list()
        buffer = '%s.items[%02d]' % (self._section, self._index)
        
        result.append('%s:' % (buffer.ljust(20),))
        result.append(str(self._error))
        return ['\n'.join(result)]
   
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # class taskItemParseError  - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


class buildFileExec(buildFileEditor) :
    """\
      A "build file" executor: execute tasks sections; process is stopped
      each time an error occur.
      
      
      static void       runFile(
                          string filename, object out,
                          list sections, dict defaults, int verbose)
      
      constructor buildFileExec(void)
      
      void      loadFile(string filename, bool raw)
      void      saveFile(string filename)
      
      void      reset(void)
      
      void      runSections(list sections)
      
      property object cvsTasks   : cvs
      property object dccTasks   : compiler
      property object resTasks   : resource building
      property object sf6Tasks   : setup factory
      property object vrsTasks   : versioning task list
    """
    
    # from 0 (silent) to 2 (lot of noise)
    VERBOSITY_RANGE = range(3)
    # defaults to "lot of noise"
    DEFAULT_VERBOSITY = 2
    
    
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Class methods - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
   
    def runFile(
          cls, filename, out,
          sections=[], defaults={}, verbose=DEFAULT_VERBOSITY) :
        """\
          void runFile(
                string filename, object out,
                list sections, dict defaults, int verbose)
          
          Run a build file with given settings; if 'verbose' level is bigger
          than 0, then 'out' CANNOT be None or else exceptions may be raised.
          'sections' list defines a selective list of sections to execute,
          if 'sections' is empty, then all defined sections are executed.
          'defaults' can be used to define or override values found in
          "DEFAULT" section.
          
          param filename              build file to execute
          param out                   process output stream
          param sections              selective sections execution,
                                        defaults to []
          param defaults              instance wide default values,
                                        defaults to {}
          param verbose               verbosity level, defaults to
                                        buildFileExec.DEFAULT_VERBOSITY
        """
        assert out is not None or 0 == verbose,                         \
               "An output stream must be supplied."
        
        filename = asqPath.normalize(os.path.abspath(filename))
        aBuilder = buildFileExec()
        
        aBuilder.loadFile(filename, False)
        aBuilder.verbosity = verbose
        aBuilder.output    = out
        
        aBuilder.updateDefaults(defaults)
        
        if 0 < verbose :
          out.write("Processing %s\n" % (filename,))
          out.write('\n')
          aBuilder._writeDefaults(out, False)
          out.write('\n')
        
        try :
          aBuilder.runSections(sections)
        except Exception, e :
          if not sections :
            aBuilder._sendMail(False, str(e))
          raise
        else :
          if not sections :
            aBuilder._sendMail(True, "all right !")
        
        if 0 < verbose :
          out.write('+-- Done\n')
   
   
    # define class methods
    runFile = classmethod(runFile)
   
   
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Constructor - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __init__(self) :
        """\
          constructor buildFileExec(void)
          
          Create a "build file" wrapper: provides easy acces to settings
          options.
        """
        self._commands  = dict()
        # whole process output
        self._output    = None
        # verbosity level
        self._verbosity = 0
        
        # reset will be called
        buildFileEditor.__init__(self)
        
        # mailTaskItem.TASKTYPE does belong to taskFactory.TASKITEM_TYPES
        for aType in taskFactory.TASKITEM_TYPES :
          self._commands[aType] = taskFactory.buildCommand(aType)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Special methods - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def reset(self) :
        """\
          void reset(void)
          
          Empty all tasks lists, reset parser, set basedir to current working
          directory; disable output and set default verbosity level.
        """
        buildFileEditor.reset(self)
        self._setOutput(None)
        self._setVerbosity(buildFileExec.DEFAULT_VERBOSITY)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def runSections(self, sections=[]) :
        """\
          void runSections(list sections)
          
          Process specified tasks sections, if given list is empty, then all
          tasks sections are processed.
        """
        if not sections :
          tskTypes = buildFileParser.TASKS_ORDER
        else :
          # quick and dirty hack :
          # if only one element is found in'sections', then
          # getTypeFromSection will yield a string :/
          sections.append(None)
          tskTypes = buildFileParser.getTypeFromSection(*sections)
        tskTypes = filter(None, tskTypes)
        
        if 0 == self._verbosity :
          for aType in tskTypes :
            self._runTaskListSilent(aType)
        elif 1 == self._verbosity :
          for aType in tskTypes :
            self._runTaskListQuiet(aType)
        else :
          for aType in tskTypes :
            self._runTaskListVerbose(aType)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
   
    def _sendMail(self, success, text) :
        """\
          void _sendMail(bool success, string text)
          
          Use mailing configuration to send a mail; a mail will be sent
          only if the mailing settings is found valid.
          
          param sucess                process status (success/failure)
          param text                  mail body
        """
        if not self._mailing.isValid() :
          return
        elif not self._mailing.enabled :
          return
        else :
          mailCmd = self._commands[self._mailing.tasktype]
          mailCmd.run(success, text, self._mailing)
   
   
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
   
    def _runTaskItem(self, aItem) :
        """\
          void _runTaskItem(object aItem)
          
          Excecute task defined by 'aItem'. If task is disabled or if there
          is no command to actually process that task, then nothing happens.
          If task command does fail, then an exception is raised, with the
          command output as argument.
          
          param aItem                 task item
        """
        if not aItem.enabled :
          return
        try :
          tskCmd = self._commands[aItem.tasktype]
        except KeyError :
          # no command for this task type
          return
        else :
          #~ aItem.debug()
          if tskCmd.run(aItem) is not None :
            raise Exception, tskCmd.cmdOutput
   
   
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
   
    def _runTaskListSilent(self, listType) :
        """\
          void _runTaskListSilent(string listType)
          
          Run task list of given 'listType' type, in silent mode: nothing
          is sent to output stream.
          
          param listType              task list type
        """
        header = buildFileParser.TASKS_SECTIONS[listType]
        tkList = self._tasklist[listType]
        
        if not tkList.enabled :
          return
        for indx in xrange(len(tkList)) :
          try :
            if tkList[indx].enabled :
              tkItem = tkList[indx].mergeWithTask(tkList)
              tkItem.completePaths(self._basedir)
              
              self._runTaskItem(tkItem)
          except Exception, e :
            raise taskItemRunError(e, header, indx)
    
    
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    
    def _runTaskListQuiet(self, listType) :
        """\
          void _runTaskListQuiet(string listType)
          
          Run task list of given 'listType' type, in quiet mode: a few
          informations are sent to output stream.
          WARNING : object's "output" attribute must have been set to a
          VALID file/stream object, or else an exception will be raised.
          
          param listType              task list type
        """
        header = buildFileParser.TASKS_SECTIONS[listType]
        tkList = self._tasklist[listType]
        
        if not tkList.enabled :
          self._output.write('+-- %s -DISABLED-\n' % (header,))
          self._output.write('|\n')
          return
        else :
          self._output.write('+-- Processing %s\n' % (header,))
          self._output.write('|\n')
          self._output.write('| task count %2d\n' % (len(tkList),))
        for indx in xrange(len(tkList)) :
          try :
            if not tkList[indx].enabled :
              self._output.write('| %2d -DISABLED-\n' % (indx,))
            else :
              tkItem = tkList[indx].mergeWithTask(tkList)
              tkItem.completePaths(self._basedir)
              
              self._output.write('| %2d %s\n' % (indx, str(tkItem)))
              self._runTaskItem(tkItem)
          except Exception, e :
            raise taskItemRunError(e, header, indx)
        self._output.write('|\n')
   
    
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    
    def _runTaskListVerbose(self, listType) :
        """\
          void _runTaskListVerbose(string listType)
          
          Run task list of given 'listType' type, in verbose mode: a lot of
          informations are sent to output stream.
          WARNING : object's "output" attribute must have been set to a
          VALID file/stream object, or else an exception will be raised.
          
          param listType              task list type
        """
        header = buildFileParser.TASKS_SECTIONS[listType]
        tkList = self._tasklist[listType]
        
        if not tkList.enabled :
          self._output.write('+-- %s -DISABLED-\n' % (header,))
          self._output.write('|\n')
          return
        else :
          self._output.write('+-- Processing %s\n' % (header,))
          self._output.write('|\n')
          self._output.write('| task count %2d\n' % (len(tkList),))
        for indx in xrange(len(tkList)) :
          try :
            if not tkList[indx].enabled :
              self._output.write(' task %2d -DISABLED-\n\n' % (indx,))
            else :
              tkItem = tkList[indx].mergeWithTask(tkList)
              tkItem.completePaths(self._basedir)
              
              self._runTaskItem(tkItem)
          except Exception, e :
            raise taskItemRunError(e, header, indx)
        self._output.write('|\n')
   
   
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _getOutput(self) :
        """\
          object _getOutput(void)
          
          Getter - get building process output.
          
          return                      output stream
        """
        return self._output


    def _getVerbosity(self) :
        """\
          int _getVerbosity(void)
          
          Getter - get building process verbosity level
          
          return                      verbosity level [0..2]
        """
        return self._verbosity


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _setOutput(self, value) :
        """\
          void _setOutput(object value)
          
          Set building process output, given 'value' must behave like a file
          object (as sys.stdout does). If value is None, then process will be
          silent. Process will also remain silent if verbosity level is set
          too low.
          
          param value                 output stream to use
        """
        if self._output is not None :
          try :
            for key in self._commands.iterkeys() :
              self._commands[key].listeners.remove(self._output)
          except ValueError :
            # if verbosity is too low, then output may not be found in
            # commands listeners list
            pass
        self._output = value
        # if verbosity value is lesser than 2, do not add output to listeners
        if self._output is not None and 1 < self._verbosity :
          for key in self._commands.iterkeys() :
            self._commands[key].listeners.append(self._output)


    def _setVerbosity(self, value) :
        """\
          void _setVerbosity(int value)
          
          Set process verbosity level; if 'given' value is not valid, does
          not belog to buildFileExec.VERBOSITY_RANGE, then verbosity level
          is set to buildFileExec.DEFAULT_VERBOSITY. Commands listeners
          will also be updated accordingly.
          
          param value                 verbosity level
        """
        if value not in buildFileExec.VERBOSITY_RANGE :
          self._verbosity = buildFileExec.DEFAULT_VERBOSITY
        else :
          self._verbosity = value
          
        # refresh commands listeners
        self._setOutput(self._output)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    output    = property(
                  doc  = "process output",
                  fget = _getOutput,
                  fset = _setOutput,
                  fdel = None )

    verbosity = property(
                  doc  = "",
                  fget = _getVerbosity,
                  fset = _setVerbosity,
                  fdel = None )

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # class buildFileExec - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -