
#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
A build.ini file wrapper. Provide an easy way to handle building task.

Since
  Python 2.2

Classes
  class         buildFileEditor(void) extends buildFileParser
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.3 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  os

import  py_netasq.commonlib.asqPath             as asqPath

from    py_netasq.delphi.building.buildFileParser               \
        import buildFileParser

from    py_netasq.delphi.building.cvsTasks      import cvsTaskItem
from    py_netasq.delphi.building.dccTasks      import dccTaskItem
from    py_netasq.delphi.building.rawTasks      import rawTaskItem
from    py_netasq.delphi.building.resTasks      import resTaskItem
from    py_netasq.delphi.building.sf6Tasks      import sf6TaskItem
from    py_netasq.delphi.building.vrsTasks      import vrsTaskItem

from    py_netasq.delphi.building.mailTasks     import mailTaskItem

from    py_netasq.delphi.building.taskFactory   import taskFactory

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if __name__ == '__main__' :
  print __file__
  print __doc__

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


class buildFileEditor(buildFileParser) :
    """\
      A "build file" wrapper: provides easy acces to settings options,
      through object attributes.
      
      
      constructor buildFileEditor(void)
      
      void      loadFile(string filename, bool raw)
      void      saveFile(string filename)
      
      void      reset(void)
      
      property object cvsTasks   : cvs
      property object dccTasks   : compiler
      property object resTasks   : resource building
      property object sf6Tasks   : setup factory
      property object vrsTasks   : versioning task list
    """
   
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Constructor - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __init__(self) :
        """\
          constructor buildFileEditor(void)
          
          Create a "build file" wrapper: provides easy acces to settings
          options.
        """
        buildFileParser.__init__(self, None)
        # basedir is used to resolve relative paths
        self._basedir  = None
        self._tasklist = dict()
        self._mailing  = mailTaskItem(False, None)
        
        # nota: there is actually no need to build a rawTaskList
        for aType in buildFileParser.TASKS_ORDER :
          self._tasklist[aType] = taskFactory.buildTaskList(aType)
        
        self.reset()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Special methods - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def loadFile(self, filename, raw=False) :
        """\
          void loadFile(string filename, bool raw)
          
          Open and parse given file; if file does not exist then an exception
          will be raised. Tasks lists are loaded from file.
          If string interpolation is set to True, then saving the same file
          will remove all "variables".
          
          e.g. in config file :
                aVar  = foo
                dummy = %(aVar)s\bar
              self. loadFile(cfgFile, False)
              => aVar  = foo
                 dummy = foo\bar
              self.saveFile(cfgFile)
              => in config file :
                 aVar  = foo
                 dummy = foo\bar
          
          param filename              config file path
          param raw                   extract data with no interpolation,
                                        defaults to False
        """
        self.reset()
        filename = asqPath.normalize(os.path.abspath(filename))
        # basedir is used to resolve relative paths
        self._basedir = os.path.dirname(filename)
        buildFileParser.readFile(self, filename)
        
        for aType in self._tasklist.iterkeys() :
          buildFileParser.extractTaskList(self, self._tasklist[aType], raw)
        # do not forget mail config
        buildFileParser.extractMailConfig(self, self._mailing, raw)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def saveFile(self, filename) :
        """\
          void saveFile(string filename)
          
          Write build file configuration to given 'file'. If file does not
          exist then it will simply be created.
          
          param filename              config file path
        """
        buildFileParser.setMailConfig(self, self.__mailing)
        for (aType, aList) in self._tasklist.iteritems() :
          buildFileParser.setTaskList(self, aList, aType)
        buildFileParser.writeFile(self, filename)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def reset(self) :
        """\
          void reset(void)
          
          Empty all tasks lists, reset parser, set basedir to current
          working directory.
        """
        # empty parser dataset
        self.clear()
        self._basedir  = os.getcwd()
        for key in self._tasklist.iterkeys() :
          self._tasklist[key].reset()
        self._mailing.reset()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def getTaskListForSection(self, aSection) :
        """\
          object getTaskListForSection(string aSection)
          
          Returns task list correponding to given 'aSection' section name;
          If no matching task list is found, then None is returned. name
          matching is not case sensitive.
          e.g.
            self.resTasks == self.getTaskListForSection('ResTasks')
            self.resTasks == self.getTaskListForSection('restAsks')
          
          param aSection              section name
        """
        try :
          tkType = buildFileParser.getTypeFromSection(aSection)
          result = self._tasklist[tkType]
        except KeyError :
          return None
        else :
          return resut


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _getCvsTasks(self) :
        """\
          object _getCvsTasks(void)
          
          Getter - get CVS tasks list
          
          return                      an instance of cvsTaskList
        """
        return self.__tasklist[cvsTaskItem.TASKTYPE]


    def _getDccTasks(self) :
        """\
          object _getDccTasks(void)
          
          Getter - get delphi compiler tasks list.
          
          return                      an instance of dccTaskList
        """
        return self._tasklist[dccTaskItem.TASKTYPE]


    def _getResTasks(self) :
        """\
          object _getResTasks(void)
          
          Getter - get ResManager tasks list.
          
          return                      an instance of resTaskList
        """
        return self._tasklist[resTaskItem.TASKTYPE]


    def _getSf6Tasks(self) :
        """\
          object _getSf6Tasks(void)
          
          Getter - get Setup Factory tasks list.
          
          return                      an instance of sf6TaskList
        """
        return self._tasklist[sf6TaskItem.TASKTYPE]


    def _getVrsTasks(self) :
        """\
          object _getVrsTasks(void)
          
          Getter - get ResModifier tasks list.
          
          return                      an instance of vrsTaskList
        """
        return self._tasklist[vrsTaskItem.TASKTYPE]


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    dccTasks = property(
                doc  = "delphi compiler tasks list",
                fget = _getDccTasks,
                fset = None,
                fdel = None )

    resTasks = property(
                doc  = "ResManager tasks list",
                fget = _getResTasks,
                fset = None,
                fdel = None )

    sf6Tasks = property(
                doc  = "Setup factory tasks list",
                fget = _getSf6Tasks,
                fset = None,
                fdel = None )

    vrsTasks = property(
                doc  = "ResModifier tasks list (versioning)",
                fget = _getVrsTasks,
                fset = None,
                fdel = None )

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # class buildFileEditor - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -