#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
Define a basic "Task item" list used by the building utility.
Notice: lists keep items order, dictionaries don't

Since
  Python 2.2

Classes
  class         taskItem(bool enabled)
  class         taskList(bool enabled) extends list, taskItem
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.8 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  copy
import  os

import  py_netasq.commonlib.asqDebug            as asqDebug
import  py_netasq.commonlib.asqPath             as asqPath
import  py_netasq.commonlib.asqString           as asqString

from    py_netasq.commonlib.asqCmdLine          import asqCmdLine

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if __name__ == '__main__' :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class taskItem(object) :
    """\
      A simple task used by the automated building system. For single
      taskItem, there is a huge difference between 'None' attribute value
      and an empty string; each attribute that is equal to 'None', means
      "use taskList value for that attribute", whereas empty string does
      override tasklist values, meaning "no value for that attribute".
      
      
      constructor taskItem(bool enabled)
      
      void      debug(int tab)
      
      void      normalize(void)
      
      void      assign(object source)
      
      bool      isValid(void)
      bool      isEmpty(void)
      bool      isCompatibleWith(string aType)
      
      void      reset(void)
      
      void      fromList(list aList, string listType)
      void      toList(list aList, bool filtered)
      
      list      getAsList(bool filtered)
      
      object    mergeWithList(list aList, bool force, string listType)
      object    mergeWithTask(object task, bool force)
      
      object    toCmdLine(void)
      
      void      expandPaths(void)
      void      completePaths(string base)
      
      property list   attributes : task attributes
      property dict   miscvalues : unrecognized attributes
      
      property bool   enabled    : is task active ?
      property string logfile    : log file
      property string comments   : about this project
      
      property string tasktype   : task type as a string
    """
   
    TASKTYPE         = "basicTask"
    KEYWORD_TASKTYPE = "tasktype"

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Constructor - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __init__(self, enabled) :
        """\
          constructor taskItem(bool enabled)
          
          A task; Its attributes are defined by the 'attributes' list.
          e.g. attributes = ['logFile', 'comments']
            => task has task.logFile and task.comments attributes.
            
          param enabled               is task enabled
        """
        super(taskItem, self).__init__()
        object.__setattr__(self, '_tasktype'  , taskItem.TASKTYPE)
        object.__setattr__(self, '_commonbase', list())
        object.__setattr__(self, '_attributes', list())
        object.__setattr__(self, '_miscvalues', dict())
        # task type compatibility
        self._compatWith = (taskItem.TASKTYPE,)
        # common base shared by all tasks
        self._commonbase.append('enabled')
        self._commonbase.append('comments')
        self._commonbase.append('logfile')
        self._attributes.extend(self._commonbase)
        # initialize all attributes to None
        self.reset(True)
        object.__setattr__(self, 'enabled', bool(enabled))


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Special methods - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __setattr__(self, name, value) :
        """\
          void __setattr__(string name, var value)
          
          Called when an attribute assignment is attempted. This is called
          instead of the normal mechanism (i.e. store the value in the
          instance dictionary). name is the attribute name, value is the
          value to be assigned to it.
          
          Each time an attribute value is set, 'normalize()' should be
          called to ensure consistency.
          
          param name                  attribute to set
          param value                 new value
        """
        object.__setattr__(self, name, value)
        if name in self._attributes : self.normalize()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __nonzero__(self) :
        """\
          bool __nonzero__(void)
          
          Called to implement truth value testing, and the built-in operation
          bool(); should return False or True, or their integer equivalents
          0 or 1.
          Tells whether or not the task is "empty": an empty task has no
          value defined (None values) for its attributes. Performed right
          after a "task.reset()" this function should return "False".
          
          return                      is task not empty
        """
        doSkip = ('enabled')
        
        for key in self._attributes :
          if key in doSkip :
            continue
          elif getattr(self, key) is not None :
            return True
        else :
          # Style Guide for Python Code : empty sequences are false
          return bool(self._miscvalues)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __str__(self) :
        """\
          string __str__(void)
          
          Called by the str() built-in function and by the print statement
          to compute the "informal" string representation of an object. This
          differs from __repr__() in that it does not have to be a valid
          Python expression: a more convenient or concise representation may
          be used instead. The return value must be a string object
          
          Returns a pseudo command line: only the most important attributes
          will be displayed. If current task is found empty, then an empty
          string will be returned.
          
          return                      pseudo command line string
        """
        return ''


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def debug(self, tab=0) :
        """\
          void debug(int tab)
          
          Prints out the main attributes values. Each string to print, will
          be prepended with 'tab' space characters.
          
          param tab                   number of char to prepend,
                                       defaults to 0
        """
        prefix = ' ' * tab
        
        print asqDebug.debugValue('tasktype', self._tasktype, tab, False)
        for key in self._attributes :
          print asqDebug.debugValue(key, getattr(self, key), tab)
         
        if not self._miscvalues : return
          
        print '%s[ TASK misc.  ] - - - - - - - - - - - - - -' % (prefix,)
        for (key, val) in self._miscvalues.iteritems() :
          print asqDebug.debugValue(key, val, tab)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def reset(self, force=True) :
        """\
          void reset(bool force)
          
          Set object's attributes to None. If 'force' is set to False, then
          existing attributes won't be erased, only missing ones will be
          created and set to None. If 'force' is set to True, then all
          values will be set to None, and , "_miscvalues" dictionary will
          be cleared.
          
          param force                 reset all values
        """
        if not force :
          for key in self._attributes :
            if not hasattr(self, key) : object.__setattr__(self, key, None)
        else :
          for key in self._attributes :
            object.__setattr__(self, key, None)
          self._miscvalues.clear()
         
        self.normalize()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def normalize(self) :
        """\
          void normalize(void)
          
          Correct values types for some peculiar attributes; e.g. the
          "enabled" attribute must be a boolean value, "None" value will be
          converted to False
        """
        if not isinstance(self.enabled, basestring) :
          value = bool(self.enabled)
        else :
          value = asqString.stringToBool(self.enabled)
        object.__setattr__(self, 'enabled', value)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def assign(self, source) :
        """\
          void assign(object source)
          
          Copies object's attributes and miscvalues from given 'source'.
          previous settings are lost.
          
          param source                task source
        """
        self.fromList(source.toList())


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def isValid(self) :
        """\
          bool isValid(void)
          
          Tells whether or not the task is "valid", id est, if mandatory
          attributes are set; an empty taskItem is not valid, but an
          invalid taskItem may not be empty ;)
          
          return                      is task valid
        """
        #~ return not self.isEmpty()
        return bool(self)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def isEmpty(self) :
        """\
          bool isEmpty(void)
          
          Tells whether or not the task is "empty": an empty task has no
          value defined (None values) for its attributes. Performed right
          after a "task.reset()" this function should return "True".
          
          return                      is task empty
        """
        doSkip = ('enabled')
        
        for key in self._attributes :
          if key in doSkip :
            continue
          elif getattr(self, key) is not None :
            return False
        else :
          # Style Guide for Python Code : empty sequences are false
          return not(self._miscvalues)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  
    def isCompatibleWith(self, aType) :
        """\
          bool isCompatibleWith(string aType)
          
          Return whether or not the given 'aType' is compatible with
          current task type.
          
          param aType                 a task type
        """
        return aType in self._compatWith


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _isValidString(self, name) :
        """\
          bool _isValidString(string name)
          
          Return whether or not the attribute named 'name' is a valid string:
          if it is ACTUALLY a string, and not empty. An "AttributeError" is
          raised if the named attribute does not exist.
          
          param name                  attribute name
          return                      is attribute a valid string value
        """
        value = getattr(self, name)
        return isinstance(value, basestring) and value


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _isValidList(self, name) :
        """\
          bool _isValidList(string name)
          
          Return whether or not the attribute named 'name' is a valid list:
          if it is ACTUALLY a list, and not empty. An "AttributeError" is
          raised if the named attribute does not exist.
          
          param name                  attribute name
          return                      is attribute a valid list value
        """
        value = getattr(self, name)
        return isinstance(value, list) and value


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _fromList(self, aList, compatible) :
        """\
          void _fromList(list aList, bool compatible)
          
          Overwrite attributes with those provided by given list. If
          'compatible' is set to False, then only the "commonbase" attributes
          will be modified, all others attributes are simply ignored.
          Else, if list is compatible, then unrecognised attributes are added
          to the "miscvalues" list.
          e.g. [('comments', 'dummy comment'), ('logFile', None), ('foo', 'bar')]
               _fromList(aList, True)
              => self.comments = 'dummy comment'
                 self.logFile  = None
                 self.miscvalues['foo'] = 'bar'
          
          Please note: current object is NOT RESET.
          
          param aList                 list of attributes
          param compatible            is list compatible with task
        """
        if not compatible :
          for (key, val) in aList :
            if key in self._commonbase :
              object.__setattr__(self, key, val)
        else :
          for (key, val) in aList :
            if key in self._attributes :
              object.__setattr__(self, key, val)
            else :
              self._miscvalues[key] = val
        self.normalize()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  
    def _mergeWithList(self, aList, compatible, force) :
        """\
          object _mergeWithList(list aList, bool compatible, bool force)
          
          Return a copy of current object, but with "None" attributes values
          replaced with those provided by 'aList'.
          If 'compatible' is set to False, then only attributes marked as
          "commonbase" will be replaced, other ones are ignored.
          Except if 'force' is set to True, "enabled" and "comments"
          properties will not be replaced, even if they do equal "None".
          
          If list is found compatible, items that are not recognised as
          attributes (or commonbase) will be added to the 'miscvalues'
          sequence.
           
          param aList                 list of attributes
          param compatible            is list compatible with task
          param force                 force attributes replacement,
          return                      a new task
        """
        assert aList is not None, "no valid list supplied"
        
        result = copy.deepcopy(self)
        doSkip = ('comments', 'enabled')
        
        if not compatible :
          for (key, val) in aList :
            if key in doSkip and not force :
              continue
            elif key not in self._commonbase :
              continue
            elif getattr(self, key) is None :
              setattr(result, key, val)
        else :
          for (key, val) in aList :
            if key in doSkip and not force :
              continue
            elif key not in self._attributes :
              if self._miscvalues.get(key, None) is None :
                result.miscvalues[key] = val
            elif getattr(self, key) is None :
              setattr(result, key, val)
         
        return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def fromList(self, aList, listType=None) :
        """\
          void fromList(list aList, string listType)
          
          Reset object's attributes and then overwrite attributes with those
          provided by given list. If there is not "tasktype" value found in
          list, then 'listType' is assumed. If type is not compatible with
          current object, then only the "commonbase" attributes will be
          modified, all others attributes are simply ignored.
          Else, if list is compatible, then unrecognised attributes are added
          to the "miscvalues" list.
          
          e.g. [('comments', 'dummy comment'), ('logFile', None), ('foo', 'bar')]
            => self.comments = 'dummy comment'
               self.logFile  = None
               self.miscvalues['foo'] = 'bar'
           
          param aList                 list of attributes
          param listType              list assumed tasktype, defaults to None
        """
        self.reset(True)
        
        buffer = aList[:]
        # remove tasktype value
        tkType = self._extractTaskType(buffer, listType)
        
        self._fromList(buffer, self.isCompatibleWith(tkType))


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def toList(self, filtered=False) :
        """\
          list toList(bool filtered)
          
          Return object's attributes, using the following format 
          [(attr, val), (attr, val), ...]. If filtered is set to True, then
          None attributes values won't be added to list.
          
          e.g. self.comments = 'dummy comment'
               self.logFile  = None
               self.miscvalues['dummy'] = None
            
              toList(False)
            => [('comments', 'dummy comment'), ('logFile', None), ('dummy', None)]
              toList(True)
            => [('comments', 'dummy comment')]
            
          param filtered              skip None values
          return                      list of attributes
        """
        result = [ (taskItem.KEYWORD_TASKTYPE, self._tasktype) ]

        for key in self._attributes :
          result.append( (key, getattr(self, key)) )
        result.extend(self._miscvalues.items())
        if filtered :
          result[:] = filter(lambda x: x[1] is not None, result)
        
        return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  
    def mergeWithList(self, aList, force=False, listType=None) :
        """\
          object mergeWithList(list aList, bool force, string listType)
          
          Return a copy of current object, but with "None" attributes values
          replaced with those provided by 'aList'.
          If 'aList' is not found compatible with current object (i.e. if
          "tasklist" option found in 'aList' does not match), then only
          attributes marked as "commonbase" will be replaced, other ones are
          ignored. If 'force' is not set, then 'enabled' and 'comments'
          attributes won't be replaced, even if they equals "None".
          
          If list is found compatible, items that are not recognised as
          attributes (or commonbase) will be added to the 'miscvalues'
          sequence.
           
          param aList                 list of attributes
          param force                 force attributes replacement,
                                        defaults to False
          param listType              list assumed tasktype, defaults to None
          return                      a new task
        """
        buffer = aList[:]
        # remove tasktype value
        tkType = self._extractTaskType(buffer, listType)
        compat = self.isCompatibleWith(tkType)

        return self._mergeWithList(buffer, compat, force)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  
    def mergeWithTask(self, aTask, force=False) :
        """\
          object mergeWithTask(list aTask, bool force)
          
          Return a copy of current object, but with "None" attributes values
          replaced with those provided by 'task'.
          If 'aTask' is not compatible, then only commonbase attributes are
          considered.
          For further infos see "mergeWithList" function.
           
          param aTask                 another task
          param force                 force attributes replacement,
                                        defaults to False
          return                      a new task
        """
        assert aTask is not None, "no valid task supplied"
        
        return self.mergeWithList(aTask.toList(True), force)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _extractTaskType(self, aList, default=None) :
        """\
          string _extractType(list aList, var default)
          
          If found, REMOVE and return tasktype value from given list, else
          'default' value is returned and given 'aList' is not modified.
          
          param aList                 list to extract values from
          param default               returned if type is not found,
                                        defaults to None
          return                      a tasktype, or None value
        """
        buffer = dict(aList)
        
        if taskItem.KEYWORD_TASKTYPE not in buffer :
          return default
        else :
          result = buffer[taskItem.KEYWORD_TASKTYPE]
          del buffer[taskItem.KEYWORD_TASKTYPE]
          aList[:] = buffer.items()
          return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def toCmdLine(self) :
        """\
          object toCmdLine(void)
          
          Returns a command line as an asqCmdLine instance. If current task
          is found empty or invalid, then an empty command is returned.
          
          return                      asqCmdLine instance
        """
        return asqCmdLine()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def expandPaths(self) :
        """\
          void expandPaths(void)
          
          Permanently replace task's file paths with "normalized" versions,
          "expandvars()", "normpath()" and "normcase()" are applied to paths.
        """
        self.logfile = self._normPath(self.logfile)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def completePaths(self, base) :
        """\
          void completePaths(string base)
          
          Permanently replace task's file paths with absolute versions,
          using 'base' value as base dir (thus the name ;).
          
          param base                  base for absolute path
        """
        self.logfile = self._completePath(base, self.logfile)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _normPath(self, path) :
        """\
          var _normPath(string path)
          
          Normalize given 'path': apply "expandvars()", "normpath()" and
          "normcase()" to path. Return None value if 'path' is an empty
          string, or already equals None.
          
          param path                  path to normalize
          return                      either None value or normalized path
        """
        if path is None :
          return path
        elif 0 < len(path) :
          return asqPath.normalize(os.path.expandvars(path))
        else :
          return None
 

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _completePath(self, base, path) :
        """\
          var _completePath(string base, string path)
          
          Return absolute version of given 'path', using 'base' value as
          base dir (thus the name ;); return None value if 'path' is an
          empty string, or already equals None
          
          param base                  base for absolute path
          param path                  path to convert
          return                      either None value or absolute path
        """
        if path is None :
          return path
        elif 0 < len(path) :
          return asqPath.absolute(path, base)
        else :
          return None
 

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _getAttributes(self) :
        """\
          object _getAttributes(void)
          
          Getter - get task attributes list.
          
          return                      a list
        """
        return self._attributes[:]


    def _getMiscValues(self) :
        """\
          object _getMiscValues(void)
          
          Getter - get 'other values' list.
          
          return                      a dictionary
        """
        return self._miscvalues


    def _getTaskType(self) :
        """\
          string _getTaskType(void)
          
          Returns task item type as a string
          
          return                      task type
        """
        return self._tasktype


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    attributes = property(
                doc  = "available task attributes",
                fget = _getAttributes,
                fset = None,
                fdel = None )

    miscvalues = property(
                doc  = "unrecognized attributes",
                fget = _getMiscValues,
                fset = None,
                fdel = None )

    tasktype   = property(
                doc  = "task type",
                fget = _getTaskType,
                fset = None,
                fdel = None )

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # class taskItem  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


class taskList(taskItem, list) :
    """\
      A list of task items, and a task by itself; a taskList does define
      the default attributes values for each of the owned taskItems.
      Unlike taskItem, empty strings and 'None' values share the same
      meaning : "no value for that task attribute".
      
      
      constructor taskList(bool enabled)
      
      void      debug(int tab)
      
      object    appendNewTask(bool enabled)
      object    toCmdLine(void)
      
      void      normalize(void)
      
      bool      isEmpty(void)
      void      clear(void)
      void      reset(void)
    """


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Constructor - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    #~ def __init__(self, enabled) :
        #~ """\
          #~ constructor taskList(void)
          
          #~ A list of task items, and a task by itself; a taskList does define
          #~ the default attributes values for each of the owned taskItems.
          #~ If a whole taskList is disabled, then NONE of its task items will
          #~ be done.
          
          #~ param enabled               is task enabled
        #~ """
        #~ super(taskList, self).__init__(enabled)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Special methods - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __nonzero__(self) :
        """\
          bool __nonzero__(void)
          
          Called to implement truth value testing, and the built-in operation
          bool(); should return False or True, or their integer equivalents
          0 or 1.
          Tells whether or not the taskList is "empty": an empty taskList
          has no items (len(self) == 0) and  no value defined (None values)
          for its attributes.
          
          return                      is task not empty
        """
        doSkip = ('enabled')
        
        if 0 != len(self) :
          return True
        for key in self._attributes :
          if key in doSkip :
            continue
          elif bool(getattr(self, key)) :
            return True
        else :
          # Style Guide for Python Code : empty sequences are false
          return bool(self._miscvalues)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def debug(self, tab=0) :
        """\
          void debug(int tab)
          
          Prints out the main attributes values. Each string to print, will
          be prepended with 'tab' space characters.
          Each item MUST have a "debug()" method defined.
          
          param tab                   number of char to prepend,
                                       defaults to 0
        """
        prefix = ' ' * tab
        length = len(self)
        
        print asqDebug.debugValue('count', length, tab, False)
        super(taskList, self).debug(tab)
        if 0 == length : return
         
        print '%s[ LIST items  ] - - - - - - - - - - - - - -' % (prefix,)
        prefix = ' ' * (1 + tab)
        for i in xrange(length) :
          print '%s+items[%u]' % (prefix, i)
          self[i].debug(1 + tab)
          print prefix


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def appendNewTask(enabled=False) :
        """\
          object appendNewTask(bool enabled)
          
          Returns a brand new task, which has been added to taskList.
          Somehow, it looks a little bit like a "factory" function.
          Actually this function does raise an exception.
          
          param enabled               is task enabled, defaults to False
          return                      None
        """
        raise Exception, "taskList.appendNewTask : abstract method"


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def toCmdLine(self) :
        """\
          object toCmdLine(void)
          
          Returns a command line as an asqCmdLine instance. An Exception
          will be raised if this function is called.
          
          return                      asqCmdLine instance
        """
        raise Exception, "A task list cannot be converted to an asqCmdLine"


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def normalize(self) :
        """\
          void normalize(void)
          
          Replace empty string attributes with None values.
        """
        super(taskList, self).normalize()
        for key in self._attributes :
          value = getattr(self, key)
          if isinstance(value, basestring) and 0 == len(value) :
            object.__setattr__(self, key, None)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def isEmpty(self) :
        """\
          bool isEmpty(void)
          
          Tells whether or not the taskList is "empty": an empty taskList
          has no items (len(self) == 0) and  no value defined (None values)
          for its attributes.
          
          return                      "is taskList empty ?"
        """
        doSkip = ('enabled')
        
        if 0 != len(self) :
          return False
        for key in self._attributes :
          if key in doSkip :
            continue
          elif bool(getattr(self, key)) :
            return False
        else :
          # Style Guide for Python Code : empty sequences are false
          return not bool(self._miscvalues)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def clear(self) :
        """\
          void clear(void)
          
          Empty tasks list.
        """
        self[:] = []


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def reset(self, force=True) :
        """\
          void reset(bool force)
          
          Set object's attributes to None. If 'force' is set to false, then
          missing attributes will be created, other attributes will remain
          unchanged. taskList is always cleared of all of its items.
           
          param force                 reset ALL attributes, defaults to True
        """
        super(taskList, self).reset(force)
        self.clear()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # class taskList  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -