#!/usr/bin/env python

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
asqString (basic) test suite
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.9 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  unittest

import  py_netasq.commonlib.asqString                   as asqString


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class test_asqString(unittest.TestCase) :

    def testProtos1(self) :
        """\
          check if missing parameters or invalid types do raise exceptions.
        """
        self.assertRaises(TypeError, asqString.reverse)
        self.assertRaises(TypeError, asqString.chunkSplit)
        self.assertRaises(TypeError, asqString.spliceString)
        self.assertRaises(TypeError, asqString.truncateString)
        self.assertRaises(TypeError, asqString.shortenString)
        self.assertRaises(TypeError, asqString.wrapString)
        self.assertRaises(TypeError, asqString.quoteString)
        self.assertRaises(TypeError, asqString.unQuoteString)
        self.assertRaises(TypeError, asqString.formatRndString)
        self.assertRaises(TypeError, asqString.stringToBool)


    def testProtos2(self) :
        """\
          check if missing parameters or invalid types do raise exceptions.
        """
        self.assertRaises(TypeError, asqString.reverse, 1)
        self.assertRaises(TypeError, asqString.reverse, list)
        
        self.assertRaises(TypeError, asqString.chunkSplit, 'aa')
        self.assertRaises(TypeError, asqString.chunkSplit, list, 3)
        
        self.assertRaises(TypeError, asqString.spliceString, 'aa', 'xx')
        self.assertRaises(TypeError, asqString.spliceString, 'aa', None, 'xx')
        self.assertRaises(TypeError, asqString.spliceString, 'aa', None, None, None)
        
        self.assertRaises(TypeError, asqString.truncateString, 5)
        self.assertRaises((AssertionError, TypeError), asqString.truncateString, 'aa', 5, dict)
        
        self.assertRaises(TypeError, asqString.shortenString, 5)
        self.assertRaises((AssertionError, TypeError), asqString.shortenString, 'aa', 5, dict)
        
        self.assertRaises((AttributeError, TypeError), asqString.wrapString, 5)
        self.assertRaises((AttributeError, TypeError), asqString.wrapString, dict, 5)
        
        self.assertRaises(TypeError, asqString.quoteString, False)
        self.assertRaises(TypeError, asqString.quoteString, 's', True)
        
        self.assertRaises(TypeError, asqString.unQuoteString, False)
        self.assertRaises(TypeError, asqString.unQuoteString, 's', True)
        
        self.assertRaises(TypeError, asqString.formatRndString, ())
        # the empty string is not a valid fstring formatter
        self.assertRaises(TypeError, asqString.formatRndString, '')
        
        self.assertRaises((AttributeError, TypeError), asqString.stringToBool, list)


    def testChunkSplit1(self) :
        """\
          check chunkSplit results.
        """
        self.assertEqual(
            ['10', '20', '30', '40', '5'],
            asqString.chunkSplit('102030405', 2)
          )
        self.assertEqual(
            ['1', '02', '03', '04', '05'],
            asqString.chunkSplit('102030405', -2)
          )


    def testReverse1(self) :
        """\
          check reverse results.
        """
        self.assertEqual(
            '504030201', asqString.reverse('102030405')
          )
        self.assertEqual(
            '102030405', asqString.reverse(asqString.reverse('102030405'))
          )


    def testSpliceString1(self) :
        """\
          check spliceString results.
        """
        buffer = 'hello world'
        
        result = asqString.spliceString(buffer, None, None, '')
        self.assertEqual('', result)
        
        result = asqString.spliceString(buffer, None, None, 'what the ...')
        self.assertEqual('what the ...', result)
        
        result = asqString.spliceString(buffer, 5, 5, '')
        self.assertEqual(buffer, result)
        
        # replacing space with ' beautiful '
        result = asqString.spliceString(buffer, 5, 6, ' beautiful ')
        self.assertEqual(buffer[:5] +' beautiful '+ buffer[6:], result)


    def testTruncateString1(self) :
        """\
          check truncateString results.
        """
        buffer = 'jacquesb@netasq.com, danielc@netasq.com, philipped@netasq.com'
        
        result = asqString.truncateString(buffer, -3, '...')
        self.assertEqual('', result)
        
        result = asqString.truncateString(buffer, len(buffer), '...')
        self.assertEqual(buffer, result)
        
        result = asqString.truncateString(buffer, 2, '...')
        self.assertEqual('', result)
        
        result = asqString.truncateString(buffer, len('...'), '...')
        self.assertEqual('...', result)
        
        result = asqString.truncateString(buffer, 5, '...')
        self.assertEqual('ja...', result)
        
        result = asqString.truncateString(buffer, 30, '...')
        self.assertEqual(30, len(result))
        
        result = asqString.truncateString(buffer, 20, '...')
        self.assertEqual(20, len(result))
        
        result = asqString.truncateString(buffer, 30, '...', ',')
        self.assertEqual('jacquesb@netasq.com,...', result)
        
        result = asqString.truncateString(buffer, 40, '...', '@')
        self.assertEqual('jacquesb@netasq.com, danielc@...', result)


    def testShortenString1(self) :
        """\
          check shortenString results.
        """
        buffer = 'c:\\python23\\lib\\site-packages\\py_netasq\\building\\core\\test'
        
        result = asqString.shortenString(buffer, -3, '...')
        self.assertEqual('', result)
        
        result = asqString.shortenString(buffer, len(buffer), '...')
        self.assertEqual(buffer, result)
        
        result = asqString.shortenString(buffer, 2, '...')
        self.assertEqual('', result)
        
        result = asqString.shortenString(buffer, len('...'), '...')
        self.assertEqual('...', result)
        
        result = asqString.shortenString(buffer, 5, '...')
        self.assertEqual('c...t', result)
        
        result = asqString.shortenString(buffer, 5, '...', '\\')
        self.assertEqual('c...t', result)
        
        # actually expected length should be lesser or equal
        result = asqString.shortenString(buffer, 30, '...')
        self.assertEqual(30, len(result))
        
        result = asqString.shortenString(buffer, 20, '...')
        self.assertEqual(20, len(result))
        
        result = asqString.shortenString(buffer, 30, '...', '\\')
        #~ print result, len(result)
        self.assert_(30 >= len(result))


    def testWrapString1(self) :
        """\
          check wrapString results.
        """
        buffer = """\
 Dennis:  "Listen. Strange women lying in ponds distributing swords \
is no basis for a system of government. Supreme executive power \
derives from a mandate from the masses, not from some farcical \
aquatic ceremony!". """
        
        result = asqString.wrapString(buffer, 0)
        #~ print ''
        #~ for item in result.split('\n') :
          #~ print repr(item)
        
        result = asqString.wrapString(buffer, 0, '>> ')
        #~ print ''
        #~ for item in result.split('\n') :
          #~ print repr(item)
        
        result = asqString.wrapString(buffer, 40)
        #~ print ''
        #~ for item in result.split('\n') :
          #~ print repr(item)
        
        result = asqString.wrapString(buffer, 50, '         ')
        #~ print ''
        #~ for item in result.split('\n') :
          #~ print repr(item)


    def testQuoteString1(self) :
        """\
          check quoteString results.
        """
        result = asqString.quoteString('', '"', '\\')
        self.assertEqual('""', result)
        
        result = asqString.quoteString('hello world', '"', '\\')
        self.assertEqual('"hello world"', result)
        
        result = asqString.quoteString('hello "world" !!', '"', '\\')
        self.assertEqual(r'"hello \"world\" !!"', result)
        
        result = asqString.quoteString('hello "world" !!', '"', '"')
        self.assertEqual(r'"hello ""world"" !!"', result)
        
        result = asqString.quoteString(r'hello "world\" !!', '"', '\\')
        self.assertEqual(r'"hello \"world\\" !!"', result)


    def testUnQuoteString1(self) :
        """\
          check unQuoteString results.
        """
        result = asqString.unQuoteString('', '"', '\\')
        self.assertEqual('', result)
        
        result = asqString.unQuoteString('hello world', '"', '\\')
        self.assertEqual('hello world', result)
        
        result = asqString.unQuoteString('"', '"', '\\')
        self.assertEqual('"', result)
        
        result = asqString.unQuoteString('""', '"', '\\')
        self.assertEqual('', result)
        
        result = asqString.unQuoteString(r'"hello \"world\" !!"', '"', '\\')
        self.assertEqual('hello "world" !!', result)
        
        result = asqString.unQuoteString(r'"hello ""world"" !!"', '"', '"')
        self.assertEqual('hello "world" !!', result)
        
        result = asqString.unQuoteString(r'"hello \"world\\" !!"', '"', '\\')
        self.assertEqual(r'hello "world\" !!', result)


    def testQuoteStringUnQuoteString1(self) :
        """\
          check quoteString and unQuoteString (used together) results.
        """
        buffer = ''
        result = asqString.quoteString(buffer, '"', '\\')
        self.assertEqual('""', result)
        result = asqString.unQuoteString(result, '"', '\\')
        self.assertEqual(buffer, result)
        
        buffer = 'hello world'
        result = asqString.quoteString(buffer, '"', '\\')
        self.assertEqual('"hello world"', result)
        result = asqString.unQuoteString(result, '"', '\\')
        self.assertEqual(buffer, result)
        
        buffer = r'hello "world\" !!'
        result = asqString.quoteString(buffer, '"', '\\')
        self.assertEqual(r'"hello \"world\\" !!"', result)
        result = asqString.unQuoteString(result, '"', '\\')
        self.assertEqual(buffer, result)
        
        buffer = 'hello "world" !!'
        result = asqString.quoteString(buffer, '"', '"')
        self.assertEqual(r'"hello ""world"" !!"', result)
        result = asqString.unQuoteString(result, '"', '"')
        self.assertEqual(buffer, result)


    def testStringToBool1(self) :
        """\
          check stringToBool results.
        """
        self.assertEqual(False, asqString.stringToBool('None'))
        self.assertEqual(False, asqString.stringToBool('False'))
        self.assertEqual(False, asqString.stringToBool(''))
        self.assertEqual(True, asqString.stringToBool('True'))
        self.assertEqual(True, asqString.stringToBool('arf'))


  # /class test_asqString - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
    unittest.main()

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -