#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
(Little) Less common, but still useful, windows only functions.

Require
  Python        2.3

Constants
  int           PLATFORM_WIN32s
  int           PLATFORM_WIN32_WINDOWS
  int           PLATFORM_WIN32_NT
  int           PLATFORM_WIN32_CE

Functions
  int           getWindowsPlatform(void)
  string        getMsDosPath(string path)
  
Classes
  class         winCmdHandler(object cmdLine)
                  extends py_netasq.commonlib.asqCommand::cmdHandler
  class         windowsCmd(void) - deprecated
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.16 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  os
import  sys

from    datetime                                import datetime

import  py_netasq.commonlib.asqCommand          as asqCommand
import  py_netasq.commonlib.asqSequence         as asqSequence
import  py_netasq.commonlib.asqString           as asqString

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

# Coping with dos charset :
#
# In the early 1980s there still were no agreed international standards like
# ISO-8859 or Unicode on how to expand US-ASCII for international users and
# many manufacturers invented their own encodings using hard-to-memorize
# numbers:
#
# * MS-DOS code pages
#
#  CP437 (DOSLatinUS)
#  The industry-standard IBM Personal Computer started out with the famous
#  code page CP437 with lots of box-drawing characters and a select few
#  foreign letters.
#
#  CP850 (DOSLatin1)
#  Some later MS-DOS versions allowed the changing of code pages on VGA
#  graphics cards to something like CP850 which presented the Latin1
#  repertoire in positions compatible to CP437 so that "line-drawing" still
#  worked.
#

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

# About windows command shells :
#
# "cmd.exe" shells (Windows NT, 2000 and XP) seem to have a friggin *HUGE*
# problem with double quoted parameters; In order to have a correct command
# with quoted params you need to add extra double quotes around the whole
# command.
#   e.g. command = '""exe" "arg1" "arg2""'
#
# "command.com" shells (Windows 95, 98 and ME), on the other hand, seems to
# behave as expected.
#   e.g. command = '"exe" "arg1" "arg2"' will do the trick fine
#
# You'd better check the following Python bug report :
#
# [ 512433 ] Quote handling in os.system & os.popen
#  http://sourceforge.net/tracker/?func=detail&aid=512433&group_id=5470&atid=105470
#   Status     : Closed
#   Resolution : Wont Fix *sigh*
#

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# We do not want to see magic numbers, do we ?
PLATFORM_WIN32s         = 0     # Win32s on Windows 3.1
PLATFORM_WIN32_WINDOWS  = 1     # Windows 95/98/ME
PLATFORM_WIN32_NT       = 2     # Windows NT/2000/XP
PLATFORM_WIN32_CE       = 3     # Windows CE

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def getWindowsPlatform() :
    """\
      int getWindowsPlatform(void)
      
      According to the underlying platform, one of the following values will
      be returned :
       -1                     : OS is not MS Windows
       PLATFORM_WIN32s        : Win32s on Windows 3.1
       PLATFORM_WIN32_WINDOWS : Windows 95/98/ME
       PLATFORM_WIN32_NT      : Windows NT/2000/XP
       PLATFORM_WIN32_CE      : Windows CE
      
      This function relies on 'sys.getwindowsversion()', which wraps the
      Win32 'GetVersionEx()' function; Unlike egenix's "platform" module,
      there should be no need to have Mark Hammond's win32 package installed.
      
      return                      current windows platform
    """
    if not sys.platform.startswith('win') :
      return -1
    else :
      return sys.getwindowsversion()[3]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def getMsDosPath(path) :
    """\
      string getMsDosPath(string path)
      
      A simple function to convert windows long file names into MS-DOS 8.3
      filenames; Be aware that this function has the following drawbacks :
      - names that are equals up to the eighth char will be converted to
       the same string.
       'long file 001.ext', 'long file 002.ext' ... => 'longfi~1.ext'
      - in the same way, extensions that are equals up to the third chars
       'long file 001.build' => 'longfi~1.bui'
       'long file 001.bui'   => 'longfi~1.bui'
      
      param path                  filename
      return                      MS-DOS 8.3 filename
    """
    path       = os.path.abspath(path).lower()
    path, kind = os.path.splitext(path)
    chunks     = path.split(os.sep)
    
    chunks = map(lambda s: s.replace(' ', ''), chunks)
    for i in range(len(chunks)) :
      if 8 < len(chunks[i]) : chunks[i] = chunks[i][:6] + '~1'
    return os.sep.join(chunks) + kind[:4]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class winCmdHandler(asqCommand.cmdHandler) :
    """\
      A simple class to ease command line execution under windows systems.
      Workarounds are used to overcome MS-DOS quote handling problem.
      An assert exception will be raised by contructor if current OS is not
      a MS windows system.
      
      Beware: this is a new-style class.
      
      constructor       winCmdHandler(object cmdLine)
    """

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __init__(self, aCmdLine=None) :
        """\
          constructor winCmdHandler(object aCmdLine)
          
          Create a new object intended to ease command line execution under
          windows systems. An assert exception will be raised by contructor
          if current OS is not recognised as a MS windows system.
          
          param aCmdLine              cmdLine to execute.
                                       defaults to None
        """
        super(winCmdHandler, self).__init__(aCmdLine)
        self.__flavor = getWindowsPlatform()
        assert -1 <> self.__flavor,                                      \
                "unrecognised windows platform version"


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _execute(self, cmdStr, raw, *arglist, **argdict) :
        """\
          var _execute(string cmdStr, bool raw, *arglist, **argdict)
          
          Execute inner command line and return the shell status.
          
          param cmdStr                command as a string to execute.
          param raw                   disable output formatting
          param *arglist              extra parameters list.
          param **argdict             extra named parameters.
          return                      shell status
        """
        # windows NT shell workaround, quote the whole command
        if PLATFORM_WIN32_NT == self.__flavor :
          cmdStr = '"%s"' % (cmdStr, )
        return super(winCmdHandler, self)._execute(cmdStr, raw, *arglist, **argdict)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    #~ def _processInput(self, line, raw, *arglist, **argdict) :
        #~ """\
          #~ void _processInput(string line, bool raw, *arglist, **argdict)
         
          #~ Process the command input line by line.
          
          #~ param line                  line from command input
          #~ param raw                   disable input formatting
          #~ param *arglist              extra parameters list.
          #~ param **argdict             extra named parameters.
        #~ """
        #~ if PLATFORM_WIN32_NT == self.__flavor :
          #~ line = asqString.unQuoteString(line, '"', '')
        #~ super(winCmdHandler, self)._processInput(line, raw, *arglist, **argdict)


    def _decodeOutput(self, line, raw, *arglist, **argdict) :
        """\
          string _decodeOutput(string line, bool raw)
          
          Command output might need some decoding.
          Note, even if "raw" is set to True, the output charset will
          be 'translated' from cp850 to latin1.
          
          param line                  line from command output.
          param raw                   disable output formatting
          param *arglist              extra parameters list.
          param **argdict             extra named parameters.
        """
        result = super(winCmdHandler, self)._decodeOutput(line, raw, *arglist, **argdict)
        # simply ignore encoding / decoding errors
        return result.decode('cp850', 'ignore').encode('latin1', 'ignore')


  # /class winCmdHandler  - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class windowsCmd(object) :
    """\
      Deprecated, kept for backward compatibility, use winCmdHandler instead.
      
      A simple class to ease command line execution under windows systems.
      Workarounds are used to overcome MS-DOS problems such as quote handling
      or shell charset.
      An assert exception will be raised by contructor if current OS is not
      a MS windows system.
      
      Beware: this is a new-style class.
      
      
      constructor windowsCmd(void)
      
      var       run(string command, string codepage)
      
      void      clearListeners(void)
      void      clearTimeStamp(void)
      
      property string  cmdOutput : last command's execution output
      property list    listeners : descriptors listening to command output
      property object  timestamp : command execution timestamp
    """
    
    # timestamp format
    DATETIME_FORMAT  = '%A, %d %b %Y %H:%M:%S'
    # Dos CodePage
    DEFAULT_CODEPAGE = 'cp850'

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Constructor - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __init__(self) :
        """\
          constructor windowsCmd(void)
          
          Create a new object intended to ease command line execution under
          windows systems. An assert exception will be raised by contructor
          if current OS is not recognised as a MS windows system.
        """
        super(windowsCmd, self).__init__()
        self.__flavor = getWindowsPlatform()
        assert -1 <> self.__flavor,                                      \
                "unrecognised windows platform version"
        self.__listeners = list()
        self.__timestamp = None
        self.__cmdoutput = os.tmpfile()
        # listeners without duplicates
        self.__uniques   = list()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Special methods - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  
    def run(self, command, codepage=DEFAULT_CODEPAGE) :
        """\
          var run(string command, string codepage)
          
          Execute given 'command' if given, or object's command build upon
          self.items. The shell status is returned, 'None' if execution was
          succesful, any other value mean failure. Command's output, stdout
          AND stderr, is saved to each descriptors found in self.output list.
          
          param command               command to execute instead,
                                       defaults to None
          param codepage              command output codepage, defaults
                                        to DEFAULT_CODEPAGE
          return                      shell status
        """
        buffer = str(command)
        # bool(str(None)) => bool('None') => True
        assert command and buffer, "no valid command supplied"
        
        # windows NT shell workaround, quote the whole command
        if PLATFORM_WIN32_NT == self.__flavor :
          buffer = '"%s"' % (buffer,)
         
        self._resetCmdOutput()
        # there is no use to have a stderr, error messages are written
        # to stdout, stderr only provides exit status.
        stdin, stdout = os.popen4(buffer)
        self._normalizeListeners()
        self._recordTimeStamp()
        self._recordCmdLine(command)
        stdin.close()
        try :
          self._recordOutput(stdout, codepage)
        finally :
          return stdout.close()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _resetCmdOutput(self) :
        """\
          void _resetCmdOutput(void)
          
          Empty command output buffer.
        """
        self.__cmdoutput.seek(0, 0)
        self.__cmdoutput.truncate(0)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _normalizeListeners(self) :
        """\
          void _normalizeListeners(void)
          
          Build a listeners list without any duplicates; the same message
          won't be send twice to a listener.
        """
        self.__uniques[:] = asqSequence.unique(self.__listeners)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _record(self, value, raw=False) :
        """\
          void _record(string value, bool raw)
          
          Records given 'value': send string to all listening descriptors of
          the 'self.listeners' list. If 'raw' is set to False, then given
          string will be wrapped.
          
          param value                 value to record
          param raw                   record raw string, defaults to False
        """
        if not raw :
          value = asqString.wrapString(value, 78, ' ')
        self.__cmdoutput.write(value)
        #~ for item in self.__listeners :
        for item in self.__uniques :
          item.write(value)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _recordCmdLine(self, command) :
        """\
          void _recordCmdLine(var command)
          
          Record command line. Given command line may be an asqCmdLine
          instance.
          
          param command               command line to record
        """
        self._record(str(command) +'\n')


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _recordTimeStamp(self) :
        """\
          void _recordTimeStamp(void)
          
          Record command execution timestamp as a "datetime" instance.
        """
        self.__timestamp = datetime.today()
        buffer = self.__timestamp.strftime(windowsCmd.DATETIME_FORMAT)
        self._record(buffer +'\n', True)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _recordOutput(self, source, codepage) :
        """\
          void _recordOutput(file source, string codepage)
          
          Write output message, read from 'source', to each item of the
          command's 'listeners' list. In the same time, those messages are
          recoded from "CP437" IBM MS-DOS charset to "latin1" charset.
          Each listener must be valid a file descriptor, or at least behave
          the same.
          
          param source                messages source
          param codepage              source codepage
        """
        buffer = source.readline()
        while buffer :
          buffer = buffer.decode(codepage).encode('latin1')
          self._record(buffer)
          buffer = source.readline()
        self._record('\n')


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def clearListeners(self) :
        """\
          void clearListeners(void)
          
          Empty listeners lists.
        """
        self.__listeners[:] = []


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def clearTimeStamp(self) :
        """\
          void clearTimeStamp(void)
          
          Empty 'timestamp' attributes.
        """
        self.__timestamp = None


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _getCmdOutput(self) :
        """\
          string _getCmdOutput(void)
          
          Returns the very last command output.
          
          return                      last command output
        """
        offset = self.__cmdoutput.tell()
        self.__cmdoutput.seek(0, 0)
        result = self.__cmdoutput.read()
        self.__cmdoutput.seek(offset, 0)
        return result.strip()


    def _getListeners(self) :
        """\
          list _getListeners(void)
          
          Returns the list of descriptors listening to command output.
          
          return                      descriptors list
        """
        return self.__listeners


    def _getTimestamp(self) :
        """\
          object _getTimestamp(void)
          
          Returns command execution timestamp as a "datetime" instance.
          
          return                      command execution timestamp
        """
        return self.__timestamp


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    cmdOutput = property(
                  doc  = "last command's execution output",
                  fget = _getCmdOutput,
                  fset = None,
                  fdel = None )

    listeners = property(
                  doc  = "list of descriptors listening to command output",
                  fget = _getListeners,
                  fset = None,
                  fdel = None )

    timestamp = property(
                  doc  = "command execution timestamp",
                  fget = _getTimestamp,
                  fset = None,
                  fdel = None )


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # class windowsCmd  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
  print __file__
  print __doc__

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -