#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
Defines a custom wrapper class around subversion commands.

Require
  Python        2.3
  PySVN         1.3

Classes
  class         objectSvn()
                  extends object
"""


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

try :
  import pysvn
except :
  sys.exit(
    '%s: error: failed to import pysvn module, '
    'please install it and check your python path' % (__file__, )
  )


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -



class objectSvn(object) :
    """\
    """

    def __init__(self, config_dir='') :
        """\
          constructor objectSvn(string config_dir)
          
          Create a new object intended to ease SubVersion command execution.
          The path to user's subversion configuration directory may be
          passed as 'config_dir' argument. If no path is provided, then the
          the default subversion configuration is used.
          
          param config_dir            path to svn configuration directory
                                        defaults to ''
        """
        self._client   = None
        # login phase, username and password
        self._username = ''
        self._password = ''
        # log message
        self._message  = ''
        
        self.__setupClient(config_dir)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __setupClient(self, config_dir) :
        """\
        """
        self._client = pysvn.Client(config_dir)
        # "long" exception style, complete with error code
        self._client.exception_style = 1
        
        self._client.callback_cancel            = self.__getCancel
        self._client.callback_get_login         = self.__getLogin
        self._client.callback_get_log_message   = self.__getLogMessage
        
        self._client.callback_notify            = self._notification
        
        self._client.callback_ssl_client_cert_prompt            = self.__sslClientCertPrompt
        self._client.callback_ssl_client_cert_password_prompt   = self.__sslClientCertPasswordPrompt
        self._client.callback_ssl_server_prompt                 = self.__sslServerPrompt
        self._client.callback_ssl_server_trust_prompt           = self.__getSslServerTrustPrompt


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _needUsername(self) :
        """\
          string _needUsername(void)
          
          Should return a 'username' string, needed by PySVN 'get_login'
          callback. This method should be overriden.
        """
        return self._username


    def _needPassword(self) :
        """\
          string _needPassword(void)
          
          Should return a 'password' string, needed by PySVN 'get_login'
          callback. This method should be overriden.
        """
        return self._password


    def _needLogMessage(self) :
        """\
          string _needMessage(void)
          
          Returns a log message which may be required by some commands.
          This method should be overriden.
        """
        return self._message


    def _needSslServerConfirm(self, trust_data) :
        """\
          int _needSslServerConfirm(dict trust_data)
          
          Returns whether or not a certificate should be trusted.
          Since certifcated can be temporary trusted (ex: for the current
          session), an int value may be returned insted of a boolean;
          
            0 => reject
            1 => temporary accept
            2 => permanent accept
          
          This method should be overriden.
        """
        return 1 # temporary accept


    def _notification(arg_dict) :
        """\
          void _notification(dict arg_dict)
          
          Svn client callback used each time an interesting event occurs
          during a command execution. Check PySVN documentation for
          further infos.
          This method should be overriden.
        """
        pass


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __getCancel(self) :
        """\
          boolean __getCancel(void)
          
          Svn client callback called frequently during long running
          commands. If 'True' is returned then the svn command may be
          cancelled before completion.
          
          Without threads, this seems a bit useless actually;
          From PySVN documentation :
          "A Client object can only be used on one thread at a time. If
           two threads attempt to call methods of Client at the same time
           one of the threads will get a pysvn.ClientError exception with
           the value 'client in use on another thread'."
          
          return                      False
        """
        return False


    def __getLogin(self, realm, username, may_save) :
        """\
          tuple __getLogin(string realm, string username, boolean may_save)
          
          Svn client callback called whenever username and password are
          needed in the realm to access a repository (if there are
          no cached credentials).
        """
        assert isinstance(username, basestring)
        
        usrval = None
        passwd = None
        
        if username :
          usrval = username
        else :
          usrval = self._needUsername()
        
        passwd = self._needPassword()
        
        assert isinstance(usrval, basestring)
        assert isinstance(passwd, basestring)
        
        if 0 == len(usrval) :
          return False, '', '', False
        else :
          return True, usrval, passwd, False


    def __getLogMessage(self) :
        """\
          string __getLogMessage(void)
          
          Svn client callback used when a log message is required to
          complete the current subversion command.
        """
        buffer = self._needLogMessage()
        if buffer is None :
          return False, ''
        else :
          return True, buffer


    def __sslClientCertPrompt(self) :
        pass

    def __sslClientCertPasswordPrompt(self) :
        pass

    def __sslServerPrompt(self) :
        pass


    def __getSslServerTrustPrompt(self, trust_data) :
        """\
          tuple __getSslServerTrustPrompt(dict trust_data)
          
          Svn client callback used each time an HTTPS server presents a
          certificate and subversion is not sure if it should be trusted.
          The callback is called with information about the certificate
          in 'trust_data' dict.
          
          pysvn expect the callback to return a tuple of three values
          '(retcode, accepted_failures, save)'.
          
          - retcode             boolean, False if no username and password
                                are available. True if subversion is to use
                                the username and password.
          - accepted_failures   int, the accepted failures allowed
          - save                boolean, return True if you want subversion
                                to remember the certificate in the
                                configuration directory. return False to
                                prevent saving the certificate.
        """
        buffer = self._needSslServerConfirm(trust_data)
        
        if 1 == buffer : # temporary accept
          return True, trust_data['failures'], False
        elif 2 == buffer : # permanent accept
          return True, trust_data['failures'], True
        else :
          return False, 0, False


  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _getClient(self) :
        """\
          object _getClient(void)
          
          Getter - client property
        """
        assert self._client is not None
        
        return self._client


    def _getUsername(self) :
        """\
          string _getUsername(void)
          
          Getter - username property
        """
        return self._username


    def _getPassword(self) :
        """\
          string _getPassword(void)
          
          Getter - password property
        """
        return self._password


    def _getMessage(self) :
        """\
          string _getMessage(void)
          
          Getter - message property
        """
        return self._message


  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _setUsername(self, value) :
        """\
          void _setUsername(string value)
          
          Setter - username property
        """
        if value is None :
          self._username = ''
        elif isinstance(value, basestring) :
          self._username = value


    def _setPassword(self, value) :
        """\
          void _setPassword(string value)
          
          Setter - password property
        """
        if value is None :
          self._password = ''
        elif isinstance(value, basestring) :
          self._password = value


    def _setMessage(self, value) :
        """\
          void _setMessage(string value)
          
          Setter - message property
        """
        if value is None :
          self._message = ''
        elif isinstance(value, basestring) :
          self._message = value


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    client      = property(
        doc  = "PySVN client object",
        fget = _getClient,
        fset = None,
        fdel = None
      )

    username    = property(
        doc  = "svn client username",
        fget = _getUsername,
        fset = _setUsername,
        fdel = None
      )

    password    = property(
        doc  = "svn client password",
        fget = _getPassword,
        fset = _setPassword,
        fdel = None
      )

    message     = property(
        doc  = "command log message",
        fget = _getMessage,
        fset = _setMessage,
        fdel = None
      )


  # /class objectSvn  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -