#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
(Little) Less common, but still useful, string manipulations.

Require
  Python        2.3

Constants
  list          FALSE_VALUES

Functions
  string        reverse(string s)
  list          chunkSplit(string s, int size)

  string        spliceString(string s, int start, int end, string value)

  string        shortenString(string s, int width, string ellipsis, string breakAt)
  string        wrapString(string s, int width, string insert)
  
  string        quoteString(string s, string quote, string escape)
  string        unQuoteString(string s, string quote, string escape)
  
  string        formatRndString(string format, int rndmin, int rndmax)
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.14 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  random
import  sys

import  asqTypes

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

# string values considered as False boolean values
FALSE_VALUES = ('', 'None', 'False')

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def reverse(s) :
    """\
      string reverse(string s)
      
      Returns a copy of given string, with characters in reverse order;
      a simple alias to s[::-1] (python 2.3)
      
      e.g.
        reverse('102030405')
        => '504030201'
      
      param s                     string to reverse
      return                      reversed string
    """
    if not isinstance(s, basestring) :
      raise TypeError("type mismatch `s` parameter, expecting string")
    
    return s[::-1]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def chunkSplit(s, size) :
    """\
      list chunkSplit(string s, int size)
      
      Split given string on parts of equal 'size'; if size is negative,
      then string is split from right to left.
      e.g.
        chunkSplit("102030405",  2)
        => ['10', '20', '30', '40', '5']
        chunkSplit("102030405", -2)
        => ['1', '02', '03', '04', '05']
      
      Based upon an activestate's cookbook recipe by Dmitry Vasiliev
      "Generator for splitting a string on parts of equal size"
      > http://aspn.activestate.com/ASPN/Cookbook/Python/Recipe/302069
      
      param s                     string to wrap
      param size                  size of a chunk
      return                      list of string chunks
    """
    if not isinstance(s, basestring) :
      raise TypeError("type mismatch `s` parameter, expecting string")
    elif not isinstance(size, int) :
      raise TypeError("type mismatch `size` parameter, expecting integer")
    
    if 0 == size :
      raise ValueError("`size` parameter can't be equal to zero")
    
    strlen = len(s)
    if 0 < size :
      return [ s[i:i+size]        for i in xrange(0, strlen,  size) ]
    else :
      # [::-1] => list is returned "reversed"
      return [ s[size-i:strlen-i] for i in xrange(0, strlen, -size) ][::-1]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def spliceString(s, start=None, end=None, value='') :
    """\
      string spliceString(string s, int start, int end, string value)
      
      Returns a copy of given 's' string, where the substring s[start:end]
      has been replaced by given replacement 'value'.
      The 'start' and 'end' arguments are interpreted as in slice notation,
      they are also optionnal parameters. If not provided, or set to 'None',
      'start' is replaced with 0; If not provided, or set to 'None', 'end'
      is replaced with "len(s)" (to match slice notation habits).
      
      Mimics the perl's `splice` function.
      
      param s                     subject string
      param start                 offset to start replacement from,
                                    defaults to None
      param end                   offset to stop replacement at
                                    defaults to None
      param value                 replacement string,
                                    defaults to ''
      return                      spliced string
    """
    if not isinstance(s, basestring) :
      raise TypeError("type mismatch `s` parameter, expecting string")
    elif not ((start is None) or isinstance(start, int)) :
      raise TypeError("type mismatch `start` parameter, expecting integer")
    elif not ((end is None) or isinstance(end, int)) :
      raise TypeError("type mismatch `end` parameter, expecting integer")
    elif not isinstance(value, basestring) :
      raise TypeError("type mismatch `value` parameter, expecting string")

    if start is None :
      start = 0
    if end is None :
      end = len(s)
    
    return value.join( (s[:start], s[end:]) )


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def truncateString(s, width=30, ellipsis='...', breakAt='') :
    """\
      string truncateString(string s, int width, string ellipsis, string breakAt)
      
      Truncates string to comply to the given 'width', characters are
      removed from the end of the string; the 'ellipsis' string is used
      to replace the missing part.
      If a string is provided for the 'breakAt' parameter, then it will
      be used as the "prefered" delimiter (see example).
      
      e.g.
        dummy = 'c:\\python23\\lib\\site-packages\\py_netasq\\building\\core\\test'
        truncateString(dummy, 30, '...')
        => 'c:\\python23\\lib\\site-packag...'
        
        truncateString(dummy, 30, '...', '\\')
        => 'c:\\python23\\lib\\...'
      
      param s                     string to truncate
      param width                 string's max length,
                                    defaults to 30
      param ellipsis              replacement string
                                    defaults to '...'
      param breakAt               prefered delimiter string,
                                    defaults to ''
      return                      truncated string
    """
    if not isinstance(s, basestring) :
      raise TypeError("type mismatch `s` parameter, expecting string")
    elif not isinstance(width, int) :
      raise TypeError("type mismatch `width` parameter, expecting integer")
    elif not isinstance(ellipsis, basestring) :
      raise TypeError("type mismatch `ellipsis` parameter, expecting string")
    elif not isinstance(breakAt, basestring) :
      raise TypeError("type mismatch `breakAt` parameter, expecting string")
    
    if not (len(s) > width) :
      # string length is below (or is equal to) the given width
      result = s
    elif len(ellipsis) == width :
      # if width does match the ellipsis length...
      result = ellipsis
    elif 2 > (width - len(ellipsis)):
      # not enough space left to insert ellipsis string
      result = ''
    else :
      length = width - len(ellipsis)
      # since 2 <= (width - len(ellipsis)), length cannot be 0
      assert 2 <= length
      
      if asqTypes.isNonEmptyInstance(breakAt, basestring) :
        offset = s.rfind(breakAt, 0, length)
        if -1 != offset : length = 1 + offset
      
      result = s[:length] + ellipsis
    
    return result


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def shortenString(s, width=30, ellipsis='...', breakAt='') :
    """\
      string shortenString(string s, int width, string ellipsis, string breakAt)
      
      Shortens the string to comply to the given 'width', characters are
      removed from the middle of the string; the 'ellipsis' string is used
      to replace the missing chunk.
      If a string is provided for the 'breakAt' parameter, then it will
      be used as the "prefered" delimiter (see example).
      
      e.g.
        dummy = 'c:\\python23\\lib\\site-packages\\py_netasq\\building\\core\\test'
        shortenString(dummy, 30, '...')
        => 'c:\\python23\\l...ding\\core\\test'
        
        shortenString(dummy, 30, '...', '\\')
        => 'c:\\python23\\...\\core\\test'
      
      Based upon a code chunk found in the 'repr' module.
      
      param s                     string to shorten
      param width                 string's max length,
                                    defaults to 30
      param ellipsis              replacement string
                                    defaults to '...'
      param breakAt               prefered delimiter string,
                                    defaults to ''
      return                      shortened string
    """
    if not isinstance(s, basestring) :
      raise TypeError("type mismatch `s` parameter, expecting string")
    elif not isinstance(width, int) :
      raise TypeError("type mismatch `width` parameter, expecting integer")
    elif not isinstance(ellipsis, basestring) :
      raise TypeError("type mismatch `ellipsis` parameter, expecting string")
    elif not isinstance(breakAt, basestring) :
      raise TypeError("type mismatch `breakAt` parameter, expecting string")
    
    if not (len(s) > width) :
      # string length is below (or is equal to) the given width
      result = s
    elif len(ellipsis) == width :
      # if width does match the ellipsis length...
      result = ellipsis
    elif 2 > (width - len(ellipsis)):
      # not enough space left to insert ellipsis string
      result = ''
    else :
      length = width - len(ellipsis)
      # since 2 <= (width - len(ellipsis)), length cannot be 0
      assert 2 <= length
      # Note : x // y is the "floor division"
      chunk1 = max(1, length // 2)
      chunk2 = len(s) - max(1, length - chunk1)
      
      if asqTypes.isNonEmptyInstance(breakAt, basestring) :
        offset = s.rfind(breakAt, 0, chunk1)
        if -1 != offset : chunk1 = 1 + offset
        offset = s.find(breakAt, chunk2)
        if -1 != offset : chunk2 = offset
      
      result = ellipsis.join( (s[:chunk1], s[chunk2:]) )
    
    return result


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def wrapString(s, width=70, insert='') :
    """\
      string wrapString(string s, int width, string insert)
      
      Wraps given string, so every line is at most 'width' characters long.
      Returns a single string, with newlines characters.
      Optionnal 'insert' parameter can be used to prepend a margin to each
      line FOLLOWING the FIRST one. Counts towards the length of each line
      except the first.
      If 'width' is lesser than 1 then 's' is simply returned.
      
      Uses the regex unit 're' to split the string.
      
      param s                     string to wrap
      param width                 line max length, defaults to 70
      param insert                string to insert between each lines,
                                    defaults to ''
      return                      wrapped string
    """
    if not isinstance(s, basestring) :
      raise TypeError("type mismatch `s` parameter, expecting string")
    elif not isinstance(width, int) :
      raise TypeError("type mismatch `width` parameter, expecting integer")
    elif not isinstance(insert, basestring) :
      raise TypeError("type mismatch `insert` parameter, expecting string")
    
    if 0 < width < len(insert) :
      raise ValueError("given `insert` string does not fit in given `width`")
    
    if 1 > width :
      return s
    
    import re
    
    result = list()
    length = 0
    buffer = len(insert)
    # even indexes : chunks, odd indexes : separators
    chunks = re.split('(\s+)', s)
    
    for index in xrange(0, len(chunks)) :
      strlen = len(chunks[index])
      if (length + strlen) < width :
        result.append(chunks[index])
        length += strlen
      elif 0 != (1 & index) :
        # odd index => separator; replace it with a new line
        result.append('\n')
        result.append(insert)
        length = buffer
      else :
        result.append('\n')
        result.append(insert)
        result.append(chunks[index])
        length = buffer + strlen
    
    del re
    return ''.join(result)


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def quoteString(s, quote, escape='') :
    """\
      string quoteString(string s, string quote, string escape)
      
      Return string 's' surrounded with 'quote' string; if occurence of
      'quote' string are found in string, then they will be replaced by
      the string 'escape' + 'quote'.
      e.g.
        quoteString('hello "world" !', '"', '\\')
        => '"hello \"world\""'
        quoteString('hello "world" !', '"', '"')
        => '"hello ""world"" !"'
      
      param s                     string to surround
      param quote                 quoting sequence
      param escape                escape sequence, defaults to ''
      return                      string surrounded with quotes
    """
    if not isinstance(s, basestring) :
      raise TypeError("type mismatch `s` parameter, expecting string")
    elif not isinstance(quote, basestring) :
      raise TypeError("type mismatch `quote` parameter, expecting string")
    elif not isinstance(escape, basestring) :
      raise TypeError("type mismatch `escape` parameter, expecting string")
    
    buffer = s.replace(quote, escape + quote)
    
    return quote + buffer + quote


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def unQuoteString(s, quote, escape='') :
    """\
      string unQuoteString(string s, string quote, string escape)
      
      Remove start and end 'quote' string from 's', if balanced; escaped
      quotes in string are replaced by a simple quote.
      e.g.
        unQuoteString('"hello \"world\""', '"', '\\')
        => hello "world"
      
      param s                     string to un-quote
      param quote                 quoting sequence
      param escape                escape sequence, defaults to ''
      return                      un-quoted string
    """
    if not isinstance(s, basestring) :
      raise TypeError("type mismatch `s` parameter, expecting string")
    elif not isinstance(quote, basestring) :
      raise TypeError("type mismatch `quote` parameter, expecting string")
    elif not isinstance(escape, basestring) :
      raise TypeError("type mismatch `escape` parameter, expecting string")
    
    result = s
    buffer = len(quote)
    
    if (2 * buffer) > len(s) :
      pass # result = s
    elif s[:buffer] != s[-buffer:] :
      pass # result = s
    elif quote != s[:buffer] :
      pass # result = s
    else :
      result = s[buffer:-buffer].replace(escape + quote, quote)
    
    return result


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def formatRndString(format, rndmin=0, rndmax=sys.maxint) :
    """\
      string formatRndString(string format, int rndmin, int rndmax)
      
      Return a string where the first numeric formatting flag ('%u', '%d',
      ...) found was replaced by a random value between 'rndmin' and
      'rndmax' (rndmin <= value <= rndmax).
      e.g.
        formatRndString('str%04u')           =>  'str527597124'
        formatRndString('str%04u', 0, 9999)  =>  'str0258'
        
      param format                string format
      param rndmin                random range min value,
                                    defaults to 0
      param rndmax                random range max value,
                                    defaults to sys.maxint
      return                      formatted string
    """
    if not isinstance(format, basestring) :
      raise TypeError("type mismatch `format` parameter, expecting string")
    elif not isinstance(rndmin, int) :
      raise TypeError("type mismatch `rndmin` parameter, expecting integer")
    elif not isinstance(rndmax, int) :
      raise TypeError("type mismatch `rndmax` parameter, expecting integer")
    
    buffer = random.randint(rndmin, rndmax)
    
    return format % (buffer, )


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def stringToBool(s, falseVals=FALSE_VALUES) :
    """\
      bool stringToBool(string s, tuple falseVals)
      
      DEPRECATED - USE asqTypes.asBoolean() instead !!
      Convert given string to boolean value: python built-in type conversion
      only considers empty string as False values, any other string will be
      converted to True 
      e.g.
        bool('None')   => True
        bool('False')  => True
      'stringToBool' is intended to fix that nasty behavior, if given string
      is found in 'falseVals' (case insensitive search), then False is
      returned.
      e.g. FALSE_VALUES = ('', 'None', 'False')
           stringToBool('None', FALSE_VALUES)   => False
           stringToBool('False', FALSE_VALUES)  => False
       
      param s                     string to convert
      param falseVals             False strings, defaults
                                    to asqString.FALSE_VALUES
      return                      boolean value
    """
    if not isinstance(s, basestring) :
      raise TypeError("type mismatch `s` parameter, expecting string")
    elif not isinstance(falseVals, (list, tuple)) :
      raise TypeError("type mismatch `falseVals` parameter, expecting sequence")
    
    buffer = [x.lower() for x in falseVals]
    
    return s.strip().lower() not in buffer


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -