#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
(Little) Less common, but still useful, sequence manipulations.

Require
  Python        2.2

Functions
  void          stuffSequence(list aSeq, int length, var default)
  list          intersection(var seqs, ...)
  list          difference(var seqs, ...)
  list          unique(list aSeq, bool ordered)
  list          sort(callable aFunc, list aSeq)
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.12 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

#~ import  nothing

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def stuffSequence(aSeq, length, default=None) :
    """\
      void stuffSequence(list aSeq, int length, var default)
      
      Pads given "aSeq" to at least "length" with "default" value. Given
      sequence will be modified 'in-place'. If sequence's length is already
      bigger or equal to "length", then nothing will happen.
      
      param aSeq                  sequence to consider
      param length                wanted minimal length
      param default               value to use for padding.
                                    Defaults to None
    """
    aSeq.extend( [default] * max(0, length - len(aSeq)) )


def intersection(*seqs) :
    """\
      list intersection(var seqs, ...)
      
      Returns the intersection of the given sequences parameters; the first
      argument is used as a "base" for items' order. This function can be
      seen as a binary "and" operation.
      e.g.
        a = ('0', '1', '3')
        b = ['5', '2', '3', '0']
        c = ['5', '4', '3', '2', '1', '0']
        intersection(a, b, c) => ['0', '3']
        
      param seqs, ...             list of sequence type
      return                      intersection of given lists
    """
    def __intersect(a, b) :
        buffer = dict()
        result = list()
        for item in b :
          buffer[item] = None
        for item in a :
          if buffer.has_key(item) :
            result.append(item)
        return result
   
    if 0 == len(seqs) :
      raise TypeError("intersection() takes at least 1 argument (0 given)")
      
    result = reduce(__intersect, seqs)
    if isinstance(result, list) :
      return result
    else :
      return list(result)


def difference(*seqs) :
    """\
      list difference(var seqs, ...)
      
      Returns the difference of the first argument with all the others.
      This function can be seen as a binary "nand" operation.
      e.g.
        a = ('0', '1', '3', '7')
        b = ['5', '2', '3', '0']
        c = ['5', '4', '3', '2', '0']
        difference(a, b, c) => ['1', '7']
      items '1' and '7' from tuple 'a' are not found in other lists.
       
      param seqs, ...             list of sequence type
      return                      difference of given lists
    """
    def __diff(a, b) :
        buffer = dict()
        result = list()
        for item in b :
          buffer[item] = None
        for item in a :
          if not buffer.has_key(item) :
            result.append(item)
        return result
    
    if 0 == len(seqs) :
      raise TypeError("difference() takes at least 1 argument (0 given)")
      
    result = reduce(__diff, seqs)
    if isinstance(result, list) :
      return result
    else :
      return list(result)


def unique(aSeq, ordered=True):
    """\
      list unique(list aSeq, bool ordered)
      
      Return a list of the elements in given sequence 'aSeq', but without
      duplicates. If 'ordered' is set to True, then sequence order will be
      kept (this may be slower).
      
      e.g.
        unique( [1, 2, 1, 3, 1, 2, 3] )
        => [1, 2, 3]
        unique( "abacabc" )
        => ['a', 'b', 'c']
        mySeq = ([1, 2], (5, 6), [2, 3], [1, 2], (1,3))
        # mySeq items are not hashable
        unique(mySeq, True)
        => [[1, 2], (5, 6), [2, 3], (1, 3)]
        unique(mySeq, False)
        => [[1, 2], [2, 3], (1, 3), (5, 6)]
      
      Based upon an activestate's cookbook recipe by Tim Peters
      "Remove duplicates from a sequence"
      > http://aspn.activestate.com/ASPN/Cookbook/Python/Recipe/52560
     
      param aSeq                  sequence to consider
      param ordered               keep sequence order, defaults to True
      return                      sequence with no duplicate items
    """
    if not aSeq :
      return aSeq
    
    try :
      result = dict()
      return [result.setdefault(x, x) for x in aSeq if x not in result]
    except TypeError:
      # all sequence item may be not hashable
      del result
   
    if not ordered :
      try :
        result = list(aSeq)
        result.sort()
      except TypeError :
        del result
      else :
        offset = 0
        buffer = result[offset]
        for i in xrange(1, len(result)) :
          if buffer != result[i] :
            offset = 1 + offset
            buffer = result[offset] = result[i]
        return result[:1 + offset]
   
    result = list()
    for item in aSeq :
      if not item in result :
        result.append(item)
    return result


def sort(aFunc, aSeq) :
    """\
      list sort(callable aFunc, list aSeq)
      
      Returns the sorted aSeq sequence; since that function is a rather
      simple wrapper around the "sort()" method, given sequence is
      modified "in place" AND returned.
      
      param aFunc                 filtering function
      param aSeq                  sequence to order
      return                      ordered sequence
    """
    aSeq.sort(aFunc)
    return aSeq


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -