#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
Custom Repr class defintion.
* aRepr : instance of objectRepr, it is used to provide the
 representation() function described below.
* representation() : shortcut to the aRepr.repr() method. It returns
 a string similar to that returned by the built-in function of the same
 name, but with limits on most sizes.

Require
  Python        2.2

Objects
  objectRepr    aRepr

Functions
  void          representation(var x)

Classes
  class         objectRepr(void)
                  extends py_netasq.commonlib.asqLogging::objectLog
                  extends repr.Repr
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.3 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  repr

from    py_netasq.commonlib     import asqLogging
from    py_netasq.commonlib     import asqString

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class objectRepr(repr.Repr, asqLogging.objectLog) :
    """\
      Class which provides formatting services for value representation; the
      logging mechanism is used to output messages. Size limits for different
      object types are added to avoid the generation of representations which
      are excessively long.
      
    """
  
    def __init__(self):
        """\
          constructor objectRepr(void)
          
          Creates a new objectRepr instance.
        """
        repr.Repr.__init__(self)
        asqLogging.objectLog.__init__(self)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  
    def _iterList(self, aList, level, length, maxlen) :
        """\
          void _iterList(list aList, int level, int length, int maxlen)
          
          Displays the representation of each sequence's items. The loop
          limit is set by 'maxlen', if the list sequence 'length' appends
          to be greater than maxlen, then an ellipsis is added.
          
          param aList                 sequence to represent
          param level                 recursion level
          param length                list length
          param maxlen                loop limit
        """
        for i in xrange(min(length, maxlen)) :
          self.repr1(aList[i], level)
          self._flush()
        if maxlen < length :
          self._print('...', True)


    def _iterDict(self, aDict, level, length, maxlen) :
        """\
          void _iterDict(dict aDict, int level, int length, int maxlen)
          
          Displays the representation of each mapping's items. The loop
          limit is set by 'maxlen', if the mapping length 'length' appends
          to be greater than maxlen, then an ellipsis is added.
          
          param aDict                 mapping to represent
          param level                 recursion level
          param length                list length
          param maxlen                loop limit
        """
        keyLst = aDict.keys()
        keyLst.sort()
        
        for i in xrange(min(length, maxlen)) :
          key = keyLst[i]
          self.repr1(key, level)
          self._print(': ')
          self.repr1(aDict[key], level)
          self._flush()
        if maxlen < length :
          self._print('...', True)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def repr(self, x):
        """\
          void repr(var x)
          
          The equivalent to the built-in repr() that uses the formatting
          imposed by the instance.
          
          param x                     value to represent
        """
        self.repr1(x, self.maxlevel)
        self._flush()


    def repr1(self, x, level):
        """\
          void repr1(var x, int level)
          
          Recursive implementation used by repr(). This uses the type of x
          to determine which formatting method to call, passing it x and
          level.
          The type-specific methods should call repr1() to perform recursive
          formatting, with level - 1 for the value of level in the recursive
          call.
          
          param x                     value to represent
          param level                 recursion level
        """
        buffer = 'repr_'+ type(x).__name__
        buffer = '_'.join(buffer.split())
        
        try :
          getattr(self, buffer)(x, level)
        except AttributeError, e :
          buffer = repr.Repr.repr1(self, x, level)
          self._print(buffer)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def repr_list(self, x, level):
        """\
          void repr_list(list x, int level)
          
          Formatting method for list type.
          
          param x                     value to represent
          param level                 recursion level
        """
        length = len(x)
        
        if 0 == length :
          self._print('[]', True)
        elif 0 >= level :
          self._print('[...]', True)
        else :
          self._print('[', True)
          self._incDepth()
          self._iterList(x, level-1, length, self.maxlist)
          self._decDepth()
          self._print(']', True)


    def repr_tuple(self, x, level):
        """\
          void repr_tuple(tuple x, int level)
          
          Formatting method for tuple type.
          
          param x                     value to represent
          param level                 recursion level
        """
        length = len(x)
        
        if 0 == length :
          self._print('()', True)
        elif 0 >= level :
          self._print('(...)', True)
        else :
          self._print('(', True)
          self._incDepth()
          self._iterList(x, level-1, length, self.maxtuple)
          self._decDepth()
          self._print(')', True)


    def repr_dict(self, x, level):
        """\
          void repr_dict(dict x, int level)
          
          Formatting method for dict type.
          
          param x                     value to represent
          param level                 recursion level
        """
        length = len(x)
        
        if 0 == length :
          self._print('{}', True)
        elif 0 >= level :
          self._print('{...}', True)
        else :
          self._print('{', True)
          self._incDepth()
          self._iterDict(x, level-1, length, self.maxdict)
          self._decDepth()
          self._print('}', True)


    def repr_array(self, x, level):
        """\
          void repr_array(array x, int level)
          
          Formatting method for array type.
          
          param x                     value to represent
          param level                 recursion level
        """
        length = len(x)
        header = "array('%s', " % (x.typecode,)
        
        self._print(header)
        if 0 == length :
          self._print('[])', True)
        elif 0 >= level :
          self._print('[...])', True)
        else :
          self._print('[', True)
          self._incDepth()
          self._iterList(x, level-1, length, self.maxarray)
          self._decDepth()
          self._print('])', True)


    def repr_str(self, x, level):
        """\
          void repr_str(str x, int level)
          
          Formatting method for string type.
          
          param x                     value to represent
          param level                 recursion level
        """
        buffer = repr.Repr.repr_str(self, x, level)
        self._print(buffer)


    def repr_long(self, x, level):
        """\
          void repr_long(long x, int level)
          
          Formatting method for long type.
          
          param x                     value to represent
          param level                 recursion level
        """
        buffer = repr.Repr.repr_long(self, x, level)
        self._print(buffer)


    def repr_instance(self, x, level):
        """\
          void repr_instance(object x, int level)
          
          Formatting method for object instance type.
          
          param x                     value to represent
          param level                 recursion level
        """
        #~ buffer = repr.Repr.repr_instance(self, x, level)
        # remove instance offset
        buffer = '<%s>' % (x.__class__.__name__,)
        buffer = asqString.shortenString(buffer, self.maxstring, '...')
        self._print(buffer)


  # /class objectRepr - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
  print __file__
  print __doc__
else :
  aRepr = objectRepr()
  representation = aRepr.repr

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -