#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
(Little) Less common, but still useful, pathname and filesystem
manipulations (i.e. not available in os.path).

Require
  Python        2.2

Functions
  string        normalize(string path)
  string        absolute(string path, string base)
  string        directory(string path)
  string        makeTmpFile(string suffix, string prefix, string dir)
  void          makeDir(string path, int mode)
  void          changeDir(string path, int mode)
  void          emptyDirs(string path, bool force)
  void          emptyDirectory(string path, bool force)
  void          delete(string path, bool force)
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.16 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  os
import  shutil
import  tempfile

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def normalize(path) :
    """\
      string normalize(string path)
      
      Normalize given 'path': apply "os.path.normpath()" and
      "os.path.normcase()" to path.
      
      - normcase(path) :
       On Unix, this returns the path unchanged; on case-insensitive
       filesystems, it converts the path to lowercase.
       On Windows, it also converts forward slashes to backward slashes
      
      - normpath(path) :
       Collapses redundant separators and up-level references, e.g. A//B,
       A/./B and A/foo/../B all become A/B. It does not normalize the case.
       On Windows, it converts forward slashes to backward slashes
      
      param path                  path to normalize
      return                      normalized path
    """
    result = os.path.normpath(path)
    result = os.path.normcase(result)
    return result


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def absolute(path, base=None) :
    """\
      string absolute(string path, string base)
      
      Return a normalized absolutized version of the pathname 'path', using
      'base'. If given path is already absolute, then it is simply returned,
      else if None is provided as base, then the function behaves just as
      os.path.abspath(path)
      e.g. with current dir : c:\dummy\path
        absolute(r'aFile.ext')             => c:\dummy\path\aFile.ext
        absolute(r'.\temp', r'c:\windows') => c:\windows\temp
        absolute(r'c:\temp', r'c:\wndows') => c:\temp
      
      param path                  pathname
      param base                  pathname "anchor", defaults to None
      return                      normalized absolutized pathname
    """
    if os.path.isabs(path) :
      return path
    elif base is None :
      return os.path.abspath(path)
    else :
      assert os.path.isabs(base)                                        ,\
              "`base` is not an absolute path %s" % (base,)
      return normalize( os.path.join(base, path) )


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def directory(path) :
    """\
      string directory(string path)
      
      Return the absolute directory name of given relative 'path'; if
      given 'path' is already a directory name, then its absolute version
      is returned.
      This function was written in order to retrieve the directory name of
      the current processed __file__
      e.g. with current dir : c:\dummy\path
        directory(__file__)             => c:\dummy\path
      
      param path                  a relative path
      return                      normalized absolutized dirname
    """
    result = absolute(path, None)
    if not os.path.isdir(result) :
      result = os.path.dirname(result)
    return normalize(result)


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def makeTmpFile(suffix='', prefix='', dir=None) :
    """\
      string makeTmpFile(string suffix, string prefix, string dir)
      
      Creates a temporary file in the most secure manner possible.
      Wraper around 'tempfile.mkstemp()', the behavior is just the same, but
      only the absolute pathname of the temp file is returned. One should
      still delete the temporary file when done with it.
      
      If suffix is specified, the file name will end with that suffix,
      otherwise there will be no suffix. makeTmpFile() does not put a dot
      between the file name and the suffix; if you need one, put it at the
      beginning of suffix. 
      
      If prefix is specified, the file name will begin with that prefix;
      otherwise, a default prefix is used. 
      
      If dir is specified, the file will be created in that directory;
      otherwise, a default directory is used.
      
      param  suffix               temp file suffix; defaults to ''
      param  prefix               temp file prefix; defaults to ''
      param  dir                  parent dir of temp file; defaults to None
      return                      absolute pathname of temp file
    """
    result = tempfile.mkstemp(suffix, prefix, dir, text=True)
    # mkstemp() returns a tuple containing an OS-level handle to
    #  an open file (as would be returned by os.open()) and the
    #  absolute pathname of that file, in that order.
    #
    # Note : on Microsoft windows it seems that the generated path
    #  a "mangled" 8.3 dos path. sometimes the path is not resolved
    #  to its full length name, causing strangely named directories
    #  to appear on the host system...
    os.close(result[0])
    return result[1]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def makeDir(path, mode=0777) :
    """\
      void makeDir(string path, int mode)
      
      Create a directory named "path" with numeric mode "mode". Handles
      complex path creation, e.g. makeDir('/new/new/new'). The default
      mode is 0777 (octal). If directory already exists, then nothings
      happen.
          
      param path                  path to directory
      param mode                  directory mode, defaults to 0777
    """
    path = absolute(path)
    if os.path.exists(path) :
      os.chmod(path, mode)
    else :
      chunks = path.split( os.sep )
      for i in xrange(len(chunks)) :
        buffer = os.sep.join(chunks[0:i+1])
        if not os.path.exists(buffer + os.sep) :
          os.mkdir(buffer, mode)


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def changeDir(path, mode=0777) :
    """\
      void changeDir(string path, int mode)
    
      Change the current working directory to path. If "path" does not
      exist, then it will be created with the given numeric mode "mode".
      The default mode is 0777 (octal).
      
      param path                  path to directory
      param mode                  directory mode, defaults to 0777
    """
    if not os.path.isdir(path) :
      makeDir(path, mode)
    os.chdir(path)


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def emptyDirs(path, force=False) :
    """\
      void emptyDirs(string path, bool force)
      
      Empty directories recursively. All sub directories will REMAIN, but
      EMPTY. If 'force' is set to True, then file permission will be set
      to 0777 before removal. An exception may be raised if a file cannot
      be removed (or if the permissions can't be modified).
      
      One may use 'shutil.rmtree' afterward to delete the directory tree.
      
      param path                  path to target directory
      param force                 force file deletion, default to False
    """
    for item in os.listdir(path) :
      item = os.path.join(path, item)
      if os.path.isdir(item) :
        emptyDirs(item, force)
      elif os.path.isfile(item) :
        if force : os.chmod(item, 0777)
        os.remove(item)


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def emptyDirectory(path, force=False) :
    """\
      void emptyDirectory(string path, bool force)
      
      Empty A directory from ALL its content (files and sub-dirs).
      If 'force' is set to True, then file permission will be set to 0777
      before removal. An exception may be raised if a file cannot be
      removed (or if the permissions can't be modified).
      
      Do not mistake this function with "emptyDirs()".
      
      param path                  path to target directory
      param force                 force file deletion, default to False
    """
    if not os.path.exists(path) :
      raise OSError("path `%s` does not exist." % (path, ))
    
    emptyDirs(path, force)
    for item in os.listdir(path) :
      item = os.path.join(path, item)
      shutil.rmtree(item)


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def delete(path, force=False) :
    """\
      void delete(string path, bool force)
      
      Wipe out given 'path', target item can either be a file or a
      directory. If 'path' refers to a directory, "emptyDirs()" will be
      called upon it, and the whole sub-tree will be trashed. If 'force'
      is set to True, then file permission will be set to 0777 before
      removal. 
      If given 'path' does not refer to an existing path then nothing
      happens. An exception may be raised if item cannot be removed
      (or if the permissions can't be modified).
      
      param path                  target item to delete
      param force                 force item deletion, default to False
    """
    if not os.path.exists(path) :
      raise OSError("path `%s` does not exist." % (path, ))
    
    if os.path.isdir(path) :
      emptyDirs(path, force)
      shutil.rmtree(path)
    elif os.path.isfile(path) :
      if force : os.chmod(path, 0777)
      os.remove(path)
    else :
      raise OSError("path `%s` is not a file, nor a directory." % (path, ))


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -