#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
(Little) Less common, but still useful, mapping manipulations.

Require
  Python        2.3

Functions
  dict          invertion(dict aDict, bool extend)
  dict          intersection(var maps, ...)
  dict          difference(var maps, ...)
  list          intersectKeys(var maps, ...)
  void          filterKeys(dict aDict, list keys [, bool exclude])
  dict          filterDict(callable aFunc, dict aDict)
  var           extractItem(var aMap, var key, var default)
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.17 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

from    py_netasq.commonlib             import asqSequence

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def invertion(aDict, extend=False) :
    """\
      dict invertion(dict aDict, bool extend)
      
      Returns a new dictionary with swapped keys and values. Inverting a
      dictionary allows you to reverse the mapping and use a value to look
      up a key.
      NOTE : since dict cannot be used as dictionary keys, each dict value
       found in 'aDict' will simply be skipped.
      
      If 'extend' is set to False, then for common values only the "last"
      key will be preserved once swapped, and list values found in 'aDict'
      will be skipped.
      If 'extend' is set to True, then all keys with common values will be
      added to a list, and list values will be converted too: each list's
      item will be considered as a dict value, and thus used as a key for
      resulting dictionary.
      e.g.
        myDict = { 'a': 1, 'b': 2, 'c':2, 'd': [5, 6], 'e': {1:0} }
        invertion(myDict, extend=False)
        => { 1: 'a', 2: 'b' }
        invertion(myDict, extend=True)
        => { 1: 'a', 2: ['c', 'b'], 5: 'd', 6: 'd' }
        invertion( invertion(myDict, True) , True)
        => { 'a': 1, 'b': 2, 'c':2, 'd': [5, 6] }
      
      Based upon an activestate's cookbook recipe by Joel Lawhead
      "Invert a dictionary (one-liner)"
      > http://aspn.activestate.com/ASPN/Cookbook/Python/Recipe/252143
        
      param aDict                 dictionary to invert
      param extend                triggers the extended invertion mode,
                                    defaults to False
      return                      a new dictionary
    """
    def __simpleInvert(aDict) :
        result = dict()
        for key, val in aDict.iteritems() :
          try :
            result[val] = key
          except TypeError :
            # TypeError: list and dict objects are unhashable
            pass
        return result
    
    def __complxInvert(aDict) :
        result = dict()
        for key, val in aDict.iteritems() :
          try :
            if not isinstance(val, list) :
              result.setdefault(val, []).append(key)
            else :
              for item in val :
                result.setdefault(item, []).append(key)
          except TypeError :
            # TypeError: dict objects are unhashable
            pass
        # unpack single values
        for key, val in result.iteritems() :
          if 2 > len(val) : result[key] = result[key][0]
        return result
   
    if not extend :
      return __simpleInvert(aDict)
    else :
      return __complxInvert(aDict)


def intersection(*maps) :
    """\
      dict intersection(var maps, ...)
      
      Returns the intersection (key + values) of the given mappings; the
      the first argument is used as a "base". This function can be seen
      as a binary "and" operation. A simple equality test is done among
      the mapping values.
      e.g.
        X = { 'a': 1, 'b': 2, 'c': 2, 'd': [5, 6], 'e': {1:0} }
        Y = { 'a': '1', 'b' : 2, 'd': [5, 6], 'e': {1:0} }
        Z = { 'e': {1:0} }
        
        intersection(X, Y)
        => { 'b': 2, 'd': [5, 6], 'e': {1:0} }
        intersection(X, Y, Z)
        => { 'e': {1:0} }
        
      param maps, ...             list of mapping type
      return                      intersection of given mappings
    """
    def __intersect(a, b) :
        result = dict()
        for key in intersectKeys(a, b) :
          if a[key] == b[key] : result[key] = a[key]
        return result
   
    if 0 == len(maps) :
      raise TypeError("intersection() takes at least 1 argument (0 given)")
    return reduce(__intersect, maps)


def difference(*maps) :
    """\
      dict difference(var maps, ...)
      
      Returns the difference of the first argument with all the others.
      This function can be seen as a binary "nand" operation. A simple
      equality test is done among the mapping values.
      e.g.
        X = { 'a': 1, 'b': 2, 'c': 2, 'd': [5, 6], 'e': {1:0} }
        Y = { 'a': '1', 'b' : 2, 'd': [5, 6], 'e': {1:0} }
        Z = { 'e': {1:0} }
        difference(X, Y)
        => { 'a': 1, 'c': 2 }
        difference(X, Y, Z)
        => { 'a': 1, 'c': 2 }
       
      param maps, ...             list of mapping type
      return                      intersection of given mappings
    """
    def __diff(a, b) :
        result = a.copy()
        for key in intersectKeys(a, b) :
          if a[key] == b[key] : result.pop(key)
        return result

    if 0 == len(maps) :
      raise TypeError("intersection() takes at least 1 argument (0 given)")
    return reduce(__diff, maps)


def intersectKeys(*maps) :
    """\
      list intersectKeys(var maps, ...)
      
      Returns the key list shared by all given mappings parameters.
      e.g.
        X = { 'php': '<?php', 'python': 'def' }
        Y = { 'python': '#!/usr/bin/env', 'perl': '$' }
        Z = { 'java': 'public static void' }
        
        intersectKeys(X, Y)
        => ['python']
        intersectKeys(X, Y, Z)
        => []
        
      param maps, ...             list of mapping type
      return                      intersection of given mappings' keys
    """
    if 0 == len(maps) :
      raise TypeError("intersectKeys() takes at least 1 argument (0 given)")
    return asqSequence.intersection(*[x.keys() for x in maps])


def filterKeys(aDict, keys, exclude=False) :
    """\
      void filterKeys(dict aDict, list keys, bool exclude)
      
      Filter 'aDict' keys in place, using 'keys' list. If 'exclude' is set
      to True, then keys found in both 'aDict' keys and 'keys' list are
      removed, else 'aDict' keys missing from 'keys' list are removed.
      e.g.
        X = { 'a' : 0, 'b' : 2, 'z' : 8 }
        filterKeys(X, ('a', 'b'), False)
        => X = { 'a' : 0, 'b' : 2 }
        
        X = { 'a' : 0, 'b' : 2, 'z' : 8 }
        filterKeys(X, ('a', 'b'), True)
        => X = { 'z' : 8 }
      
      param aDict                 dictionnary to filter
      param keys                  key list
      param exclude               exclude or keep 'keys' from 'aDict'
                                    Defaults to False (keep)
    """
    if exclude :
      for key in asqSequence.intersection(aDict.keys(), keys) :
        aDict.pop(key)
    else :
      for key in asqSequence.difference(aDict.keys(), keys) :
        aDict.pop(key)


def filterDict(aFunc, aDict) :
    """\
      dict filterDict(callable aFunc, dict aDict)
      
      Constructs a dict from those values of aDict for which aFunc
      function returns True. The function behaves as "filter()",
      especially if 'aFunc' is None.
      
      param aFunc                 filtering function
      param aDict                 dictionnary to filter
      return                      filtered dict
    """
    if aFunc is None :
      def __filterItem(item) : return bool(item[1])
    else :
      def __filterItem(item) : return aFunc(item[1])
    
    return dict(filter(__filterItem, aDict.items()))


def extractItem(aMap, key, default=None) :
    """\
      var extractItem(var aMap, var key, var default)
      
      Removes from map and returns value corresponding to given 'key'.
      If 'key' is not in the map, then 'default' value will be returned.
      'aMap' can either be a dictionnary or a list object.
      
      param aMap                  map to extract values from
      param key                   item key
      param default               returned if key is not found,
                                    defaults to None
      return                      value for given key
    """
    if isinstance(aMap, list) :
      try :
        buffer = [ x[0] for x in aMap ]
        return aMap.pop( buffer.index(key) )[1]
      except (ValueError, IndexError), e :
        return default
    elif isinstance(aMap, dict) :
      try :
        result = aMap[key]
        del aMap[key]
        return result
      except KeyError, e :
        return default
    else :
      return default


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -