#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
Simple functions to setup and send email. Most of the code is loosely
based on Python Library Reference examples.

Require
  Python        2.2

Functions
  void          sendMailMessage(object message, string host)

  object        MimeTextFromFile(
                  string path, string subtype, string charset, **params)
  object        MimeImageFromFile(
                  string path, string subtype, **params)
  object        MimeAudioFromFile(
                  string path, string subtype, **params)
  object        MimeBaseFromFile(
                  string path, string maintype, string subtype, **params)
  object        MimeFromFile(string path, **params)

  void          attachTextFileTo(
                  object message, string path, var filename, string subtype,
                  string charset, **params)
  void          attachImageFileTo(
                  object message, string path, var filename, string subtype,
                  **params)
  void          attachAudioFileTo(
                  object message, string path, var filename, string subtype,
                  **params)
  void          attachBaseFileTo(
                  object message, string path, var filename, string maintype,
                  string subtype, **params)
  void          attachFileTo(
                  object message, string path, string filename, **params)

  void          setupMessage(
                  object message, string sender, var recipients,
                  string subject)
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.5 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  os
import  mimetypes
import  smtplib

from    email                                   import Encoders
from    email.Message                           import Message
from    email.MIMEAudio                         import MIMEAudio
from    email.MIMEBase                          import MIMEBase
from    email.MIMEImage                         import MIMEImage
from    email.MIMEText                          import MIMEText

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

# recipients delimiter
COMMASPACE = ', '

# netasq smtp host address
SMTP_HOST  = '10.0.0.3'

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def sendMailMessage(message, host=SMTP_HOST) :
    """\
      void sendMailMessage(object message, string host)
      
      Send given email.Message object using 'host' as the SMTP server
      address.
      
      param message               message to send
      param host                  smtp host, defaults to SMTP_HOST
    """
    sender = message['From']
    target = COMMASPACE.join(message.get_all('To', None))
    
    assert isinstance(sender, basestring) and 0 < len(sender),          \
           "mail sender address is missing"
    assert isinstance(target, basestring) and 0 < len(target),          \
           "mail recipients adrress are missing"
    
    server = smtplib.SMTP()
    try :
      server.connect(host)
      server.sendmail(sender, target, message.as_string())
    finally :
      server.quit()
  

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def MimeTextFromFile(path, subtype='plain', charset='ISO-8859-1', **params) :
    """\
      object MimeTextFromFile(
              string path, string subtype, string charset, **params)
      
      Build a MIME object of major type "text"; 'subtype' is the minor type
      and defaults to "plain". 'charset' is the character set of the text
      and is passed as a parameter to the MIMENonMultipart constructor;
      it defaults to "ISO-8859-1". No guessing or encoding is performed on
      the text data
      
      'params' are passed straight through to the MIMEText constructor.
      
      param path                  path to file
      param subtype               minor MIME type, defaults to 'plain'
      param charset               character set, defaults to 'ISO-8859-1'
      param **params              complementary parameters
      return                      a MIMEText object
    """
    assert os.path.isfile(path), "file not found: %s" % (path,)
    opened = open(path, 'rb')
    try :
      return MIMEText(opened.read(), subtype, charset, **params)
    finally :
      opened.close()


def MimeImageFromFile(path, subtype=None, **params) :
    """\
      object MimeImageFromFile(string path, string subtype, **params)
       
      Build a MIME message objects of major type "image". If image data
      can be decoded by the standard Python module imghdr, then the subtype
      will be automatically included in the Content-Type: header.
      Otherwise you can explicitly specify the image subtype via the
      'subtype' parameter. If the minor type could not be guessed and
      'subtype' was not given, then TypeError is raised.
      
      'params' are passed straight through to the MIMEImage constructor.
      
      param path                  path to file
      param subtype               minor MIME type, defaults to None
      param **params              complementary parameters
      return                      a MIMEImage object
    """
    assert os.path.isfile(path), "file not found: %s" % (path,)
    opened = open(path, 'rb')
    try :
      return MIMEImage(opened.read(), subtype, encoder, **params)
    finally :
      opened.close()


def MimeAudioFromFile(path, subtype=None, **params) :
    """\
      object MimeAudioFromFile(string path, string subtype, **params)
      
      Build a MIME message objects of major type audio. If audio data can
      be decoded by the standard Python module sndhdr, then the subtype will
      be automatically included in the Content-Type: header.
      Otherwise you can explicitly specify the audio subtype via the
      'subtype' parameter. If the minor type could not be guessed and
      'subtype' was not given, then TypeError is raised. 
     
      'params' are passed straight through to the MIMEAudio constructor.
      
      param path                  path to file
      param subtype               minor MIME type, defaults to None
      param encoder               data encoder, defaults to None
      param **params              complementary parameters
      return                      a MIMEAudio object
    """
    assert os.path.isfile(path), "file not found: %s" % (path,)
    opened = open(path, 'rb')
    try :
      return MIMEAudio(opened.read(), subtype, **params)
    finally :
      opened.close()


def MimeBaseFromFile(path, maintype, subtype, **params) :
    """\
      object MimeBaseFromFile(
              string path, string maintype, string subtype, **params)
        
      Build a MIME-specific object; this function is provided primarily as
      a convenient base class for more specific MIME-aware subclasses.
      'maintype' is the Content-Type: major type (e.g. text or image), and
      'subtype' is the Content-Type: minor type (e.g. plain or gif). 'params'
      is a parameter key/value dictionary and is passed directly to
      Message.add_header(). 
      
      The MIMEBase class always adds a Content-Type: header (based on 
      'maintype', 'subtype', and 'params'), and a MIME-Version: header
      (always set to 1.0). 
      
      param path                  path to file
      param maintype              major MIME type
      param subtype               minor MIME type
      param **params              complementary parameters
      return                      a MIMEBase object
    """
    assert os.path.isfile(path), "file not found: %s" % (path,)
    opened = open(path, 'rb')
    try :
      result = MIMEBase(maintype, subtype)
      result.add_payload(opened.read())
      # Encode the payload using Base64
      Encoders.encode_base64(result)
      return result
    finally :
      opened.close()


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def MimeFromFile(path, **params) :
    """\
      object MimeFromFile(string path, **params)
      
      Tries to build an accurate MIME object; file mime type is guessed
      using file's extension. If content type is not recognised, then 
      "application/octet-stream" is assumed.
      
      param path                  path to file
      param **params              complementary parameters
      return                      a MIME object
    """
    assert os.path.isfile(path), "file not found: %s" % (path,)
    
    # Guess the Content-Type: based on the file's extension.  Encoding
    # will be ignored, although we should check for simple things like
    # gzip'd or compressed files
    ctype, encoding = mimetypes.guess_type(path)
    
    if ctype is not None and encoding is None :
      ubrtype, subtype = ctype.split('/', 1)
    else :
      # No guess could be made, or the file is encoded (compressed), so
      # use a generic bag-of-bits type.
      ubrtype = 'application'
      subtype = 'octet-stream'
      
    if   'text'  == ubrtype :
      return MimeTextFromFile(path, subtype, **params)
    elif 'image' == ubrtype :
      return MimeImageFromFile(path, subtype, **params)
    elif 'audio' == ubrtype :
      return MimeAudioFromFile(path, subtype, **params)
    else :
      return MimeBaseFromFile(path, ubrtype, subtype, **params)


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def attachTextFileTo(message, path, filename=None, subtype='plain', charset='ISO-8859-1', **params) :
    """\
      void attachTextFileTo(
            object message, string path, var filename, string subtype,
            string charset, **params)
      
      Attach a file as MIMEText object to given 'message'; 'filename' is
      the attachment file name, if no set, then it will be extracted from
      the path. Others parameters keep the same meaning from
      "MimeTextFromFile".
      
      param message               enclosing message
      param path                  path to file
      param filename              attachment name, defaults to None
      param subtype               minor MIME type, defaults to 'plain'
      param charset               character set, defaults to 'ISO-8859-1'
      param **params              complementary parameters
    """
    assert message is not None, "no valid Message object supplied"
    
    buffer = MimeTextFromFile(path, subtype, charset, **params)
    if filename is None : filename = os.path.basename(path)
    buffer.add_header('Content-Disposition', 'attachment', filename=filename)
    
    message.attach(buffer)


def attachImageFileTo(message, path, filename=None, subtype=None, **params) :
    """\
      void attachImageFileTo(
            object message, string path, var filename, string subtype,
            **params)
      
      Attach a file as MIMEImage object to given 'message'; 'filename' is
      the attachment file name, if no set, then it will be extracted from
      the path. Others parameters keep the same meaning from
      "MimeImageFromFile".
      
      param message               enclosing message
      param path                  path to file
      param filename              attachment name, defaults to None
      param subtype               minor MIME type, defaults to None
      param **params              complementary parameters
    """
    assert message is not None, "no valid Message object supplied"
    
    buffer = MimeImageFromFile(path, subtype, **params)
    if filename is None : filename = os.path.basename(path)
    buffer.add_header('Content-Disposition', 'attachment', filename=filename)
    
    message.attach(buffer)


def attachAudioFileTo(message, path, filename=None, subtype=None, **params) :
    """\
      void attachAudioFileTo(
            object message, string path, var filename, string subtype,
            **params)
      
      Attach a file as MIMEAudio object to given 'message'; 'filename' is
      the attachment file name, if no set, then it will be extracted from
      the path. Others parameters keep the same meaning from
      "MimeAudioFromFile".
      
      param message               enclosing message
      param path                  path to file
      param filename              attachment name, defaults to None
      param subtype               minor MIME type, defaults to None
      param **params              complementary parameters
    """
    assert message is not None, "no valid Message object supplied"
    
    buffer = MimeAudioFromFile(path, subtype, **params)
    if filename is None : filename = os.path.basename(path)
    buffer.add_header('Content-Disposition', 'attachment', filename=filename)
    
    message.attach(buffer)


def attachBaseFileTo(message, path, filename=None, maintype='application', subtype='octet-stream', **params) :
    """\
      void attachBaseFileTo(
            object message, string path, var filename, string maintype,
            string subtype, **params)
      
      Attach a file as MIMEBase object to given 'message'; 'filename' is
      the attachment file name, if no set, then it will be extracted from
      the path. Others parameters keep the same meaning from
      "MimeBaseFromFile".
      
      param message               enclosing message
      param path                  path to file
      param filename              attachment name, defaults to None
      param maintype              major MIME type, defaults to 'application'
      param subtype               minor MIME type, defaults to 'octet-stream'
      param **params              complementary parameters
    """
    assert message is not None, "no valid Message object supplied"
    
    buffer = MimeBaseFromFile(path, maintype, subtype, **params)
    if filename is None : filename = os.path.basename(path)
    buffer.add_header('Content-Disposition', 'attachment', filename=filename)
    
    message.attach(buffer)


def attachFileTo(message, path, filename=None, **params) :
    """\
      void attachFileTo(
            object message, string path, string filename, **params)
      
      Attach a MIME object, guessing its type from the file's extension;
      'filename' is the attachment file name, if no set, then it will be
      extracted from the path. Others parameters keep the same meaning from
      "MimeFromFile".
      
      param path                  path to file
      param filename              attachment name, defaults to None
      param **params              complementary parameters
    """
    assert message is not None, "no valid Message object supplied"
    
    buffer = MimeFromFile(path, **params)
    if filename is None : filename = os.path.basename(path)
    buffer.add_header('Content-Disposition', 'attachment', filename=filename)
    
    message.attach(buffer)


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def setupMessage(message, sender, recipients, subject='') :
    """\
      void setupMessage(
            object message, string sender, var recipients, string subject)
      
      A convenient, and safe, way to set up a message in order to send it
      throug "sendMailMessage()". The 'recipients' param can either be a
      string (for a single mail address), or a list (multiple recipients).
      
      param message               message to set up
      param sender                message sender
      param recipients            message recipients
      param subject               message subject, defaults to ''
    """
    assert message is not None, "no valid Message object supplied"
    assert isinstance(sender, basestring),                              \
           "invalid parameter: 'sender', string expected"
    if isinstance(recipients, basestring) :
      recipients = [recipients.strip()]
    elif isinstance(recipients, list) :
      recipients = filter(lambda x: isinstance(x, basestring), recipients)
      recipients = map(lambda s: s.strip(), recipients)
    else :
      assert False,                                                     \
             "invalid parameter: 'recipients', string or list expected"
     
    message['From'   ] = sender.strip()
    message['To'     ] = COMMASPACE.join(recipients)
    message['Subject'] = str(subject)


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -