#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
Defines a custom logger class to use with the standard 'logging' module.

Require
  Python        2.3

Constants
  int           TRACE

Classes
  class         objectLog(object logger, int logLvl)
                  extends object

  class         SilentHandler(int level)
                  extends logging::Handler

  class         RotatingFileHandler(
                                string filename, string mode,
                                int maxBytes, int backupCount)
                  extends logging::RotatingFileHandler

  class         customLogger(string name, int level)
                  extends logging::Logger
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.14 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  logging
import  os
import  shutil

# Somehow, the shortcut to 'logging.handlers' seems to be badly needed
#  by python's parser (I'm a bit puzzled).
from    logging                 import handlers as logHandlers


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

# Custom log level definition 
# Trace should be used to build the application function map
TRACE = 101


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class objectLog(object) :
    """\
      A custom object with a built-in logging utility. A simple buffering
      mechanism is used, messages will not be send as long as the _flush()
      method is not called.
      
      constructor       objectLog(object logger, int logLvl)
      
      property  object  logger
      property  int     logLevel
    """
    
    def __init__(self, logger=None, logLvl=logging.NOTSET) :
        """\
          constructor objectLog(object logger, int logLvl)
          
          Creates an object with inner logging utility. Messages will
          be rated using the logLvl and sent to the logger.
          
          param logger                logger object
                                        defaults to None
          param logLvl                logging level
                                        defaults to logging.NOTSET
        """
        super(objectLog, self).__init__()
        # buffered output, flush() must be explicitly called
        self.__buffer = list()
        self.__logger = logger
        self.__logLvl = logLvl


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  
    def _print(self, s, flush=False) :
        """\
          void _print(string s, bool flush)
          
          Saves given string to the inner buffer, if 'flush' is set to
          True then "_flush()" will be called. If 's' parameter is not
          a string then an assertion error may be raised.
          
          param s                     message
          param flush                 do flush buffer ?
                                        defaults to False
        """
        assert isinstance(s, basestring)
        
        self.__buffer.append(s)
        if flush : self._flush()


    def _flush(self) :
        """\
          void _flush(void)
          
          Actually sends buffered message string to the logger, if sets,
          and clears the buffer.
        """
        if not self.__buffer :
          return
        elif self.__logger is not None :
          self.__logger.log(self.__logLvl, ''.join(self.__buffer))
        self.__buffer[:] = []


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _getDepth(self) :
        """\
          int _getDepth(void)
          
          Returns the depth value corresponding to inner logLevel. If
          logger is not set, or does not support the depth mechanism,
          then -1 will be returned.
          
          return                      depth
        """
        if self.__logger is None :
          return -1
        try :
          self.__logger.getDepth(self.__logLvl)
        except AttributeError, e :
          return -1


    def _setDepth(self, value) :
        """\
          int _setDepth(int value)
          
          Sets the depth for the inner logLevel.
          
          param value                 depth value.
        """
        if self.__logger is None :
          return -1
        try :
          return self.__logger.setDepth(value, self.__logLvl)
        except AttributeError, e :
          return -1


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _incDepth(self, value=1, level=None) :
        """\
          int _incDepth(int value, int level)
          
          Increments message depth for logging 'level' by given 'value'.
          If 'level' is None, then inner logLevel value is assumed.
          The new depth value for the level is returned, if level is not
          found among the inner depths list, then -1 will be returned.
          
          param value                 depth increment. Defaults to 1
          param level                 logging level. Defaults to None
          return                      new depth value
        """
        if self.__logger is None :
          return -1
        if level is None :
          level = self.__logLvl
        try :
          return self.__logger.incDepth(value, level)
        except AttributeError, e :
          return -1


    def _decDepth(self, value=1, level=None) :
        """\
          int _decDepth(int value, int level)
          
          Decrements message depth for logging 'level' by given 'value'.
          If 'level' is None, then inner logLevel value is assumed.
          The new depth value for the level is returned, if level is not
          found among the inner depths list, then -1 will be returned.
          
          param value                 depth decrement. Defaults to 1
          param level                 logging level. Defaults to None
          return                      new depth value
        """
        if self.__logger is None :
          return -1
        if level is None :
          level = self.__logLvl
        try :
          return self.__logger.decDepth(value, level)
        except AttributeError, e :
          return -1


  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _getLogger(self) :
        """\
          object _getLogger(void)
          
          Getter - logger property
        """
        return self.__logger


    def _getLogLvl(self) :
        """\
          int _getLogLvl(void)
          
          Getter - logLevel property
        """
        return self.__logLvl


  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _setLogger(self, value) :
        """\
          void _setLogger(object value)
          
          Setter - logger property
        """
        self.__logger = value


    def _setLogLvl(self, value) :
        """\
          void _setLogLvl(int value)
          
          Setter - logLevel property
        """
        self.__logLvl = int(value)


  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    logger     = property(
                doc  = "logger used to print value",
                fget = _getLogger,
                fset = _setLogger,
                fdel = None )

    logLevel   = property(
                doc  = "logging level used to send message to logger",
                fget = _getLogLvl,
                fset = _setLogLvl,
                fdel = None )


  # /class objectLog  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -



# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class SilentHandler(object, logging.Handler) :
    """\
      Dummy logging handler, does nothing with received messages. This
      class should be used to supply a default handler for logger that
      do not need one, thus avoiding the display of the error message :
        "No handlers could be found for logger <logger name>"
      
      For further informations, please check Python-Dev mailing list
      archive: July 2004 - "Logging with no handlers configured"
      > http://mail.python.org/pipermail/python-dev/2004-July/046396.html
      
      WARNING : logging.Handler still does not derivate from "object"
        (note logging.Filterer is the "mother" class) :/
    """


    def __init__(self, level=logging.NOTSET) :
        """\
          constructor SilentHandler(int level)
          
          Initializes the instance - basically setting the formatter to
          None and the filter list to empty.
          
          param level                 handler level.
                                        Defaults to logging.NOTSET
        """
        logging.Handler.__init__(self, level)


    def handle(self, record):
        """
          bool handle(string record)
        
          Conditionally emit the specified logging record.
          
          Emission depends on filters which may have been added to the
          handler. Wrap the actual emission of the record with
          acquisition/release of the I/O thread lock. Returns whether the
          filter passed the record for emission.
          
          param record                message to handle
          return                      did the message pass the filter ?
        """
        return False


  # /class SilentHandler  - - - - - - - - - - - - - - - - - - - - - - - - - - -



# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class RotatingFileHandler(object, logHandlers.RotatingFileHandler) :
    """\
      The RotatingFileHandler class; now shipping with 'doRollover()' fix
      for the Microsoft Windows only "OSError permission denied" exception.
      
      WARNING : RotatingFileHandler still does not derivate from "object"
        (note logging.Filterer is the "mother" class) :/
    """


    def __init__(self, filename, mode="a", maxBytes=0, backupCount=0) :
        """
          constructor RotatingFileHandler(string filename, string mode, int maxBytes, int backupCount)
        
          Open the specified file and use it as the stream for logging.
          
          By default, the file grows indefinitely. You can specify
          particular values of maxBytes and backupCount to allow the file
          to rollover at a predetermined size.
          Note : If maxBytes is zero, rollover never occurs.
          
          param filename              log file name
          param mode                  how the file is to be opened
                                        Defaults to 'a'
          param maxBytes              size limit for log file
                                        Defaults to 0
          param backupCount           max count for backup files
                                        Defaults to 0
        """
        logHandlers.RotatingFileHandler.__init__(self, filename, mode, maxBytes, backupCount)


    def doRollover(self):
        """\
          void doRollover(void)
          
          Do a rollover upon log files. Fix for Microsoft Windows.
          
          Rollover occurs whenever the current log file is nearly maxBytes
          in length. If backupCount is >= 1, the system will successively
          create new files with the same pathname as the base file, but
          with extensions ".1", ".2" etc. appended to it.
          For example, with a backupCount of 5 and a base file name of
          "app.log", you would get "app.log", "app.log.1", "app.log.2", ...
          through to "app.log.5".
          The file being written to is always "app.log" - when it gets
          filled up, it is closed and renamed to "app.log.1", and if
          files "app.log.1", "app.log.2" etc. exist, then they are renamed
          to "app.log.2", "app.log.3" etc. respectively.
          
          If maxBytes is zero, rollover never occurs.
        """
        try :
          logHandlers.RotatingFileHandler.doRollover(self)
        except OSError, e :
          if 1 > self.backupCount :
            raise # nothing to do with file rotation ??
          else :
            self.stream.close()
            # assume that back up files have been correctly renamed
            backup= self.baseFilename + '.1'
            if os.path.exists(backup) : os.remove(backup)
            shutil.copy2(self.baseFilename, backup)
            
            self.stream = open(self.baseFilename, "w")
            # empty file content
            self.stream.seek(0, 0)
            self.stream.truncate(0)


  # /class RotatingFileHandler  - - - - - - - - - - - - - - - - - - - - - - - -



# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class customLogger(object, logging.Logger) :
    """\
      A custom logger class, adds a 'depth' to messages; strings are
      prepended with 'customLogger.depth' * 'customLogger.spacer' (as a
      left margin).
     
      Note: python 2.3, "logging.Logger" is still an old fashioned class.
     
     
      constructor       customLogger(string name, int level)
      
      void              trace(string msg, *largs, **kargs)
      
      int               getDepth(int level)
      int               setDepth(int value, int level)
      
      int               incDepth(void)
      int               decDepth(void)
      
      int               incDepthDebug(int value)
      int               incDepthInfo(int value)
      int               incDepthWarning(int value)
      int               incDepthError(int value)
      int               incDepthCritical(int value)
      int               incDepthTrace(int value)
      
      int               decDepthDebug(int value)
      int               decDepthInfo(int value)
      int               decDepthWarning(int value)
      int               decDepthError(int value)
      int               decDepthCritical(int value)
      int               decDepthTrace(int value)
      
      property  list    excludeLvl
      property  dict    depths
      property  string  spacer
    """


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __init__(self, name, level=logging.NOTSET):
        """\
          constructor customLogger(string name, int level)
          
          Initialize the logger with a name and an optional level.
          The 'depth' and 'spacer' are set to their default values :
            depth = 0 and spacer = ' '
            
          param name                  global logger's name
          param level                 logger level.
                                        defaults to logging.NOTSET
        """
        logging.Logger.__init__(self, name, level)
        self.__excludeLvl = list()
        self.__depths     = dict()
        self.__spacer     = '. '
        self._fillDepths()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  
    def filter(self, record):
        """\
          Determine if a record is loggable by consulting all the filters.
  
          The default is to allow the record to be logged; any filter can
          veto this and the record is then dropped. Returns a zero value
          if a record is to be dropped, else non-zero.
          
          param record                LogRecord to consider
          return                      should the record be kept ?
        """
        if record.levelno in self.__excludeLvl :
          return 0
        else :
          return logging.Logger.filter(self, record)
  
  
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def debug(self, msg, *largs, **kargs) :
        """\
          void debug(string msg, *largs, **kargs)
        
          Logs a message with level DEBUG on this logger. The 'msg' is the
           message format string, and the 'args' are the arguments which
           are merged into 'msg'. The only keyword argument in 'kwargs'
           which is inspected is "exc_info" which, if it does not evaluate
           as False, causes exception information
           (via a call to sys.exc_info()) to be added to the logging
           message.
          
          param msg                   message format string
        """
        buffer = self._prependMargin(msg, logging.DEBUG)
        logging.Logger.debug(self, buffer, *largs, **kargs)


    def info(self, msg, *largs, **kargs) :
        """\
          void info(string msg, *largs, **kargs)
          
          Logs a message with level INFO on this logger. The arguments are
          interpreted as for debug(). 
          
          param msg                   message format string
        """
        buffer = self._prependMargin(msg, logging.INFO)
        logging.Logger.info(self, buffer, *largs, **kargs)


    def warning(self, msg, *largs, **kargs) :
        """\
          void warning(string msg, *largs, **kargs)
          
          Logs a message with level WARNING on this logger. The arguments
          are interpreted as for debug().
          
          param msg                   message format string
        """
        buffer = self._prependMargin(msg, logging.WARNING)
        logging.Logger.warning(self, buffer, *largs, **kargs)


    def error(self, msg, *largs, **kargs) :
        """\
          void error(string msg, *largs, **kargs)
          
          Logs a message with level ERROR on this logger. The arguments are
          interpreted as for debug(). 
          
          param msg                   message format string
        """
        buffer = self._prependMargin(msg, logging.ERROR)
        logging.Logger.error(self, buffer, *largs, **kargs)


    def critical(self, msg, *largs, **kargs) :
        """\
          void critical(string msg, *largs, **kargs)
          
          Logs a message with level CRITICAL on this logger. The arguments
          are interpreted as for debug(). 
          
          param msg                   message format string
        """
        buffer = self._prependMargin(msg, logging.CRITICAL)
        logging.Logger.critical(self, buffer, *largs, **kargs)


    def log(self, lvl, msg, *largs, **kargs) :
        """\
          void info(int lvl, string msg, *largs, **kargs)
          
          Logs a message with level lvl on this logger. The other arguments
          are interpreted as for debug().
          
          param msg                   message format string
        """
        buffer = self._prependMargin(msg, lvl)
        logging.Logger.log(self, lvl, buffer, *largs, **kargs)


    def exception(self, msg, *largs) :
        """\
          void exception(string msg, *largs)
          
          Logs a message with level ERROR on this logger. The arguments
          are interpreted as for debug(). Exception info is added to the
          logging message. This method should only be called from an
          exception handler. 
          
          param msg                   message format string
        """
        buffer = self._prependMargin(msg, logging.ERROR)
        logging.Logger.exception(self, buffer, *largs)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def trace(self, msg, *largs, **kargs) :
        """\
          void trace(string msg, *largs, **kargs)
          
          Logs a message with level TRACE on this logger. The arguments
          are interpreted as for debug(). 
          
          param msg                   message format string
        """
        buffer = self._prependMargin(msg, TRACE)
        logging.Logger.log(self, TRACE, buffer, *largs, **kargs)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def getDepth(self, level=logging.NOTSET) :
        """\
          int getDepth(int level)
          
          Returns the depth value corresponding to the given
          logging 'level'.
          
          param level                 logging level.
                                        Defaults to logging.NOTSET
          return                      depth
        """
        if level in self.__depths :
          return self.__depths[level]
        else :
          return -1


    def setDepth(self, value, level=logging.NOTSET) :
        """\
          int setDepth(int value, int level)
          
          Set the depth for the given logging level. If 'level' is set to
          logging.NOTSET, then all depths are set to the given value.
          The new depth value for the level is returned, if level is not
          found among the inner depths list, then -1 will be returned.
          
          param value                 depth value.
          param level                 logging level.
                                        Defaults to logging.NOTSET
          return                      new depth value
        """
        assert isinstance(value, int)
        
        buffer = max(0, value)
        if level not in self.__depths :
          return -1
        elif logging.NOTSET != level :
          self.__depths[level] = buffer
        else :
          for key in self.__depths :
            self.__depths[key] = buffer
        return buffer

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def incDepth(self, value=1, level=logging.NOTSET) :
        """\
          int incDepth(int value, int level)
          
          Increments message depth for logging level 'level' by given
          'value'. If 'level' is set to logging.NOTSET, then all depths
          will be incremented by 'value'.
          The new depth value for the level is returned, if level is not
          found among the inner depths list, then -1 will be returned.
          
          param value                 depth increment.
                                        Defaults to 1
          param level                 logging level.
                                        Defaults to logging.NOTSET
          return                      new depth value
        """
        try :
          buffer = abs(int(value))
          
          if level not in self.__depths :
            return -1
          elif logging.NOTSET != level :
            self.__depths[level] += buffer
          else :
            for key in self.__depths.iterkeys() :
              self.__depths[key] += buffer
          return self.__depths[level]
          
        except (ValueError, TypeError) :
          return -1


    def decDepth(self, value=1, level=logging.NOTSET) :
        """\
          int decDepth(int value, int level)
          
          Decrements message depth for logging level 'level' by given
          'value'. If 'level' is set to logging.NOTSET, then all depths
          will be decremented by 'value'.
          The new depth value for the level is returned, if level is not
          found among the inner depths list, then -1 will be returned.
          
          param value                 depth decrement.
                                        Defaults to 1
          param level                 logging level.
                                        Defaults to logging.NOTSET
          return                      new depth value
        """
        try :
          buffer = abs(int(value))
          
          if level not in self.__depths :
            return -1
          elif logging.NOTSET != level :
            self.__depths[level] = max(0, self.__depths[level] - buffer)
          else :
            for key in self.__depths.iterkeys() :
              self.__depths[key] = max(0, self.__depths[key] - buffer)
          return self.__depths[level]
          
        except (ValueError, TypeError) :
          return -1


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def incDepthDebug(self, value=1) :
        """\
          int incDepthDebug(int value)
          
          Increments message depth for logging level DEBUG by given 'value'.
          The new depth value for the level is returned, if level is not
          found among the inner depths list, then -1 will be returned.
          
          param value                 depth increment.
                                        Defaults to 1
          return                      new depth value
        """
        return self.incDepth(value, logging.DEBUG)


    def incDepthInfo(self, value=1) :
        """\
          int incDepthInfo(int value)
          
          Increments message depth for logging level INFO by given 'value'.
          The new depth value for the level is returned, if level is not
          found among the inner depths list, then -1 will be returned.
          
          param value                 depth increment.
                                        Defaults to 1
          return                      new depth value
        """
        return self.incDepth(value, logging.INFO)


    def incDepthWarning(self, value=1) :
        """\
          int incDepthWarning(int value)
          
          Increments message depth for logging level WARNING by given
          'value'.
          The new depth value for the level is returned, if level is not
          found among the inner depths list, then -1 will be returned.
          
          param value                 depth increment.
                                        Defaults to 1
          return                      new depth value
        """
        return self.incDepth(value, logging.WARNING)


    def incDepthError(self, value=1) :
        """\
          int incDepthError(int value)
          
          Increments message depth for logging level ERROR by given 'value'.
          The new depth value for the level is returned, if level is not
          found among the inner depths list, then -1 will be returned.
          
          param value                 depth increment.
                                        Defaults to 1
          return                      new depth value
        """
        return self.incDepth(value, logging.ERROR)


    def incDepthCritical(self, value=1) :
        """\
          int incDepthCritical(int value)
          
          Increments message depth for logging level CRITICAL by given
          'value'.
          The new depth value for the level is returned, if level is not
          found among the inner depths list, then -1 will be returned.
          
          param value                 depth increment.
                                        Defaults to 1
          return                      new depth value
        """
        return self.incDepth(value, logging.CRITICAL)


    def incDepthTrace(self, value=1) :
        """\
          int incDepthTrace(int value)
          
          Increments message depth for logging level TRACE by given 'value'.
          TRACE is defined in the asqLogging unit.
          The new depth value for the level is returned, if level is not
          found among the inner depths list, then -1 will be returned.
          
          param value                 depth increment.
                                        Defaults to 1
          return                      new depth value
        """
        return self.incDepth(value, TRACE)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def decDepthDebug(self, value=1) :
        """\
          int decDepthDebug(int value)
          
          Decrements message depth for logging level DEBUG by given 'value'.
          The new depth value for the level is returned, if level is not
          found among the inner depths list, then -1 will be returned.
          
          param value                 depth decrement.
                                        Defaults to 1
          return                      new depth value
        """
        return self.decDepth(value, logging.DEBUG)


    def decDepthInfo(self, value=1) :
        """\
          int decDepthDebug(int value)
          
          Decrements message depth for logging level INFO by given 'value'.
          The new depth value for the level is returned, if level is not
          found among the inner depths list, then -1 will be returned.
          
          param value                 depth decrement.
                                        Defaults to 1
          return                      new depth value
        """
        return self.decDepth(value, logging.INFO)


    def decDepthWarning(self, value=1) :
        """\
          int decDepthWarning(int value)
          
          Decrements message depth for logging level WARNING by given
          'value'.
          The new depth value for the level is returned, if level is not
          found among the inner depths list, then -1 will be returned.
          
          param value                 depth decrement.
                                        Defaults to 1
          return                      new depth value
        """
        return self.decDepth(value, logging.WARNING)


    def decDepthError(self, value=1) :
        """\
          int decDepthError(int value)
          
          Decrements message depth for logging level ERROR by given 'value'.
          The new depth value for the level is returned, if level is not
          found among the inner depths list, then -1 will be returned.
          
          param value                 depth decrement.
                                        Defaults to 1
          return                      new depth value
        """
        return self.decDepth(value, logging.ERROR)


    def decDepthCritical(self, value=1) :
        """\
          int decDepthCritical(int value)
          
          Decrements message depth for logging level CRITICAL by given
          'value'.
          The new depth value for the level is returned, if level is not
          found among the inner depths list, then -1 will be returned.
          
          param value                 depth decrement.
                                        Defaults to 1
          return                      new depth value
        """
        return self.decDepth(value, logging.CRITICAL)


    def decDepthTrace(self, value=1) :
        """\
          int decDepthTrace(int value)
          
          Decrements message depth for logging level TRACE by given 'value'.
          TRACE is defined in the asqLogging unit.
          The new depth value for the level is returned, if level is not
          found among the inner depths list, then -1 will be returned.
          
          param value                 depth decrement.
                                        Defaults to 1
          return                      new depth value
        """
        return self.decDepth(value, TRACE)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  
    def _fillDepths(self) :
        """\
          void _fillDepths(void)
          
          Fills inner __depths dictionnary, for each level (as integer value)
          found in logging module, a depth value is available.
        """
        for key in logging._levelNames.iterkeys() :
          if not isinstance(key, int) :
            pass # not an integer value
          elif key not in self.__depths :
            self.__depths[key] = 0


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  
    def _prependMargin(self, msg, lvl=logging.NOTSET) :
        """\
          string _prependMargin(string msg, int lvl)
          
          Returns 'msg' with a left margin (spacer * depth[lvl]).
          
          param msg                   message.
          param lvl                   logging level (debug, info, error...)
          return                      spacer * depth[lvl] + msg
        """
        result = str(msg)
        if 0 == len(result) :
          return result
        elif lvl not in self.__depths :
          return result
        else :
          return '%s%s' % (self.__spacer * self.__depths[lvl], result)
  

  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _getExcludeLvl(self) :
        """\
          list _getExcludeLvl(void)
          
          Getter - excludeLvl property
        """
        return self.__excludeLvl


    def _getDepths(self) :
        """\
          dict _getDepths(void)
          
          Getter - depths property
        """
        return dict(self.__depths.items())


    def _getSpacer(self) :
        """\
          int _getSpacer(void)
          
          Getter - spacer property
        """
        return self.__spacer


  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _setSpacer(self, value) :
        """\
          void _setSpacer(string value)
          
          Setter - spacer property
        """
        self.__spacer = str(value)


  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    excludeLvl = property(
                doc  = "logging levels to ignore",
                fget = _getExcludeLvl,
                fset = None,
                fdel = None )

    depths     = property(
                doc  = "depth values, for each logging level",
                fget = _getDepths,
                fset = None,
                fdel = None )

    spacer     = property(
                doc  = "string prepended to messages",
                fget = _getSpacer,
                fset = _setSpacer,
                fdel = None )


  # /class customLogger - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
  print __file__
  print __doc__
else :
  logging.addLevelName(TRACE, 'TRACE')
  logging.setLoggerClass(customLogger)


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -