#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
A INI file parsern derivated from ConfigParser.ConfigParser.

Require
  Python        2.3.4

Classes
  class         asqIniParser(void) extends ConfigParser.ConfigParser
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.22 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  ConfigParser
import  os
import  sys

from    py_netasq.commonlib     import asqMapping
from    py_netasq.commonlib     import asqPath

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

# SF Bug #997050 : ConfigParser behavior change
#
# https://sourceforge.net/tracker/?func=detail&atid=105470&aid=997050&group_id=5470
#
# ConfigParser.set() doesn't allow non-string arguments for 'value'
# any more. Actually, this restriction should only be encountered with
# SafeConfigParser, check the cvs revision :
#
# cvs: python/python/dist/src/Lib/ConfigParser.py
# Revision: 1.68, Sun Oct 3 15:55:09 2004 UTC by goodger
#  Document, test, & check for non-string values in ConfigParser.
#  Moved the new string-only restriction added in rev. 1.65 to the
#  SafeConfigParser class, leaving existing ConfigParser & RawConfigParser
#  behavior alone, and documented the conditions under which non-string
#  values work.
#

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


class asqIniParser(object, ConfigParser.ConfigParser) :
    """\
      A INI file parser, derivated from ConfigParser.ConfigParser.
      Provides some more convenient functions to handle file parsing.
      WARNING : ConfigParser does not derivate from "object" :/
      
      void      readFile(string filename)
      void      writeFile(string filename)
      
      void      addSection(string section)
      
      void      assertSection(string section)
      
      void      clear(void)
      void      clearDefaults(void)
      void      clearSection(string section)
      
      void      updateDefaults(dict aDict, var default)
      void      updateSection(string section, dict aDict, var default)
      
      dict      getSectionDict(string section, bool full, bool raw)
      
      string    getOption(string section, string option, var default)
      int       getIntOption(string section, string option, var default)
      float     getFloatOption(string section, string option, var default)
      
      void      setDefault(string option, var value, var default)
      void      setOption(string section, string option, var value, var default)
      void      setItems(string section, list items, var default)
      
      property string dirname      current file's directory name
      property string filename     current file's name
      property string fullpath     current file's fullpath
      property int    ljustify     option's key justification
    """
    
    DEFAULT_LJUSTIFY = 15
    DEFAULT_FILENAME = 'cfgFile.ini'


  # Constructor - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __init__(self, defaults=None) :
        """\
          constructor asqIniParser(dict defaults)
          
          Create a new build file parser, 'defaults' dictitonary will be
          used to set the "DEFAULT" section's options; those options can be
          used as variables.
          
          param defaults              DEFAULT options
        """
        # "defaults" dict is available through ConfigParser._defaults
        # attributes but WE WON'T RELY ON THAT (may break at anytime).
        if isinstance(defaults, dict) :
          self.__defaults = defaults.copy()
        else :
          self.__defaults = dict()
        # super() cannot be used, ConfigParser is not a "new style" class
        ConfigParser.ConfigParser.__init__(self, self.__defaults)
        
        self._ljustify = self.DEFAULT_LJUSTIFY
        # set dirname and filename to their default values
        self._guessContext(None)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def readFile(self, filename) :
        """\
          void readFile(string filename)
          
          Read and parse configuration data from the specified file. The
          opened descriptor will be properly close. If given file does not
          exist then a exception will be raised.
          
          param filename              file to read
        """
        source = open(filename, 'r')
        try :
          self.readfp(source)
        finally :
          source.close()


    def writeFile(self, filename) :
        """\
          void writeFile(string filename)
          
          Write a representation of the configuration to the specified file.
          If given file does not exist then it will be created.
          
          param filename              file to write
        """
        target = open(filename, 'w')
        try :
          self.write(target)
        finally :
          target.close()


    def addSection(self, section) :
        """\
          void addSection(string section)
          
          Add a section named section to the instance. If a section by the
          given name already exists, then nothing happens.
          
          param section               section name
        """
        try :
          self.add_section(section)
        except ConfigParser.DuplicateSectionError, e :
          # mute DuplicateSectionError exceptions
          pass


    def assertSection(self, section) :
        """\
          void assertSection(string section)
          
          Check that given 'section' do exist; if not an assert except will
          be raised.
          
          param section               section to check
        """
        try :
          self.options(section)
        except ConfigParser.NoSectionError, e :
          buffer = "Section not found '%s'" % (str(section), )
          raise AssertionError(buffer)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _guessContext(self, fp) :
        """\
          void _guessContext(object fp)
          
          Sets dirname and filename properties according to given (assumed)
          file object 'fp'; If none can be guessed then the current working
          directory and a default file name will be used instead.
          
          param fp                    file object
        """
        try :
          self._dirname  = asqPath.directory(fp.name)
          self._filename = os.path.basename(fp.name)
        except AttributeError, e :
          self._dirname  = asqPath.directory(os.getcwd())
          self._filename = os.path.basename(self.DEFAULT_FILENAME)


    def _read(self, fp, fpname) :
        """\
          void _read(object fp, string fpname)
          
          Parse a sectioned setup file.
          The dirname and filename properties are then respectively set to
          the file's directory and the file's name; If none can be guessed
          then the current working directory and a default file name will
          be used instead.
          The sections in setup file contains a title line at the top,
          indicated by a name in square brackets (`[]'), plus key/value
          options lines, indicated by `name: value' format lines.
          Continuations are represented by an embedded newline then
          leading whitespace.  Blank lines, lines beginning with a '#',
          and just about everything else are ignored.
          
          param fp                    file object
          param fpname                file name
        """
        self._guessContext(fp)
        ConfigParser.ConfigParser._read(self, fp, fpname)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _getSectionLines(self, section, options, default='') :
        """\
          list _getSectionLines(string section, var options, string default)
          
          Formats section's options as a list of strings, in order to be
          displayed; given 'options' must either be a dictionary
          { key: val, key: val } or a list [(key, val), (key, val)].
          If one of the given 'options' list value is None,  then it is
          replaced by the 'default' value.
          Note : no line separators are added.
          
          param section               section name
          param options               sections options sequence
          param default               default value, defaults to ''
          return                      writeable strings list
        """
        result = list()
        result.append('[%s]' % (section, ))
        
        # better use an ordered list than a dictionary structure, since
        # dictionariess do not care about key order....
        if isinstance(options, dict) :
          options = options.items()
        for (key, val) in options :
          if '__name__' == key :
            continue # skip
          else :
            key = key.ljust(self._ljustify)
          if val is None :
            val = str(default).replace('\n', '\n\t')
          else :
            val = str(val).replace('\n', '\n\t')
          result.append('%s = %s' % (key, val, ))
        result.append('')
        
        return result


    def _writeDefaults(self, out, raw=True) :
        """\
          void _writeDefaults(object out, bool raw=True)
          
          Write "DEFAULT" section, if 'raw' is set to False, then all the
          "%" interpolations will be expanded. The 'out' object must behave
          like a file descriptor: at least the "write" method must be
          present (sys.stdout can do the trick too).
          
          param out                   output stream/file descriptor
          param raw                   get raw values (no interpolations)
                                      defaults to True.
        """
        if self.__defaults :
          if raw :
            self._writeSection(ConfigParser.DEFAULTSECT, self.__defaults, out)
          else :
            buffer = self.__defaults.copy()
            for key in self.__defaults.iterkeys() :
              buffer[key] = self.get(ConfigParser.DEFAULTSECT, key, False)
            self._writeSection(ConfigParser.DEFAULTSECT, buffer, out)


    def _writeSection(self, section, options, out, default='') :
        """\
          void _writeSection(string section, var options, object out)
          
          Write section to given output; given 'options' must either be a
          dictionary { key: val, key: val } or a list [(key, val), (key, val)]
          The 'out' object must behave like a file descriptor: at least the
          "write" method must be present (sys.stdout can do the trick too).
          If one of the given 'options' list value is None,  then it is
          replaced by the 'default' value.
          
          param section               section name
          param options               sections options sequence
          param out                   output stream/file descriptor
          param default               default value, defaults to ''
        """
        buffer = self._getSectionLines(section, options, default)
        out.write('\n'.join(buffer))


    def _writeOption(self, key, val, out, default='') :
        """\
          void _writeOption(string key, string val, object out)
          
          Write option line to given output, justify 'key' string. The 'out'
          object must behave like a file descriptor: at least the "write"
          method must be present (sys.stdout can do the trick too). If given
          'val' is None,  then it is replaced by the 'default' value.
          
          param key                   option key
          param val                   option value
          param out                   output stream/file descriptor
          param default               default value, defaults to ''
        """
        key = key.ljust(self._ljustify)
        if val is None :
          val = str(default).replace('\n', '\n\t')
        else :
          val = str(val).replace('\n', '\n\t')
        out.write('%s = %s\n' % (key, val))


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def clear(self) :
        """\
          void clear(void)
          
          Empty parser dataset, all options and sections are trashed.
        """
        self.clearDefaults()
        for section in self.sections() :
          for option in self.options(section) :
            self.remove_option(section, option)
          self.remove_section(section)


    def clearDefaults(self) :
        """\
          void clearDefaults(void)
          
          Empty parser DEFAULT section, all default options are trashed.
          WARNING : future version of ConfigParser may break this function.
        """
        self.__defaults.clear()


    def clearSection(self, section=None) :
        """\
          void clearSection(string section)
          
          Empty a section, or all sections if no section name is provided.
          If a 'section' is given, but does not exist then nothing happens.
          Section will do remain, only options will be trashed.
          
          param section               section name, defaults to None
        """
        if section is None :
          for section in self.sections() :
            for option in self.options(section) :
              self.remove_option(section, option)
        elif self.has_section(section) :
          for option in self.options(section) :
            self.remove_option(section, option)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def updateDefaults(self, aDict, default='') :
        """\
          void updateDefaults(dict aDict, var default)
          
          Update instance wide defaults values with provided 'aDict'
          dictionary; if a value of given dictionary is found to be None,
          then it is replaced with the 'default' value.
          
          e.g. :
            defaults()
            => foo = bar
               ham = ha
          
            aDict = { 'foo': 'oof', 0: None, 1: 'mr' }
            updateDefaults(aDict, '<none>')
            => foo = oof
               ham = ha
               0   = <none>
               1   = mr
          
          param aDict                 new value
          param default               default value, defaults to ''
        """
        for (key, val) in aDict.iteritems() :
          self.setDefault(key, val, default)


    def updateSections(self, section, aDict, default='') :
        """\
          void updateSection(string section, dict aDict, var default)
          
          Update 'section' values with provided 'aDict' dictionary;
          if a value of given dictionary is found to be None, then
          it is replaced with the 'default' value.
          
          param section               section name
          param aDict                 new value
          param default               default value, defaults to ''
        """
        # ensures that section does exist in dataset
        self.addSection(section)
        for (key, val) in aDict.iteritems() :
          self.setOption(section, key, val, default)


    def defaults(self, raw=True) :
        """\
          dict defaults(bool raw)
          
          Return a dictionary containing the instance-wide defaults.
          Whereas the inherited method only return raw values, if
          'raw' is set to False, then all the "%" interpolations will
          be expanded.
          
          param raw                   get raw values (no interpolations)
                                      defaults to True.
          return                      instance-wide defaults
        """
        if raw :
          return super(asqIniParser, self).defaults()
        else :
          result = self.items(ConfigParser.DEFAULTSECT, raw, vars=None)
          return dict(result)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _extractOption(self, name, aList, default='') :
        """\
          string _extractOption(string name, list aList, var default)
          
          Return value for named option, if option is not found, then
          'default' is returned; Given 'aList' is modified, if option is
          found, then it is deleted from the list.
          
          param name                  option name
          param aList                 section options as a list
          param default               returned if type is not found,
                                        defaults to ''
          return                      value
        """
        for indx in xrange(len(aList)) :
          if name == aList[indx][0] :
            return aList.pop(indx)[1]
        else :
          return default


    def _extractIntOption(self, name, aList, default=0) :
        """\
          int _extractIntOption(string name, list aList, var default)
          
          Return value for named option, if option is not found, then
          'default' is returned; Given 'aList' is modified, if option is
          found, then it is deleted from the list.
          
          param name                  option name
          param aList                 section options as a list
          param default               returned if type is not found,
                                        defaults to ''
          return                      value
        """
        result = self._extractOption(name, aList, None)
        if result is not None :
          return int(result)
        else :
          return default


    def _extractFloatOption(self, name, aList, default=0.0) :
        """\
          float _extractFloatOption(string name, list aList, var default)
          
          Return value for named option, if option is not found, then
          'default' is returned; Given 'aList' is modified, if option is
          found, then it is deleted from the list.
          
          param name                  option name
          param aList                 section options as a list
          param default               returned if type is not found,
                                        defaults to ''
          return                      value
        """
        result = self._extractOption(name, aList, None)
        if result is not None :
          return float(result)
        else :
          return default


    def _extractBoolOption(self, name, aList, default=False) :
        """\
          bool _extractBoolOption(string name, list aList, var default)
          
          Return value for named option, if option is not found, then
          'default' is returned; Given 'aList' is modified, if option is
          found, then it is deleted from the list.
          
          param name                  option name
          param aList                 section options as a list
          param default               returned if type is not found,
                                        defaults to ''
          return                      value
        """
        result = self._extractOption(name, aList, None)
        if result is not None :
          return bool(result)
        else :
          return default


  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def getSectionDict(self, section, full=False, raw=True) :
        """\
          dict getSectionDict(string section, bool full, bool raw)
          
          Returns a dictionary containing the 'section' options, if 'full'
          is set to True, then instance-wide defaults will be included.
          The "defaults()" method should be used to retrieve the instance
          wide DEFAULT values.
          An NoSectionError may be raised if given 'section' is missing
          from the dataset.
          
          param section               section name
          param full                  include defaults options
                                        defaults to False
          param raw                   get raw values (no interpolations)
                                      defaults to True.
          return                      a dictionnary
        """
        # self.options() may raise NoSectionError
        buffer = self.options(section)
        result = dict()
        
        for item in buffer :
          result[item] = self.get(section, item, raw)
        if full :
          return result
        else :
          return asqMapping.difference(result, self.defaults(raw))


    def getOption(self, section, option, default='') :
        """\
          string getOption(string section, string option, var default)
          
          Get an option value for the named section. If 'section' does not
          contain given 'option', then the 'default' value is returned.
          If section does not exist, then an excpetion may be raised.
          Since 'default' can be of ANY type, there is no guarantee that
          a string will always be returned.
          
          param section               section
          param option                option to get
          param default               default value if not found,
                                        defaults to ''
          return                      option value
        """
        try :
          return self.get(section, option)
        except ConfigParser.NoOptionError :
          return default


    def getIntOption(self, section, option, default=0) :
        """\
          int getIntOption(string section, string option, var default)
          
          A convenience method which coerces the option in the specified
          section to an integer. If 'section' does not contain given
          'option', then the 'default' value is returned.
          If section does not exist, then an excpetion may be raised.
          Since 'default' can be of ANY type, there is no guarantee that
          a int will always be returned.
          
          param section               section
          param option                option to get
          param default               default value if not found,
                                        defaults to 0
          return                      option value
        """
        try :
          return self.getint(section, option)
        except ConfigParser.NoOptionError :
          return default


    def getFloatOption(self, section, option, default=0.0) :
        """\
          float getFloatOption(string section, string option, var default)
          
          A convenience method which coerces the option in the specified
          section to a floating point number. If 'section' does not contain
          given 'option', then the 'default' value is returned.
          If section does not exist, then an excpetion may be raised.
          Since 'default' can be of ANY type, there is no guarantee that
          a int will always be returned.
          
          param section               section
          param option                option to get
          param default               default value if not found,
                                        defaults to 0.0
          return                      option value
        """
        try :
          return self.getfloat(section, option)
        except ConfigParser.NoOptionError :
          return default


    def getBooleanOption(self, section, option, default=False) :
        """\
          bool getBooleanOption(string section, string option, var default)
          
          A convenience method which coerces the option in the specified
          section to a Boolean value. Note that the accepted values for the
          option are "1", "yes", "true", and "on", which cause this method
          to return True, and "0", "no", "false", and "off", which cause it
          to return False.
          These string values are checked in a case-insensitive manner. Any
          other value will cause it to raise ValueError.
          If 'section' does not contain given 'option', then the 'default'
          value is returned.
          If section does not exist, then an excpetion may be raised.
          Since 'default' can be of ANY type, there is no guarantee that
          a int will always be returned.
          
          param section               section
          param option                option to get
          param default               default value if not found,
                                        defaults to False
          return                      option value
        """
        try :
          return self.getboolean(section, option)
        except ConfigParser.NoOptionError :
          return default


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _getDirName(self) :
        """\
          string _getDirName(void)
          
          Getter - dirname property
          Returns current file's directory name.
          
          return                      current file's directory name
        """
        return self._dirname


    def _getFileName(self) :
        """\
          string _getFileName(void)
          
          Getter - filename property
          Returns current file's name.
          
          return                      current file's name
        """
        return self._filename


    def _getFullpath(self) :
        """\
          string _getFullpath(void)
          
          Getter - fullpath property
          Returns the fullpath string for the current file
          
          return                      current file's path
        """
        return os.path.join(self._dirname, self._filename)


    def _getLJustify(self) :
        """\
          int _getLJustify(void)
          
          Returns option's key left justification value.
          
          return                      left justification value
        """
        return self._ljustify


  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def setDefault(self, option, value, default='') :
        """\
          void setDefault(string option, var value, var default)
          
          Add option to the DEFAULT section. This function should behave just
          as 'setOption()'; If given 'value' is found to be None, then it is
          replaced by the 'default' value.
          e.g. :
            setDefault(option, None, '0') => option = 0
          
          param option                option to set
          param value                 option value
          param default               default value, defaults to ''
        """
        self.setOption(ConfigParser.DEFAULTSECT, option, value, default)


    def setOption(self, section, option, value, default='') :
        """\
          void setOption(
                  string section, string option, var value, var default)
          
          If the given section exists, set the given option to the specified
          value; otherwise raise NoSectionError. If given 'value' is found
          to be None, then it is replaced by the 'default' value.
          e.g. :
            setOption(section, option, None, '0') => option = 0
          
          Note :
            ConfigParser replaces 'None' values with empty string values.
          Note :
            Please be aware of Bug #997050 (sourceforge) "ConfigParser
            behavior change"; as a result, values WILL ALWAYS BE converted
            to strings !!
          
          param section               section
          param option                option to set
          param value                 option value
          param default               default value, defaults to ''
        """
        if value is None :
          self.set(section, option, str(default))
        else :
          self.set(section, option, str(value))


    def setItems(self, section, items, default='') :
        """\
          void      setItems(string section, list items, var default)
          
          Set given 'section' options using 'items', this function acts as
          the counterpart to the 'items()' method. If one of the 'items'
          value is found to be None, then it is replaced by the 'default'
          value (check "setOption()" function).
          e.g.
            items = [('command', 'update'), ('module', 'unittest'), ('dummy', None)]
            setItems('cvsTask_0', items, '')
            =>  [cvsTask_0]
                command = update
                module = unittest
                dummy = 
          
          param section               section
          param items                 list of tuple (key, val)
          param default               default value, defaults to ''
        """
        for (key, val) in items :
          self.setOption(section, key, val, default)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _setLJustify(self, value) :
        """\
          void _setLJustify(int value)
          
          Set option's key left justification value
          
          param value                 left justification value
        """
        self._ljustify = int(value)


  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    dirname  = property(
                doc  = "current file's directory name",
                fget = _getDirName,
                fset = None,
                fdel = None )

    filename = property(
                doc  = "current file's name",
                fget = _getFileName,
                fset = None,
                fdel = None )

    fullpath = property(
                doc  = "current file's path",
                fget = _getFullpath,
                fset = None,
                fdel = None )

    ljustify  = property(
                doc  = "options keys left justification",
                fget = _getLJustify,
                fset = _setLJustify,
                fdel = None )


  # /class asqIniParser - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -