#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
(Little) Less common, but still useful, debug utilities.

Require
  Python        2.2

Functions
  string        calleeName(void)
  string        callerName(void)
  var           varName(var value, bool getAll, tuple lookIn, tuple exclude)
  list          parameters(*argList, **argDict)
  string        debugValue(string name, var value, int tab)
  string        debugMessage(string message, int tab)

Classes
  class         objectDebug(object aClass, string objName)
                  extends py_netasq.commonlib.asqLogging::objectLog
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.22 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  inspect
import  sys

import  py_netasq.commonlib.asqLogging          as asqLogging
import  py_netasq.commonlib.asqSequence         as asqSequence

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def calleeName() :
    """\
      string calleeName(void)
      
      Returns the definition name of the currently running function. Useful
      to create error messages that don't need to be changed as the function
      name is modified.
      
      Based upon an activestate's cookbook recipe by Alex Martelli
      "Determining Current Function Name"
      > http://aspn.activestate.com/ASPN/Cookbook/Python/Recipe/66062
        
      return                      definition name of current function
    """
    return sys._getframe(1).f_code.co_name


def callerName() :
    """\
      string callerName(void)
      
      Returns the definition name of currently running function caller.
      
      Based upon an activestate's cookbook recipe by Alex Martelli
      "Determining Current Function Name"
      > http://aspn.activestate.com/ASPN/Cookbook/Python/Recipe/66062
        
      return                      definition name of current function caller
    """
    return sys._getframe(2).f_code.co_name


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def varName(value, getAll=False, lookIn=('__main__',), exclude=()) :
    """\
      var varName(var value, bool getAll, tuple lookIn, tuple exclude)
      
      Returns the variable name(s) of given "value". The result's type
      depends on the "getAll" parameter.
      If "getAll" is set to False, then the result can either be None,
      if "value" is an anonymous instance, or a string
      Else if "getAll" is set to True, then a list will always be returned;
      All names in the list will be unique. An empty list could be returned.
      
      Not all modules will be checked, only those present in "lookIn" will
      be considered; The "exclude" list is a kind of name filter, the
      strings from that list won't be returned by the function.
      
      Warning, if "value" can be accessed from different names, there
      is NO WAY to guess which name will be returned first.
      
      Based upon an faqts entry, by Wolfgang Lipp
      "Introspective: How to find out the names of variables and objects"
      > http://www.faqts.com/knowledge_base/view.phtml/aid/7019/fid/593
      
      param value                 value to identify
      param getAll                search all variable names
                                    defaults to False
      param lookIn                modules to look into
                                    defaults to ('__main__', )
      param exclude               var name to exclude from result
                                    defaults to ()
      return                      name(s) "value" can be accessed under
    """
    if not getAll :
      # default value
      result = None
    else :
      # store names as dict key to avoid duplicated items
      result = dict()
    
    buffer = getattr(value, '__name__', None)
    
    if buffer is None :
      pass
    elif buffer in exclude :
      pass
    elif not getAll :
      return buffer
    else :
      result[buffer] = None
    
    buffer = asqSequence.intersection(sys.modules.keys(), lookIn)
    buffer = [ sys.modules[x] for x in buffer ]
    
    for module in filter(None, buffer) :
      for (key, val) in module.__dict__.iteritems() :
        if val is not value :
          continue
        elif key in exclude :
          continue
        elif not getAll :
          return key
        else :
          result[key] = None
      
    if result is None :
      return result
    else :
      return result.keys()


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def parameters(*argList, **argDict) :
    """\
      list parameters(*argList, **argDict)
      
      Returns given parameters as a list of STRING values. The "repr()"
      built in function is used to convert values to string.
      Very useful to display the arguments passed to a function.
      e.g.
        parameters(True, baseDir=r'C:\.')
        => [ "True", "baseDir='C:\\.'" ]
      
      param *argList              dynamic args list
                                    ex [ arg1, arg2, ... ]
      param **argDict             dynamic named args dict
                                    ex { 'argA' : valA, 'argB' : valB, ... }
      return                      args as a list of strings
    """
    result = []
    if argList :
      buffer = map(repr, argList)
      result.extend( buffer )
    if argDict :
      buffer = [ '%s=%s' % (x[0], repr(x[1])) for x in argDict.items() ]
      result.extend( buffer )
    return result


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def debugValue(name, value, tab=0, showType=True) :
    """\
      string debugValue(string name, var value, int tab, bool showType)
      
      Returns a string especially formatted for debugging purposes :
        "<name> <type(value)> <value>"
      the string is prepended with 'tab' space chars (left margin width).
      
      param name                  name of given value
      param value                 value
      param tab                   left margin length, defaults to 0
      param showType              do show value type ?, defaults to True
      return                      formatted string
    """
    prefix = ' ' * tab
    result = list()
    
    result.append(prefix + name.ljust(15))
    if showType : result.append(str(type(value)).ljust(20))
    result.append(str(value))
    return ' '.join(result)


def debugMessage(message, tab=0) :
    """\
      string debugMessage(string message, int tab)
      
      Returns a string especially formatted for debugging purposes.
      The string is prepended with 'tab' space chars (left margin width).
      
      param message               a string
      param tab                   left margin length, defaults to 0
      return                      formatted string
    """
    if not message :
      return ''
    else :
      result = list()
      result.append(' ' * tab)
      result.append(message)
      return ''.join(result)


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class objectDebug(asqLogging.objectLog) :
    """\
      Debug wrapper around given class and one of its instance. Displays
      the method calls and attribute settings code line before actually
      doing it.
      
      constructor       objectDebug(object aClass, string objName)
      
      void              classAttributes(void)
      
      void              objecInit(*argList, **argDict)
      void              objecDel(void)
      
      var               objectGetAttr(string attr)
      void              objectSetAttr(string attr, var value)
      void              objectDelAttr(string attr)
      
      var               objectCallFunc(string func, *argList, **argDict)
    """
    
    def __init__(self, aClass, objName=None) :
        """\
          constructor objectDebug(object aClass, string objName)
          
          Create a wrapper around given class 'aClass', 'objName' will be
          used for display messages for the name of the object instanciated
          from aClass.
          If 'objName' is not set, then a name will be built from the class
          name; e.g. if class name is 'dummy', the instance will be named
          'aDummy'.
          
          param aClass                class to debug
          param objName               instance name
                                        defaults to None
        """
        assert aClass is not None
        
        super(objectDebug, self).__init__()
        self.__class   = aClass
        self.__object  = None
        self.__clsName = aClass.__name__
        self.__objName = objName
        
        if not isinstance(objName, basestring) :
          self.__objName = 'a%s' % (capitalize(self.__clsName),)
        elif 0 == len(objName) :
          self.__objName = 'a%s' % (capitalize(self.__clsName),)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _resolveAttrPath(self, attr) :
        """\
          tuple _resolveAttrPath(string attr)
          
          Parses 'complex' attribute path, such as "aObject.prop.subProp",
          and returns a pair, (head, tail) where 'tail' is the last
          string chunk after the last dot and 'head' is the concerned
          sub property.
          AttributeError exception may be raised.
          
          e.g.
            "aObject.prop.subProp"      => (self.prop, 'subProp')
            "aObject.prop"              => (self, 'prop')
          
          param attr                  attribute name
          return                      (object, name)
        """
        buffer = filter(None, attr.split('.'))
        attrib = buffer.pop(-1)
        target = self.__object
        
        for chunk in buffer :
          target = getattr(target, chunk)
        
        return (target, attrib)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  
    def classAttributes(self) :
        """\
          void classAttributes(void)
          
          Prints out (actually to the logging system) the class's
          attributes, name and value.
        """
        def __isNotMethod(x) :
          if inspect.ismethod(x) :
            return False
          elif inspect.isbuiltin(x) :
            return False
          elif inspect.ismethoddescriptor(x) :
            return False
          elif inspect.isdatadescriptor(x) :
            return False
          else :
            return True
        
        def __isVisible(x) :
          return not x[0].startswith('_')
        
        assert self.__class is not None
        
        buffer = inspect.getmembers(self.__class, __isNotMethod)
        buffer = filter(__isVisible, buffer)
        
        if not buffer :
          self._print('* no propreties for class %s' % (self.__clsName,), True)
        else :
          for (key, val) in buffer :
            self._print('> %s.%s' % (self.__clsName, key), True)
            self._incDepth()
            self._print(repr(val), True)
            self._decDepth()
  
  
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def objectInit(self, *argList, **argDict) :
        """\
          void objecInit(*argList, **argDict)
          
          Instanciates an object from the inner class. Parameters are
          directly passed to the constructor.
          
          param *argList              dynamic args list
                                        ex [ arg1, arg2, ... ]
          param **argDict             dynamic named args dict
                                        ex { 'argA' : valA, 'argB' : valB, ... }
        """
        parmtrs = parameters(*argList, **argDict)
        parmtrs = ', '.join(parmtrs)
        command = '> %s = %s(%s)' % (self.__objName, self.__clsName, parmtrs)
        
        self._print(command, True)
        self._incDepth()
        try :
          self.__object = self.__class(*argList, **argDict)
        finally :
          self._decDepth()


    def objectDel(self) :
        """\
          void objecDel(void)
          
          Deletes the inner instance object.
        """
        command = '> del %s' % (self.__objName)
        
        self._print(command, True)
        self._incDepth()
        try :
          
          del self.__object
          self.__object = None
          
        finally :
          self._decDepth()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def objectGetAttr(self, attr) :
        """\
          var objectGetAttr(string attr)
          
          Get attribute of inner instance; Tries to resolve "complex"
          attribute path, such as "aObject.prop.subProp". AttributeError
          exceptions are caught.
          
          param attr                  attribute name
          return                      attribute value
        """
        if self.__object is None :
          raise Exception('no object instanciated, use objectInit() first')
        elif not isinstance(attr, basestring) :
          raise TypeError( '"attr" string value expected got %s' % str(type(attr)) )
        
        command = '> %s.%s' % (self.__objName, attr)
        #~ self._print(command, True)
        self._incDepth()
        try :
          
          try :
            (target, attrib) = self._resolveAttrPath(attr)
            return getattr(target, attrib)
          except AttributeError, e :
            self._print('* %s.%s not found' % (self.__objName, attr))
            return None
          
        finally :
          self._decDepth()


    def objectSetAttr(self, attr, value) :
        """\
          void objectSetAttr(string attr, var value)
          
          Set attribute of inner instance with given value; Tries to
          resolve "complex" attribute path, such as "aObject.prop.subProp".
          AttributeError exceptions are caught.
          
          param attr                  attribute name
          param value                 attribute value
        """
        if self.__object is None :
          raise Exception('no object instanciated, use objectInit() first')
        elif not isinstance(attr, basestring) :
          raise TypeError( '"attr" string value expected got %s' % str(type(attr)) )
        
        command = '> %s.%s = %s' % (self.__objName, attr, repr(value))
        self._print(command, True)
        self._incDepth()
        try :
          
          try :
            (target, attrib) = self._resolveAttrPath(attr)
            setattr(target, attrib, value)
          except AttributeError, e :
            self._print('* %s.%s not found' % (self.__objName, attr))
          
        finally :
          self._decDepth()


    def objectDelAttr(self, attr) :
        """\
          void objectDelAttr(string attr)
          
          Deletes inner instance attribute; Tries to resolve "complex"
          attribute path, such as "aObject.prop.subProp".
          AttributeError exceptions are caught.
          
          param attr                  attribute name
        """
        if self.__object is None :
          raise Exception('no object instanciated, use objectInit() first')
        elif not isinstance(attr, basestring) :
          raise TypeError( '"attr" string value expected got %s' % str(type(attr)) )
        
        command = '> del %s.%s' % (self.__objName, attr)
        self._print(command, True)
        self._incDepth()
        try :
          
          try :
            (target, attrib) = self._resolveAttrPath(attr)
            delattr(target, attrib)
          except AttributeError, e :
            self._print('* %s.%s not found' % (self.__objName, attr))
          
        finally :
          self._decDepth()


    def objectBatchSetAttr(self, aDict) :
        """\
          void objectSetAttr(dict aDict)
          
          Set attributes of inner instance with given values, dictionnary's
          keys stand for attribute names; Tries to resolve "complex"
          attribute path, such as "aObject.prop.subProp".
          AttributeError exceptions are caught.
          
          param aDict                 attributes { name : value, ... }
        """
        if self.__object is None :
          raise Exception('no object instanciated, use objectInit() first')
        elif not isinstance(aDict, dict) :
          raise TypeError( '"aDict" dictionnary value expected got %s' % str(type(aDict)) )
        
        for (key, val) in aDict.iteritems() :
          self.objectSetAttr(key, val)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def objectCallFunc(self, func, *argList, **argDict) :
        """\
          var objectCallFunc(string func, *argList, **argDict)
          
          Call inner instance method with given parameters, returns
          the function result.
          AttributeError exceptions are caught.
          
          param func                  method name
          param *argList              dynamic args list
                                        ex [ arg1, arg2, ... ]
          param **argDict             dynamic named args dict
                                        ex { 'argA' : valA, 'argB' : valB, ... }
          return                      function result
        """
        assert self.__object is not None                                , \
                'no object instanciated, use objectInit() first'
        assert isinstance(func, basestring)
       
        parmtrs = parameters(*argList, **argDict)
        parmtrs = ', '.join(parmtrs)
        command = '> %s.%s(%s)' % (self.__objName, func, parmtrs)
       
        self._print(command, True)
        self._incDepth()
        try :
          
          try :
            (target, attrib) = self._resolveAttrPath(func)
            method = getattr(target, attrib)
          except AttributeError, e :
            self._print('* %s.%s not found' % (self.__objName, attrib))
            return None
          else :
            return method(*argList, **argDict)
          
        finally :
          self._decDepth()


  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _getObject(self) :
        """\
          object _getObject(void)
          
          Getter - object property
        """
        return self.__object

  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    object    = property(
                  doc  = "clas instance",
                  fget = _getObject,
                  fset = None,
                  fdel = None )


  # /class objectDebug  - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -