#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
Command line utilities.
cmdLine is a simple command line building class, provide methods
 to surround arguments with quotes and ease ;).
cmdHandler is a simple object to execute commands, given as
 cmdLine instances.

Require
  Python        2.3
  
Classes
  class         cmdLine(void) extends list
  class         cmdHandler(object cmdLine)
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.18 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  os
import  time

import  py_netasq.commonlib.asqString           as asqString

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class cmdLine(list) :
    """\
      A simple class to ease command line building. The command is handled
      as a simple list : the first item is the command's exe, others are
      command's args (or at least it should be).
      
      This class implements the __str__ function.
      Beware: this is a new-style class (derivated from 'list')
      
      
      void              appendQuoted(string s)
      void              insertQuoted(int i, string s)
      void              removeQuoted(string s)
      
      void              clear(void)
    """

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __str__(self) :
        """\
          string __str__(void)
          
          Called by the 'str()' built-in function and by the 'print' statement
          to compute the "informal" string representation of an object. This
          differs from __repr__() in that it does not have to be a valid
          Python expression: a more convenient or concise representation may
          be used instead. The return value must be a string object.
          
          Return full command line, ready to be used.
          
          return                      string representation
        """
        return ' '.join(filter(None, self[:]))


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def appendQuoted(self, s) :
        """\
          void appendQuoted(string s)
          
          Append given string 's' to command's chunks (self.items), double
          quotes are added on each sides of the string.
          
          param s                     string to append
        """
        self.append(asqString.quoteString(s, '"', '\\'))


    def insertQuoted(self, i, s) :
        """\
          void insertQuoted(int i, string s)
          
          Insert given string 's' at given position 'i' to command's chunks;
          double quotes are added on each sides of the string.
          
          param i                     string index in list
          param s                     string to append
        """
        self.insert(i, asqString.quoteString(s, '"', '\\'))


    def removeQuoted(self, s) :
        """\
          void removeQuoted(string s)
          
          Removes the first string that matches the quoted version of 's'.
          
          param s                     string to remove
        """
        self.remove(asqString.quoteString(s, '"', '\\'))


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def assign(self, source) :
        """\
          void assign(object source)
          
          Overwrites current cmdLine values with values provided by
          "source" object. "source" should be an cmdLine instance.
          
          param source                cmdLine to copy values from
        """
        if not isinstance(source, cmdLine) :
          self.clear()
        else :
          self[:] = source[:]


    def clear(self) :
        """\
          void clear(void)
          
          Empty command line items
        """
        self[:] = []


  # /class cmdLine  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -



# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


class cmdHandler(object) :
    """\
      A simple object to execute commands, given as cmdLine instances.
      
      
      constructor       cmdHandler(object aCmdLine)
      
      var               run(object command, bool raw, *arglist, **argdict)
      
      property object   cmdLine   command line to execute
      property object   cmdTime   command execution timestamp
      property list     output    output from the last command ran
    """


    def __init__(self, aCmdLine=None) :
        """\
          constructor cmdHandler(object aCmdLine)
          
          Create a new object intended to ease command line execution.
          
          param aCmdLine              cmdLine to execute.
                                       defaults to None
        """
        # output : retains the output of the very last ran task
        self._output  = list()
        # command execution timestamp, as a struct_time object
        self._cmdTime = None
        # command line object, hold the shell command string
        self._cmdLine = cmdLine()
        # use the setter method - self._cmdLine.assign(aCmdLine)
        self.cmdLine  = aCmdLine
        
        # custom event handlers
        self._onProcessInput  = None
        self._onProcessOutput = None


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def run(self, aCmdLine=None, raw=False, *arglist, **argdict) :
        """\
          var run(object aCmdLine, bool raw, *arglist, **argdict)
          
          Execute inner command line, if "command" is given, it will replace
          the current cmdLine instance; The shell status is returned,
          'None' if execution was succesful, any other value means failure.
          If "raw" is set to True, then no peculiar operation should be done
          upon the command's output.
          
          param aCmdLine              cmdLine to execute.
                                       defaults to None
          param raw                   disable output formatting
                                        defaults to False
          param *arglist              extra parameters list.
          param **argdict             extra named parameters.
          return                      shell status
        """
        self._output[:] = []
        self._cmdTime   = time.localtime()
        
        if aCmdLine is not None :
          self.cmdLine = aCmdLine
        if not bool(self._cmdLine) :
          return None
        
        # the cmdLine list may not be empty, but filled with whitespace
        cmdStr = str(self._cmdLine).strip()
        if 0 == len(cmdStr) :
          return None
        
        self._setup(*arglist, **argdict)
        try :
          
          self._processInput(cmdStr, raw, *arglist, **argdict)
          result = self._execute(cmdStr, raw, *arglist, **argdict)
          
        finally :
          self._teardown(*arglist, **argdict)
          
        return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _setup(self, *arglist, **argdict) :
        """\
          void _setup(*arglist, **argdict)
          
          Called right before the actual command execution; should be
          overidden to set up the execution environment.
          
          param *arglist              extra parameters list.
          param **argdict             extra named parameters.
        """
        pass


    def _teardown(self, *arglist, **argdict) :
        """\
          void _teardown(*arglist, **argdict)
          
          Called right after the actual command execution; should be
          overidden to clean up the execution environment.
          
          param *arglist              extra parameters list.
          param **argdict             extra named parameters.
        """
        pass


    def _execute(self, cmdStr, raw, *arglist, **argdict) :
        """\
          var _execute(string cmdStr, bool raw, *arglist, **argdict)
          
          Execute inner command line and return the shell status.
          Each line from stdout and stderr will be decoded and processed,
          i.e. lines will go through _decodeOutput() and _processOutput().
          
          param cmdStr                command as a string to execute.
          param raw                   disable output formatting
          param *arglist              extra parameters list.
          param **argdict             extra named parameters.
          return                      shell status
        """
        #~ self._processInput(cmdStr, raw, *arglist, **argdict)
        
        stdin, stdout = os.popen4(cmdStr)
        stdin.close()
        
        buffer = stdout.readline()
        while buffer :
          buffer = self._decodeOutput(buffer, raw)
          self._processOutput(buffer, raw, *arglist, **argdict)
          buffer = stdout.readline()
        
        return stdout.close()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _processInput(self, line, raw, *arglist, **argdict) :
        """\
          void _processInput(string line, bool raw, *arglist, **argdict)
         
          Process the command input line by line.
          
          param line                  line from command input
          param raw                   disable input formatting
          param *arglist              extra parameters list.
          param **argdict             extra named parameters.
        """
        if callable(self._onProcessInput) :
          self._onProcessInput(line, raw, *arglist, **argdict)


    def _processOutput(self, line, raw, *arglist, **argdict) :
        """\
          void _processOutput(string line, bool raw, *arglist, **argdict)
         
          Process the command output line by line. Records given string
          to the command output buffer.
          
          param line                  line from command output.
          param raw                   disable output formatting
          param *arglist              extra parameters list.
          param **argdict             extra named parameters.
        """
        if callable(self._onProcessOutput) :
          self._onProcessOutput(line, raw, *arglist, **argdict)
        self._output.append(line)


    def _decodeOutput(self, line, raw, *arglist, **argdict) :
        """\
          string _decodeOutput(string output, bool raw, *arglist, **argdict)
          
          Command output might need some decoding.
          Strip trailing whitespace characters from the given string.
          
          param line                  line from command output.
          param raw                   disable output formatting
          param *arglist              extra parameters list.
          param **argdict             extra named parameters.
        """
        return line.rstrip()


  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _getCmdLine(self) :
        """\
          object _getCmdLine(void)
          
          Getter - cmdLine property
        """
        return self._cmdLine


    def _getCmdTime(self) :
        """\
          object _getCmdTime(void)
          
          Getter - cmdTime property
          If the 'cmdTime' was not set (equals None), then the current
          local time is returned.
        """
        if self._cmdTime is None :
          self._cmdTime = time.localtime()
        return self._cmdTime


    def _getOutput(self) :
        """\
          list _getOutput(void)
          
          Getter - output property
        """
        return self._output


    def _getOnProcessInput(self) :
        """\
          var _getOnProcessInput(void)
          
          Getter - onProcessInput property
          Handler called upon each line of the command input.
        """
        return self._onProcessInput


    def _getOnProcessOutput(self) :
        """\
          var _getOnProcessOutput(void)
          
          Getter - onProcessOutput property
          Handler called upon each line of the command output.
        """
        return self._onProcessOutput


  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _setCmdLine(self, value) :
        """\
          void _setCmdLine(object value)
          
          Setter - cmdLine property
        """
        self._cmdLine.assign(value)


    def _setOnProcessInput(self, value) :
        """\
          void _setOnProcessInput(var value)
          
          Setter - onProcessInput property
          Handler called upon each line of the command input.
        """
        if not callable(value) :
          self._onProcessInput = None
        else :
          self._onProcessInput = value


    def _setOnProcessOutput(self, value) :
        """\
          void _setOnProcessOutput(var value)
          
          Setter - onProcessOutput property
          Handler called upon each line of the command output.
        """
        if not callable(value) :
          self._onProcessOutput = None
        else :
          self._onProcessOutput = value


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    cmdLine     = property(
        doc  = "command line to execute",
        fget = _getCmdLine,
        fset = _setCmdLine,
        fdel = None
      )

    cmdTime     = property(
        doc  = "command execution timestamp",
        fget = _getCmdTime,
        fset = None,
        fdel = None
      )

    output      = property(
        doc  = "retains the output from the last command ran",
        fget = _getOutput,
        fset = None,
        fdel = None
      )

    onProcessInput  = property(
        doc  = "Handler called upon each line of the command input.",
        fget = _getOnProcessInput,
        fset = _setOnProcessInput,
        fdel = None
      )

    onProcessOutput = property(
        doc  = "Handler called upon each line of the command output.",
        fget = _getOnProcessOutput,
        fset = _setOnProcessOutput,
        fdel = None
      )


  # /class CmdHandler - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -