#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
Define a ResBuilder task item used by the building utility. ResBuilder is
used to set the application version.

Require
  Python        2.2

Constants
  string        vrsTaskType

Classes
  class         vrsTaskItem(bool enabled)
                  extends py_netasq.building.core::taskItem
  class         vrsTaskList(bool enabled)
                  extends py_netasq.building.core::taskList
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.18 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  os
import  sys

from    py_netasq.commonlib     import asqString, asqTypes

from    py_netasq               import building as asqBuild
from    py_netasq.building      import core     as asqBuildCore


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

vrsTaskType = 'vrsTask'

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class vrsTask(asqBuildCore.task) :
    """\
      A versioning task: write application version infos to binary files.
    """

    # class task type, nuff said :)
    classTaskType = vrsTaskType

    # task type compatibility list
    compatibility = asqBuildCore.task.compatibility + (
        classTaskType,
      )

    # Properties defined for that task type and its descendant.
    # Note : attribute names should be "lower case" only.
    attributes    = (
        'taskname',             # task name (should held section name)
        'tasktype',             # task type as a string
        'enabled',              # is task active ?
        'workdir',              # task base directory
        'comments',             # about this task
        
        'modifier',             # path to resModifier application
        #~ 'directory',            # start recursion from directory
        'cmdextra',             # extra application parameters
        'companyname',          # app. info, CompanyName
        'legalcopyright',       # app. info, LegalCopyright
        'filedescription',      # app. info, FileDescription
        'fileversion',          # app. info, FileVersion
        'productversion',       # app. info, label (alpha, beta, rc...)
        
        'tmpfile',              # temp. file for constant values definitions
        'others'                # unrecognized attributes
      )

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def isValid(self) :
        """\
          bool isValid(void)
          
          Tells whether or not the task is "valid", id est, if mandatory
          attributes are set; an empty task is not valid, but an invalid
          task may not be empty ;)
          
          return                      is task valid
        """
        asqBuild.logger.trace('vrsTask.isValid()')
        asqBuild.logger.incDepthTrace()
        try :
          
          if not super(vrsTask, self).isValid() :
            return False
          else :
            return asqTypes.isNonEmptyInstance(self.modifier, basestring)
          
        finally :
          asqBuild.logger.decDepthTrace()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _checkFileVersion(self, status, name, label, vital=False) :
        """\
          void _checkFileVersion(object status, string name, string label, bool vital)
          
          Helper function used by _attributeCheck(). Provided to reduce
          redundancy. If "vital" is set to True, then ANY abnormal setting
          will produce errors instead of warnings.
          
          param status                checkResult object
          param name                  attribute name
          param label                 attribute formal name
          param vital                 is attribute setting vital ?
                                        defaults to False
        """
        buffer = self.__dict__[name]
        if buffer is None :
          if vital :
            status.addMsgError('%-15s %s is not set' % (name, label))
          else :
            status.addMsgWarning('%-15s %s is not set' % (name, label))
        elif not asqTypes.isNonEmptyInstance(buffer, basestring) :
          status.addMsgError('%-15s %s is invalid' % (name, label))
        else :
          try :
            self._binaryFileVersion(buffer)
          except ValueError, e :
            status.addMsgError('%-15s %s is invalid, fields must be numeric values' % (name, label))


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _attributeCheck(self, status, name) :
        """\
          void _attributeCheck(object status, string name)
          
          Callback used by self.check(...)
          Checks the value of attribute; Appends error or warning
          messages to given check status.
          Should be overriden to set the behavior for children classes.
          
          param status                checkResult object
          param name                  attribute name
        """
        if 'modifier' == name :
          self._checkFilePath(
              status, name, 'resModifier executable',
              vital=isinstance(self, asqBuildCore.taskItem)
            )
        elif 'fileversion' == name :
          self._checkFileVersion(status, name, 'FileVersion')
        elif 'companyname' == name :
          self._checkValue(status, name, 'CompanyName')
        elif 'legalcopyright' == name :
          self._checkValue(status, name, 'LegalCopyright')
        elif 'filedescription' == name :
          self._checkValue(status, name, 'FileDescription')
        elif 'productversion' == name :
          self._checkValue(status, name, 'ProductVersion')
        else :
          super(vrsTask, self)._attributeCheck(status, name)


    def _attributeIsPath(self, name) :
        """\
          bool _attributeIsPath(string name)
          
          Callback used by self.completePaths(...)
          Returns True if attribute is a path, please note that the
          'workdir' attribute should NOT be considered as a path here.
          Should be overriden to set the behavior for children classes.
          
          param name                  attribute name
          return                      is attribute a path
        """
        if name in ('modifier', ) :
          return True
        else :
          return super(vrsTask, self)._attributeIsPath(name)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _binaryFileVersion(self, version) :
        """\
          string _binaryFileVersion(string version)
          
          Return a binary friendly version from the given one; An exception
          may be raised if given 'version' can't be converted (because of
          invalid format).
          The binary FILEVERSION consists of two 32-bit integers, defined by
          four 16-bit integers; Each field of 'version' must be valid 16-bit
          integer values. Out of range values (e.g. '1042') will be converted
          modulus 256 (e.g. 1042 % 256 => 18).
          
          param version               FileVersion string
          return                      FILEVERSION string
        """
        try :
          chunks = version.split('.')[:4]
          chunks = map(lambda s: str(int(s) % 256), chunks)
        except ValueError, e :
          buffer = "fields must be numeric values [%s]" % (repr(version), )
          # add previous traceback to exception
          raise ValueError(buffer), None, sys.exc_info()[2]
        else :
          return ','.join(chunks)


  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # /class vrsTask  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


class vrsTaskItem(vrsTask, asqBuildCore.taskItem) :
    """\
      A versioning task: write application version infos to binary files.
    """

    ant_name    = 'resModifier'

    # Properties defined for that task type and its descendant.
    # Note : attribute names should be "lower case" only.
    attributes    = (
        'taskname',             # task name (should held section name)
        'tasktype',             # task type as a string
        'enabled',              # is task active ?
        'workdir',              # task base directory
        'comments',             # about this task
        
        'modifier',             # path to resModifier application
        'directory',            # start recursion from directory
        'cmdextra',             # extra application parameters
        'companyname',          # app. info, CompanyName
        'legalcopyright',       # app. info, LegalCopyright
        'filedescription',      # app. info, FileDescription
        'fileversion',          # app. info, FileVersion
        'productversion',       # app. info, label (alpha, beta, rc...)
        
        'tmpfile',              # temp. file for constant values definitions
        'others'                # unrecognized attributes
      )

    # properties that could be overwritten by a "Merge" method
    mergeable     = (
        'workdir',
        'modifier',
        'cmdextra',
        'companyname',
        'legalcopyright',
        'filedescription',
        'fileversion',
        'productversion',
      )

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def isValid(self) :
        """\
          bool isValid(void)
          
          Tells whether or not the task is "valid", id est, if mandatory
          attributes are set; an empty task is not valid, but an invalid
          task may not be empty ;)
          
          return                      is task valid
        """
        asqBuild.logger.trace('vrsTaskItem.isValid()')
        asqBuild.logger.incDepthTrace()
        try :
          
          if not super(vrsTaskItem, self).isValid() :
            return False
          else :
            return asqTypes.isNonEmptyInstance(self.directory, basestring)
          
        finally :
          asqBuild.logger.decDepthTrace()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _attributeCheck(self, status, name) :
        """\
          void _attributeCheck(object status, string name)
          
          Callback used by self.check(...)
          Checks the value of attribute; Appends error or warning
          messages to given check status.
          Should be overriden to set the behavior for children classes.
          
          param status                checkResult object
          param name                  attribute name
        """
        if 'directory' == name :
          self._checkDirPath(status, name, 'root directory', True)
        else :
          super(vrsTaskItem, self)._attributeCheck(status, name)


    def _attributeIsPath(self, name) :
        """\
          bool _attributeIsPath(string name)
          
          Callback used by self.completePaths(...)
          Returns True if attribute is a path, please note that the
          'workdir' attribute should NOT be considered as a path here.
          Should be overriden to set the behavior for children classes.
          
          param name                  attribute name
          return                      is attribute a path
        """
        if name in ('directory', ) :
          return True
        else :
          return super(vrsTaskItem, self)._attributeIsPath(name)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _antAction(self) :
        """\
          str _antAction(void)
          
          Callback used by antDisplay()
          Returns a short string explaining the task's action.
          Should be overriden to set the behavior for children classes.
          Disabled and non valid tasks are already handled.
          
          return                      task's action
        """
        result = super(vrsTaskItem, self)._antAction()
        
        # test if "result" is an empty string
        if not asqTypes.isNonEmptyInstance(result, basestring) :
          chunks = list()
          buffer = asqString.shortenString(self.directory, 36, '...', '\\')
          
          chunks.append('versioning')
          chunks.append('`%s`' % (buffer, ))
          result = ' '.join(chunks)
        
        return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _fillCmdLine(self, aCmdLine) :
        """\
          void _fillCmdLine(object aCmdLine)
          
          Callback used by toCmdLine()
          Build the task command line, task is assumed valid.
          Should be overriden to set the behavior for children classes.
          
          param aCmdLine              cmdLine object
        """
        asqBuild.logger.trace('vrsTaskItem._fillCmdLine(aCmdLine)')
        asqBuild.logger.incDepthTrace()
        try :
          
          # path to resModifier.exe
          aCmdLine.appendQuoted(self.modifier)
          # start recursion from
          aCmdLine.appendQuoted(self.directory)
          # config file, set somme neede constants
          if asqTypes.isNonEmptyInstance(self.tmpfile, basestring) :
            aCmdLine.append('-v')
            aCmdLine.appendQuoted(self.tmpfile)
          # res Modifier extra params
          if asqTypes.isNonEmptyInstance(self.cmdextra, basestring) :
            aCmdLine.append(self.cmdextra)
          
        finally :
          asqBuild.logger.decDepthTrace()


    def _fillTmpFile(self, aFile) :
        """\
          void _fillTmpFile(object aFile)
          
          Callback used by buildTmpFile()
          Dumps config values to given file object, which must have
          been opened.
          Should be overriden to set the behavior for children classes.
          
          param aFile                 file object
        """
        if asqTypes.isNonEmptyInstance(self.fileversion, basestring) :
          binary = self._binaryFileVersion(self.fileversion)
          aFile.write('FILEVERSION=%s\n'     % (binary, ))
          aFile.write('PRODUCTVERSION=%s\n'  % (binary, ))
          
        if asqTypes.isNonEmptyInstance(self.companyname, basestring) :
          aFile.write('CompanyName=%s\n'     % (self.companyname, ))
        else :
          aFile.write('CompanyName= \n')
          
        if asqTypes.isNonEmptyInstance(self.legalcopyright, basestring) :
          aFile.write('LegalCopyright=%s\n'  % (self.legalcopyright, ))
        else :
          aFile.write('LegalCopyright= \n')
          
        if asqTypes.isNonEmptyInstance(self.filedescription, basestring) :
          aFile.write('FileDescription=%s\n' % (self.filedescription, ))
        else :
          aFile.write('FileDescription= \n')
          
        if asqTypes.isNonEmptyInstance(self.fileversion, basestring) :
          aFile.write('FileVersion=%s\n'     % (self.fileversion, ))
        else :
          aFile.write('FileVersion= \n')
          
        if asqTypes.isNonEmptyInstance(self.productversion, basestring) :
          aFile.write('ProductVersion=%s\n'  % (self.productversion, ))
        else :
          aFile.write('ProductVersion= \n')


  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # /class vrsTaskItem  - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


class vrsTaskList(vrsTask, asqBuildCore.taskList) :
    """\
      A list of vrsTaskItems, the list defines "default" attributes that
      each single item can override; if an item doesn't set an attribute,
      then the list's values will be used.
    """

    # task item class
    taskClass   = vrsTaskItem

    # Properties defined for that task type and its descendant.
    # Note : attribute names should be "lower case" only.
    attributes    = (
        'taskname',             # task name (should held section name)
        'count',                # task item count
        'tasktype',             # task type as a string
        'enabled',              # is task active ?
        'workdir',              # task base directory
        'comments',             # about this task
        
        'modifier',             # path to resModifier application
        #~ 'directory',            # start recursion from directory
        'cmdextra',             # extra application parameters
        'companyname',          # app. info, CompanyName
        'legalcopyright',       # app. info, LegalCopyright
        'filedescription',      # app. info, FileDescription
        'fileversion',          # app. info, FileVersion
        'productversion',       # app. info, label (alpha, beta, rc...)
        
        'tmpfile',              # temp. file for constant values definitions
        'others'                # unrecognized attributes
      )

  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # /class vrsTaskList  - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -