#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
Define a "Subversion Task item" list used by the building utility.

Require
  Python        2.2

Constants
  string        svnTaskType

Classes
  class         svnTaskItem(bool enabled)
                  extends py_netasq.building.core::taskItem
  class         svnTaskList(bool enabled)
                  extends py_netasq.building.core::taskList
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.2 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  os
import  re

from    py_netasq.commonlib     import asqSequence, asqString, asqTypes

from    py_netasq               import building as asqBuild
from    py_netasq.building      import core     as asqBuildCore


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

svnTaskType = 'svnTask'

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class svnTask(asqBuildCore.task) :
    """\
      A Subversion task: a svn command, which can be either 'checkout',
      'export' or 'update'.
      Please note that a svn checkout can be performed on more than one
      source URL AT ONCE.
    """

    # class task type, nuff said :)
    classTaskType = svnTaskType

    # task type compatibility list
    compatibility = asqBuildCore.task.compatibility + (
        classTaskType,
      )

    # Properties defined for that task type and its descendant.
    # Note : attribute names should be "lower case" only.
    attributes    = (
        'taskname',             # task name (should held section name)
        'tasktype',             # task type as a string
        'enabled',              # is task active ?
        'workdir',              # task base directory
        'comments',             # about this task
        
        'client',               # subversion client (exe) to use
        #~ 'urllist',              # dynamic URL list of the item(s) in repository
        'command',              # svn command to perform
        'cmdextra',             # extra svn CLIENT parameters
        'destdir',              # retrieved files destination
        'revision',             # revision to get
        
        'others'                # unrecognized attributes
      )

    # svn tasks can only RETRIEVE files
    allowed_commands = ('checkout', 'export', 'update')

    # recognize url list items : "urlXX"
    reUrlItem     = re.compile(r'^url(?P<urlId>\d+)$', re.IGNORECASE)

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def isValid(self) :
        """\
          bool isValid(void)
          
          Tells whether or not the task is "valid", id est, if mandatory
          attributes are set; an empty taskItem is not valid, but an
          invalid taskItem may not be empty ;)
          
          return                      is task valid
        """
        asqBuild.logger.trace('svnTask.isValid()')
        asqBuild.logger.incDepthTrace()
        try :
          
          if not super(svnTask, self).isValid() :
            return False
          elif not asqTypes.isNonEmptyInstance(self.client, basestring) :
            return False
          elif not asqTypes.isNonEmptyInstance(self.command, basestring) :
            return False
          else :
            return (self.command in self.allowed_commands)
          
        finally :
          asqBuild.logger.decDepthTrace()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _checkCommand(self, status, name, label, vital=False) :
        """\
          void _checkCommand(object status, string name, string label, bool vital)
          
          Helper function used by _attributeCheck(). Provided to reduce
          redundancy. If "vital" is set to True, then ANY abnormal setting
          will produce errors instead of warnings.
          
          param status                checkResult object
          param name                  attribute name
          param label                 attribute formal name
          param vital                 is attribute setting vital ?
                                        defaults to False
        """
        buffer = self.__dict__[name]
        if buffer is None :
          if vital :
            status.addMsgError('%-15s %s is not set'   % (name, label))
          else :
            status.addMsgWarning('%-15s %s is not set'   % (name, label))
        elif not asqTypes.isNonEmptyInstance(buffer, basestring) :
          status.addMsgError('%-15s %s is invalid'  % (name, label))
        elif buffer not in self.allowed_commands :
          status.addMsgError('%-15s %s not allowed' % (name, label))


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _attributeCheck(self, status, name) :
        """\
          void _attributeCheck(object status, string name)
          
          Callback used by self.check(...)
          Checks the value of attribute; Appends error or warning
          messages to given check status.
          Should be overriden to set the behavior for children classes.
          
          param status                checkResult object
          param name                  attribute name
        """
        if 'client' == name :
          self._checkFilePath(
              status, name, 'svn client executable',
              vital=isinstance(self, asqBuildCore.taskItem)
            )
        elif 'command' == name :
          self._checkCommand(
              status, name, 'svn command',
              vital=isinstance(self, asqBuildCore.taskItem)
            )
        elif 'destdir' == name :
          self._checkDirPath(status, name, 'destination directory')
        else :
          super(svnTask, self)._attributeCheck(status, name)


    def _attributeIsPath(self, name) :
        """\
          bool _attributeIsPath(string name)
          
          Callback used by self.completePaths(...)
          Returns True if attribute is a path, please note that the
          'workdir' attribute should NOT be considered as a path here.
          Should be overriden to set the behavior for children classes.
          
          param name                  attribute name
          return                      is attribute a path
        """
        if name in ('client', 'destdir', ) :
          return True
        else :
          return super(svnTask, self)._attributeIsPath(name)


  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # /class svnTask  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


class svnTaskItem(svnTask, asqBuildCore.taskItem) :
    """\
      A svn task: a svn command, which can be either 'checkout',
      'export' or 'update'.
      Please note that a svn checkout can be performed on more than one
      source URL AT ONCE.
      
      property int      urllen
    """

    ant_name    = 'svn'

    # Properties defined for that task type and its descendant.
    # Note : attribute names should be "lower case" only.
    attributes  = (
        'taskname',             # task name (should held section name)
        'tasktype',             # task type as a string
        'enabled',              # is task active ?
        'workdir',              # task base directory
        'comments',             # about this task
        
        'client',               # subversion client (exe) to use
        'urllist',              # dynamic URL list of the item(s) in repository
        'command',              # svn command to perform
        'cmdextra',             # extra svn CLIENT parameters
        'destdir',              # retrieved files destination
        'revision',             # revision to get
        
        'others'                # unrecognized attributes
      )

    # properties that could be overwritten by a "Merge" method
    mergeable   = (
        'workdir',
        'client',
        'command',
        'cmdextra',
        'destdir',
        'revision'
      )

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __getattr__(self, name) :
        """\
          var __getattr__(string name)
         
          Special method - Called when an attribute lookup has not found
          the attribute in the usual places (i.e. it is not an instance
          attribute nor is it found in the class tree for self). name is
          the attribute name. This method should return the (computed)
          attribute value or raise an AttributeError exception.
          
          Handles 'urlXX' attributes, similary as __setattr__.
          
          param name                  requested attribute
          return                      attribute value
        """
        asqBuild.logger.trace('svnTaskItem.__getattr__(%s)', name)
        asqBuild.logger.incDepthTrace()
        try :
          
          if 'url' == name :
            # may raise an "out of bound" exception
            return self.urllist[0]
          
          # recognize urlXX attributes
          matchd = self.reUrlItem.search(name)
          if matchd is None :
            object.__getattr__(self, name)
          else :
            offset = int(matchd.group('urlId'))
            # may raise an "out of bound" exception
            return self.urllist[offset]
          
        finally :
          asqBuild.logger.decDepthTrace()


    def __setattr__(self, name, value) :
        """\
          void __setattr__(string name, var value)
          
          Special method - Called when an attribute assignment is attempted.
          This is called instead of the normal mechanism (i.e. store the
          value in the instance dictionary). name is the attribute name,
          value is the value to be assigned to it.
          
          If name is not found in defined attributes list, and is not a
          mangled name (_classname__name), then (name, value) is added to
          the object's "others" dictionnary.
          
          Each time an attribute value is set, 'normalize()' should be
          called to ensure consistency.
          
          param name                  attribute to set
          param value                 new value
        """
        asqBuild.logger.trace('svnTaskItem.__setattr__(%s, %s)', name, repr(value))
        asqBuild.logger.incDepthTrace()
        try :
          
          if 'urllen' == name :
            # pad self.urllist to at least match given length
            buffer = asqTypes.asInteger(value, radix=0, default=0)
            asqSequence.stuffSequence(self.urllist, buffer, None)
          elif 'url' == name :
            # `url`is an alias for url0 <=> urllist[0]
            asqSequence.stuffSequence(self.urllist, 1, None)
            self.urllist[0] = value
          else :
            # recognize urlXX attributes
            matchd = self.reUrlItem.search(name)
            if matchd is None :
              object.__setattr__(self, name, value)
            else :
              offset = int(matchd.group('urlId'))
              # ensuring that offset is in range
              asqSequence.stuffSequence(self.urllist, 1+offset, None)
              self.urllist[offset] = value
          
        finally :
          asqBuild.logger.decDepthTrace()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def isValid(self) :
        """\
          bool isValid(void)
          
          Tells whether or not the task is "valid", id est, if mandatory
          attributes are set; an empty taskItem is not valid, but an
          invalid taskItem may not be empty ;)
          
          return                      is task valid
        """
        asqBuild.logger.trace('svnTaskItem.isValid()')
        asqBuild.logger.incDepthTrace()
        try :
          
          if not super(svnTaskItem, self).isValid() :
            return False
          elif 'update' == self.command :
            pass # svn update : no need to check urllist
          elif not isinstance(self.urllist, list) :
            return False
          elif not filter(None, self.urllist) :
            return False
          
          return True
          
        finally :
          asqBuild.logger.decDepthTrace()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _checkUrlList(self, status, name, label, vital=False) :
        """\
          void _checkUrlList(object status, string name, string label, bool vital)
          
          Helper function used by _attributeCheck(). Provided to reduce
          redundancy. If "vital" is set to True, then ANY abnormal setting
          will produce errors instead of warnings.
          
          param status                checkResult object
          param name                  attribute name
          param label                 attribute formal name
          param vital                 is attribute setting vital ?
                                        defaults to False
        """
        buffer = self.__dict__[name]
        if buffer is None :
          if vital :
            status.addMsgError('%-15s %s is not set' % (name, label))
          else :
            status.addMsgWarning('%-15s %s is not set' % (name, label))
        elif not isinstance(buffer, list) :
          status.addMsgError('%-15s %s is invalid' % (name, label))
        elif not filter(None, buffer) : # empty list
          status.addMsgWarning('%-15s %s is empty' % (name, label))


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _attributeCheck(self, status, name) :
        """\
          void _attributeCheck(object status, string name)
          
          Callback used by self.check(...)
          Checks the value of attribute; Appends error or warning
          messages to given check status.
          Should be overriden to set the behavior for children classes.
          
          param status                checkResult object
          param name                  attribute name
        """
        if 'urllist' != name :
          super(svnTaskItem, self)._attributeCheck(status, name)
        else :
          self._checkUrlList(status, name, 'svn repository url(s)', vital=True)


    def _attributeToList(self, aList, name) :
        """\
          void _attributeToList(list aList, string name)
          
          Callback used by self.toList(...)
          Appends attribute to given "aList" list.
          Should be overriden to set the behavior for children classes.
          
          param aList                 list of attributes
          param name                  attribute name
        """
        if 'urllist' != name :
          super(svnTaskItem, self)._attributeToList(aList, name)
        else :
          buffer = filter(None, self.urllist)
          if 1 == len(buffer) :
            # use `url` alias
            aList.append( ('url', buffer[0]) )
          else :
            aList.append( ('urllen', self.urllen) )
            for i in xrange(self.urllen) :
              aList.append( ('url%d' % (i,), self.urllist[i]) )


    def _attributeReset(self, name) :
        """\
          void _attributeReset(string name)
          
          Callback used by self.reset()
          Resets attribute value.
          Should be overriden to set the behavior for children classes.
          
          param name                  attribute name
        """
        if 'urllist' != name :
          super(svnTaskItem, self)._attributeReset(name)
        elif not hasattr(self, 'urllist') :
          object.__setattr__(self, 'urllist', list())
        else :
          self.urllist[:] = []


    def _attributeNonZero(self, name) :
        """\
          bool _attributeNonZero(string name)
          
          Callback used by self.__nonzero__()
          Check that attribute is not empty.
          Should be overriden to set the behavior for children classes.
          
          param name                  attribute name
          return                      is attribute NOT empty ?
        """
        if 'urllist' != name :
          return super(svnTaskItem, self)._attributeNonZero(name)
        else :
          return bool(self.urllist)


    def _attributeDebug(self, name) :
        """\
          void _attributeDebug(string name)
          
          Callback used by self.debug()
          Prints out debug information for given attribute.
          Should be overriden to set the behavior for children classes.
          
          param name                  attribute name
        """
        if 'urllist' != name :
          super(svnTaskItem, self)._attributeDebug(name)
        elif not self.urllist :
          # url list seems empty
          self._debugKeyVal(name, self.__dict__[name])
        else :
          asqBuild.logger.debug(name)
          asqBuild.logger.incDepthDebug()
          for i in xrange(len(self.urllist)) :
            self._debugKeyVal('url%d' % (i,), self.urllist[i])
          asqBuild.logger.decDepthDebug()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _attributeMerge(self, name, value) :
        """\
          void _attributeMerge(string name, var value)
          
          Callback used by mergeWithList(...)
          Merge attribut value with given one. The url list should NOT be
          merged !
          Should be overriden to set the behavior for children classes.
          
          param name                  attribute name
          param value                 attribute value
        """
        if 'urllen' == name :
          pass
        elif 'url' == name :
          pass
        elif self.reUrlItem.match(name) :
          pass
        else :
          super(svnTaskItem, self)._attributeMerge(name, value)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _antAction(self) :
        """\
          str _antAction(void)
          
          Callback used by antDisplay()
          Returns a short string explaining the task's action.
          Should be overriden to set the behavior for children classes.
          Disabled and non valid tasks are already handled.
          
          return                      task's action
        """
        result = super(svnTaskItem, self)._antAction()
        
        # test if "result" is an empty string
        # if so, then the antAction string is built
        if asqTypes.isNonEmptyInstance(result, basestring) :
          pass
        elif 'update' == self.command :
          chunks = list()
        
          chunks.append(self.command)
          buffer = asqString.shortenString(self.workdir, 42, '...', '\\')
          chunks.append(buffer)
          result = ' '.join(chunks)
        else :
          result = list()
          chunks = list()
          
          for urlItem in self.urllist :
            chunks[:] = []
            chunks.append(self.command)
            buffer = asqString.shortenString(urlItem, 42, '...', '/')
            chunks.append(buffer)
            if asqTypes.isNonEmptyInstance(self.revision, basestring) :
              chunks.append('-r')
              chunks.append(self.revision)
            result.append( ' '.join(chunks) )
          
        return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _fillCmdLine(self, aCmdLine) :
        """\
          void _fillCmdLine(object aCmdLine)
          
          Callback used by toCmdLine()
          Build the task command line, task is assumed valid.
          Should be overriden to set the behavior for children classes.
          
          param aCmdLine              cmdLine object
        """
        asqBuild.logger.trace('svnTaskItem._fillCmdLine(aCmdLine)')
        asqBuild.logger.incDepthTrace()
        try :
          
          # svn executable
          aCmdLine.appendQuoted(self.client)
          if asqTypes.isNonEmptyInstance(self.cmdextra, basestring) :
            # svn extra parameters
            aCmdLine.append(self.cmdextra)
          # svn command
          aCmdLine.append(self.command)
          if asqTypes.isNonEmptyInstance(self.revision, basestring) :
            aCmdLine.append('-r')
            aCmdLine.append(self.revision)
          
          if 'update' != self.command :
            for item in filter(None, self.urllist) :
              aCmdLine.appendQuoted(item.strip())
            if asqTypes.isNonEmptyInstance(self.destdir, basestring) :
              aCmdLine.appendQuoted(self.destdir)
          
        finally :
          asqBuild.logger.decDepthTrace()


  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _getUrlLen(self) :
        """\
          int _getUrlLen(void)
          
          Getter - urllen property
        """
        return len(self.urllist)


  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    urllen    = property(
                  doc  = "length of dynamic repository url list",
                  fget = _getUrlLen,
                  fset = None,
                  fdel = None )


  # /class svnTaskItem  - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


class svnTaskList(svnTask, asqBuildCore.taskList) :
    """\
      A list of svnTaskItems, the list defines "default" attributes that
      each single item can override; if an item doesn't set an attribute,
      then the list's values will be used.
    """

    # task item class
    taskClass   = svnTaskItem

    # Properties defined for that task type and its descendant.
    # Note : attribute names should be "lower case" only.
    attributes    = (
        'taskname',             # task name (should held section name)
        'count',                # task item count
        'tasktype',             # task type as a string
        'enabled',              # is task active ?
        'workdir',              # task base directory
        'comments',             # about this task
        
        'client',               # subversion client (exe) to use
        #~ 'urllist',              # dynamic URL list of the item(s) in repository
        'command',              # svn command to perform
        'cmdextra',             # extra svn CLIENT parameters
        'destdir',              # retrieved files destination
        'revision',             # revision to get
        
        'others'                # unrecognized attributes
      )

  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # /class svnTaskList  - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -