#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
Define a Indigo Rose Setup Factory task item used by the
building utility.

Require
  Python        2.2

Constants
  string        sf6TaskType

Classes
  class         sf6TaskItem(bool enabled)
                  extends py_netasq.building.core::taskItem
  class         sf6TaskList(bool enabled)
                  extends py_netasq.building.core::taskList
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.12 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  os

from    py_netasq.commonlib     import asqString, asqTypes

from    py_netasq               import building as asqBuild
from    py_netasq.building      import core     as asqBuildCore


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

sf6TaskType = 'sf6Task'

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class sf6Task(asqBuildCore.task) :
    """\
      A setup Factory task: create a "setup" executable for
      an application.
    """

    # class task type, nuff said :)
    classTaskType = sf6TaskType

    # task type compatibility list
    compatibility = asqBuildCore.task.compatibility + (
        classTaskType,
      )

    # Properties defined for that task type and its descendant.
    # Note : attribute names should be "lower case" only.
    attributes    = (
        'taskname',             # task name (should held section name)
        'tasktype',             # task type as a string
        'enabled',              # is task active ?
        'workdir',              # task base directory
        'comments',             # about this task
        
        'factory',              # path to setup factory design app
        #~ 'project',              # path to setup Factory project (*.sf6)
        'cmdextra',             # extra application parameters
        
        'tmpfile',              # temp. file for constant values definitions
        'others'                # unrecognized attributes
      )

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def isValid(self) :
        """\
          bool isValid(void)
          
          Tells whether or not the task is "valid", id est, if mandatory
          attributes are set; an empty task is not valid, but an invalid
          task may not be empty ;)
          
          return                      is task valid
        """
        asqBuild.logger.trace('sf6Task.isValid()')
        asqBuild.logger.incDepthTrace()
        try :
          
          if not super(sf6Task, self).isValid() :
            return False
          else :
            return asqTypes.isNonEmptyInstance(self.factory, basestring)
          
        finally :
          asqBuild.logger.decDepthTrace()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _attributeCheck(self, status, name) :
        """\
          void _attributeCheck(object status, string name)
          
          Callback used by self.check(...)
          Checks the value of attribute; Appends error or warning
          messages to given check status.
          Should be overriden to set the behavior for children classes.
          
          param status                checkResult object
          param name                  attribute name
        """
        if 'factory' == name :
          self._checkFilePath(
              status, name, 'setup factory executable',
              vital=isinstance(self, asqBuildCore.taskItem)
            )
        else :
          super(sf6Task, self)._attributeCheck(status, name)


    def _attributeIsPath(self, name) :
        """\
          bool _attributeIsPath(string name)
          
          Callback used by self.completePaths(...)
          Returns True if attribute is a path, please note that the
          'workdir' attribute should NOT be considered as a path here.
          Should be overriden to set the behavior for children classes.
          
          param name                  attribute name
          return                      is attribute a path
        """
        if name in ('factory', ) :
          return True
        else :
          return super(sf6Task, self)._attributeIsPath(name)


  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # /class sf6Task  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


class sf6TaskItem(sf6Task, asqBuildCore.taskItem) :
    """\
      A setup Factory task: create a "setup" executable for
      an application.
    """
    
    ant_name = 'sf6Design'
    
    # Properties defined for that task type and its descendant.
    # Note : attribute names should be "lower case" only.
    attributes  = (
        'taskname',             # task name (should held section name)
        'tasktype',             # task type as a string
        'enabled',              # is task active ?
        'workdir',              # task base directory
        'comments',             # about this task
        
        'factory',              # path to setup factory design app
        'project',              # path to setup Factory project (*.sf6)
        'cmdextra',             # extra application parameters
        
        'tmpfile',              # temp. file for constant values definitions
        'others'                # unrecognized attributes
      )

    # properties that could be overwritten by a "Merge" method
    mergeable   = (
        'workdir',
        'factory',
        'cmdextra',
      )

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def isValid(self) :
        """\
          bool isValid(void)
          
          Tells whether or not the task is "valid", id est, if mandatory
          attributes are set; an empty task is not valid, but an invalid
          task may not be empty ;)
          
          return                      is task valid
        """
        asqBuild.logger.trace('sf6TaskItem.isValid()')
        asqBuild.logger.incDepthTrace()
        try :
          
          if not super(sf6TaskItem, self).isValid() :
            return False
          else :
            return asqTypes.isNonEmptyInstance(self.project, basestring)
          
        finally :
          asqBuild.logger.decDepthTrace()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _attributeCheck(self, status, name) :
        """\
          void _attributeCheck(object status, string name)
          
          Callback used by self.check(...)
          Checks the value of attribute; Appends error or warning
          messages to given check status.
          Should be overriden to set the behavior for children classes.
          
          param status                checkResult object
          param name                  attribute name
        """
        if 'project' == name :
          self._checkFilePath(status, name, 'setup factory project file', vital=True)
        else :
          super(sf6TaskItem, self)._attributeCheck(status, name)


    def _attributeIsPath(self, name) :
        """\
          bool _attributeIsPath(string name)
          
          Callback used by self.completePaths(...)
          Returns True if attribute is a path, please note that the
          'workdir' attribute should NOT be considered as a path here.
          Should be overriden to set the behavior for children classes.
          
          param name                  attribute name
          return                      is attribute a path
        """
        if name in ('project', ) :
          return True
        else :
          return super(sf6TaskItem, self)._attributeIsPath(name)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _antAction(self) :
        """\
          str _antAction(void)
          
          Callback used by antDisplay()
          Returns a short string explaining the task's action.
          Should be overriden to set the behavior for children classes.
          Disabled and non valid tasks are already handled.
          
          return                      task's action
        """
        result = super(sf6TaskItem, self)._antAction()
        
        # test if "result" is an empty string
        if not asqTypes.isNonEmptyInstance(result, basestring) :
          chunks = list()
          buffer = asqString.shortenString(self.project, 32, '...', '\\')
          
          chunks.append('processing')
          chunks.append('`%s`' % (buffer, ))
          result = ' '.join(chunks)
        
        return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _fillCmdLine(self, aCmdLine) :
        """\
          void _fillCmdLine(object aCmdLine)
          
          Callback used by toCmdLine()
          Build the task command line, task is assumed valid.
          Should be overriden to set the behavior for children classes.
          
          param aCmdLine              cmdLine object
        """
        asqBuild.logger.trace('sf6TaskItem._fillCmdLine(aCmdLine)')
        asqBuild.logger.incDepthTrace()
        try :
          
          # path to SUF60Design.exe
          aCmdLine.appendQuoted(self.factory)
          # project file *.sf6 to process
          aCmdLine.appendQuoted(self.project)
          # config file, set some needed constants
          if self.tmpfile is not None :
            aCmdLine.appendQuoted('/B:%s' % (self.tmpfile, ))
          # set up factory extra params
          if self.cmdextra is not None :
            aCmdLine.append(self.cmdextra)
          
        finally :
          asqBuild.logger.decDepthTrace()


    def _fillTmpFile(self, aFile) :
        """\
          void _fillTmpFile(object aFile)
          
          Callback used by buildTmpFile()
          Dumps config values to given file object, which must have
          been opened.
          Should be overriden to set the behavior for children classes.
          
          param aFile                 file object
        """
        def _isSf6Constant(value) :
          return 2 < len(value) and ('$' == value[0] == value[-1])
        
        keylst = [ x for x in self.others.iterkeys() if _isSf6Constant(x) ]
        
        aFile.write('[Constants]\n')
        for key in keylst :
          # value MUST NOT BE SURROUNDED by quotes !!
          val = asqString.unQuoteString(self.others[key], '"', '\\')
          # remove surrounding '$' chars
          key = asqString.unQuoteString(key, '$', '')
          aFile.write('#%s# = %s\n' % (key, val))


  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # /class sf6TaskItem  - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


class sf6TaskList(sf6Task, asqBuildCore.taskList) :
    """\
      A list of sf6TaskItems, the list defines "default" attributes that
      each single item can override; if an item doesn't set an attribute,
      then the list's values will be used.
    """

    # task item class
    taskClass   = sf6TaskItem

    # Properties defined for that task type and its descendant.
    # Note : attribute names should be "lower case" only.
    attributes    = (
        'taskname',             # task name (should held section name)
        'count',                # task item count
        'tasktype',             # task type as a string
        'enabled',              # is task active ?
        'workdir',              # task base directory
        'comments',             # about this task
        
        'factory',              # path to setup factory design app
        #~ 'project',              # path to setup Factory project (*.sf6)
        'cmdextra',             # extra application parameters
        
        'tmpfile',              # temp. file for constant values definitions
        'others'                # unrecognized attributes
      )

  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # /class sf6TaskList  - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -