#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
Define a ResManager task item used by the building utility.

Require
  Python        2.2

Constants
  string        resTaskType

Classes
  class         resTaskItem(bool enabled)
                  extends py_netasq.building.core::taskItem
  class         resTaskList(bool enabled)
                  extends py_netasq.building.core::taskList
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.11 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  os

from    py_netasq.commonlib     import asqString, asqTypes

from    py_netasq               import building as asqBuild
from    py_netasq.building      import core     as asqBuildCore


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

resTaskType = 'resTask'

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class resTask(asqBuildCore.task) :
    """\
      A ResManager task: compile images ressources files.
    """

    # class task type, nuff said :)
    classTaskType = resTaskType

    # task type compatibility list
    compatibility = asqBuildCore.task.compatibility + (
        classTaskType,
      )

    # Properties defined for that task type and its descendant.
    # Note : attribute names should be "lower case" only.
    attributes    = (
        'taskname',             # task name (should held section name)
        'tasktype',             # task type as a string
        'enabled',              # is task active ?
        'workdir',              # task base directory
        'comments',             # about this task
        
        'manager',              # path to resManager executable
        #~ 'project',              # path to resManager project (*.rmc)
        'cmdextra',             # extra application parameters
        'inifile',              # resManager application options (*.ini)
        'resdir',               # path to image directory
        'srcdir',               # ???
        
        'others'                # unrecognized attributes
      )

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _attributeCheck(self, status, name) :
        """\
          void _attributeCheck(object status, string name)
          
          Callback used by self.check(...)
          Checks the value of attribute; Appends error or warning
          messages to given check status.
          Should be overriden to set the behavior for children classes.
          
          param status                checkResult object
          param name                  attribute name
        """
        if 'manager' == name :
          self._checkFilePath(
              status, name, 'resManager executable',
              vital=isinstance(self, asqBuildCore.taskItem)
            )
        elif 'resdir' == name :
          self._checkDirPath(status, name, 'image directory')
        elif 'srcdir' == name :
          self._checkDirPath(status, name, 'directory')
        else :
          super(resTask, self)._attributeCheck(status, name)


    def _attributeIsPath(self, name) :
        """\
          bool _attributeIsPath(string name)
          
          Callback used by self.completePaths(...)
          Returns True if attribute is a path, please note that the
          'workdir' attribute should NOT be considered as a path here.
          Should be overriden to set the behavior for children classes.
          
          param name                  attribute name
          return                      is attribute a path
        """
        if name in ('manager', 'inifile', 'srcdir', 'resdir') :
          return True
        else :
          return super(resTask, self)._attributeIsPath(name)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def isValid(self) :
        """\
          bool isValid(void)
          
          Tells whether or not the task is "valid", id est, if mandatory
          attributes are set; an empty task is not valid, but an invalid
          task may not be empty ;)
          
          return                      is task valid
        """
        asqBuild.logger.trace('resTask.isValid()')
        asqBuild.logger.incDepthTrace()
        try :
          
          if not super(resTask, self).isValid() :
            return False
          else :
            return asqTypes.isNonEmptyInstance(self.manager, basestring)
          
        finally :
          asqBuild.logger.decDepthTrace()


  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # /class resTask  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


class resTaskItem(resTask, asqBuildCore.taskItem) :
    """\
      A ResManager task: compile images ressources files.
    """
    
    ant_name    = 'resManager'
    
    # Properties defined for that task type and its descendant.
    # Note : attribute names should be "lower case" only.
    attributes  = (
        'taskname',             # task name (should held section name)
        'tasktype',             # task type as a string
        'enabled',              # is task active ?
        'workdir',              # task base directory
        'comments',             # about this task
        
        'manager',              # path to resManager executable
        'project',              # path to resManager project (*.rmc)
        'cmdextra',             # extra application parameters
        'inifile',              # resManager application options (*.ini)
        'resdir',               # path to image directory
        'srcdir',               # ???
        
        'others'                # unrecognized attributes
      )

    # properties that could be overwritten by a "Merge" method
    mergeable   = (
        'workdir',
        'manager',
        'cmdextra',
        'inifile',
        'resdir',
        'srcdir',
      )

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def isValid(self) :
        """\
          bool isValid(void)
          
          Tells whether or not the task is "valid", id est, if mandatory
          attributes are set; an empty task is not valid, but an invalid
          task may not be empty ;)
          
          return                      is task valid
        """
        asqBuild.logger.trace('resTaskItem.isValid()')
        asqBuild.logger.incDepthTrace()
        try :
          
          if not super(resTaskItem, self).isValid() :
            return False
          else :
            return asqTypes.isNonEmptyInstance(self.project, basestring)
          
        finally :
          asqBuild.logger.decDepthTrace()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _attributeCheck(self, status, name) :
        """\
          void _attributeCheck(object status, string name)
          
          Callback used by self.check(...)
          Checks the value of attribute; Appends error or warning
          messages to given check status.
          Should be overriden to set the behavior for children classes.
          
          param status                checkResult object
          param name                  attribute name
        """
        if 'project' == name :
          self._checkFilePath(status, name, 'resManager project file', vital=True)
        else :
          super(resTaskItem, self)._attributeCheck(status, name)


    def _attributeIsPath(self, name) :
        """\
          bool _attributeIsPath(string name)
          
          Callback used by self.completePaths(...)
          Returns True if attribute is a path, please note that the
          'workdir' attribute should NOT be considered as a path here.
          Should be overriden to set the behavior for children classes.
          
          param name                  attribute name
          return                      is attribute a path
        """
        if name in ('project', ) :
          return True
        else :
          return super(resTaskItem, self)._attributeIsPath(name)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _antAction(self) :
        """\
          str _antAction(void)
          
          Callback used by antDisplay()
          Returns a short string explaining the task's action.
          Should be overriden to set the behavior for children classes.
          Disabled and non valid tasks are already handled.
          
          return                      task's action
        """
        result = super(resTaskItem, self)._antAction()
        
        # test if "result" is an empty string
        if not asqTypes.isNonEmptyInstance(result, basestring) :
          chunks = list()
          buffer = asqString.shortenString(self.project, 36, '...', '\\')
          
          chunks.append('compiling')
          chunks.append('`%s`' % (buffer, ))
          result = ' '.join(chunks)
        
        return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _fillCmdLine(self, aCmdLine) :
        """\
          void _fillCmdLine(object aCmdLine)
          
          Callback used by toCmdLine()
          Build the task command line, task is assumed valid.
          Should be overriden to set the behavior for children classes.
          
          param aCmdLine              cmdLine object
        """
        asqBuild.logger.trace('resTaskItem._fillCmdLine(aCmdLine)')
        asqBuild.logger.incDepthTrace()
        try :
          
          # path to resManager.exe
          aCmdLine.appendQuoted(self.manager)
          # project *.rmc file
          aCmdLine.append('-p')
          aCmdLine.appendQuoted(self.project)
          if asqTypes.isNonEmptyInstance(self.inifile, basestring) :
            aCmdLine.append('-o')
            aCmdLine.appendQuoted(self.inifile)
          if asqTypes.isNonEmptyInstance(self.resdir, basestring) :
            aCmdLine.append('-r')
            aCmdLine.appendQuoted(self.resdir)
          if asqTypes.isNonEmptyInstance(self.srcdir, basestring) :
            aCmdLine.append('-s')
            aCmdLine.appendQuoted(self.srcdir)
          if asqTypes.isNonEmptyInstance(self.cmdextra, basestring) :
            aCmdLine.append(self.cmdextra)
          
        finally :
          asqBuild.logger.decDepthTrace()


  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # /class resTaskItem  - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


class resTaskList(resTask, asqBuildCore.taskList) :
    """\
      A list of resTaskItems, the list defines "default" attributes that
      each single item can override; if an item doesn't set an attribute,
      then the list's values will be used.
    """

    # task item class
    taskClass   = resTaskItem

    # Properties defined for that task type and its descendant.
    # Note : attribute names should be "lower case" only.
    attributes    = (
        'taskname',             # task name (should held section name)
        'count',                # task item count
        'tasktype',             # task type as a string
        'enabled',              # is task active ?
        'workdir',              # task base directory
        'comments',             # about this task
        
        'manager',              # path to resManager executable
        #~ 'project',              # path to resManager project (*.rmc)
        'cmdextra',             # extra application parameters
        'inifile',              # resManager application options (*.ini)
        'resdir',               # path to image directory
        'srcdir',               # ???
        
        'others'                # unrecognized attributes
      )

  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # /class resTaskList  - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
  print __file__
  print __doc__

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -