#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
A delphi building utility: Wrapper for raw tasks.

Require
  Python        2.2

Classes
  class         rawTaskHandler(object aTask, object logger)
                  extends py_netasq.building.core::taskHandler
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.3 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  os

from    py_netasq.commonlib     import asqFile, asqPath, asqWin

from    py_netasq               import building as asqBuild
from    py_netasq.building      import core     as asqBuildCore

import  tasks


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

# token to replace in template file
TOK_CMDSTR  = '$cmdStr$'
TOK_ENVFILE = '$envFile$'

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class rawTaskHandler(asqBuildCore.taskHandler) :
    """\
      Wrapper for raw tasks. Settings have to be set through object's
      "task" (class rawTaskItem) attribute.
    """

    # task class
    taskClass = tasks.rawTaskItem

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __init__(self, aTask=None) :
        """\
          constructor rawTaskHandler(object aTask)
          
          Create a new object intended to ease command line execution.
          
          param aTask                 task to execute.
                                       defaults to None
        """
        super(rawTaskHandler, self).__init__(aTask)
        # path to temporary environment file, None => no temp file
        self._envFile   = None


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _execute(self, cmdStr, raw, *arglist, **argdict) :
        """\
          var _execute(string cmdStr, bool raw, *arglist, **argdict)
          
          Execute inner command line and return the shell status.
          Each line from stdout and stderr will be decoded and processed,
          i.e. lines will go through _decodeOutput() and _processOutput().
          
          param cmdStr                command as a string to execute.
          param raw                   disable output formatting
          param *arglist              extra parameters.
          param **argdict             extra named parameters.
          return                      shell status
        """
        asqBuild.logger.trace('rawTaskHandler._execute(%s, %s)' % (cmdStr, str(raw)))
        asqBuild.logger.incDepthTrace()
        try :
          
          #~ print '// rawTaskHandler._execute'
          #~ for (key, val) in os.environ.iteritems() :
            #~ print key, '=', val
          
          if not self._task.update_environ :
            return super(rawTaskHandler, self)._execute(cmdStr, raw, *arglist, **argdict)
          else :
            self._updateTmpFile(cmdStr)
            newCmd = self.__newCmdString()
            status = super(rawTaskHandler, self)._execute(newCmd, raw, *arglist, **argdict)
            
            os.environ.clear()
            # [ 1110478 ] os.environ.update doesn't work (python 2.4)
            # > https://sourceforge.net/tracker/?func=detail&atid=105470&aid=1110478&group_id=5470
            #
            # Submitted :       2005-01-26 23:22
            # Closed :          2005-01-29 05:34
            #
            #~ os.environ.update(self._parseEnvFile())
            buffer = self._parseEnvFile()
            for (key, val) in buffer.iteritems() :
              os.environ[key] = val
            
            #~ print ''
            #~ print '// os.environ.update'
            #~ for (key, val) in os.environ.iteritems() :
              #~ print key, '=', val
            
            return status
         
        finally :
          asqBuild.logger.decDepthTrace()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _setup(self, *arglist, **argdict) :
        """\
          void _setup(*arglist, **argdict)
          
          Called right before the actual command execution; should be
          overidden to set up the execution environment.
          
          - Wraps the command line in a script in order to retrieve the
           changes made to the OS environment (only if task.update_environ
           has been set to True)
          
          param *arglist              extra parameters.
          param **argdict             extra named parameters.
        """
        asqBuild.logger.trace('rawTaskHandler._setup()')
        asqBuild.logger.incDepthTrace()
        try :
          
          super(rawTaskHandler, self)._setup(*arglist, **argdict)
          
          if self._task.update_environ :
            self._createTmpFile(dir=self._task.workdir)
            self._createEnvFile()
            self._task.buildTmpFile(self._tmpFile)
          
            #~ print '//rawTaskHandler._setup'
            #~ buffer = open(self._task.tmpfile, 'r')
            #~ try :
              #~ print ''.join( buffer.readlines() )
            #~ finally :
              #~ buffer.close
          
        finally :
          asqBuild.logger.decDepthTrace()


    def _teardown(self, *arglist, **argdict) :
        """\
          void _teardown(*arglist, **argdict)
          
          Called right after the actual command execution; should be
          overidden to clean up the execution environment.
          
          - delete temporary command wrapper
          
          param *arglist              extra parameters.
          param **argdict             extra named parameters.
        """
        asqBuild.logger.trace('rawTaskHandler._teardown()')
        asqBuild.logger.incDepthTrace()
        try :
          
          if self._task.update_environ :
            # Comment out the following code lines to keep temporary files
            #  This is sometimes useful to understand why a build did fail
            self._deleteEnvFile()
            self._deleteTmpFile()
            #~ pass
          
          super(rawTaskHandler, self)._teardown(*arglist, **argdict)
          
        finally :
          asqBuild.logger.decDepthTrace()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _createTmpFile(self, suffix="", dir=None) :
        """\
          void _createTmpFile(string suffix, string dir)
          
          Creates a temporary file in the most secure manner possible.
          No dot will be automatically put between the file name and
          the suffix, it should be added manually.
          Self._tmpFile is used to held the result of absolute path of the
          file. use mkstemp(). New in version 2.3.
          
          param suffix                file name suffix
                                        defaults to ""
          param dir                   parent dir of temp file
                                        defaults to None
        """
        if -1 == asqWin.getWindowsPlatform() :
          raise asqBuildCore.taskError('TODO : rawTaskHandler._createTmpFile()', self._task)
        else :
          super(rawTaskHandler, self)._createTmpFile('.bat', dir)


    def _updateTmpFile(self, cmdStr) :
        """\
          void _updateTmpFile(string cmdStr)
          
          Replaces variables in the wrapper template file.
          
          param cmdStr                command line to wrap
        """
        buffer = asqFile.readFile(self._tmpFile)
        
        buffer = buffer.replace(TOK_CMDSTR  , cmdStr)
        buffer = buffer.replace(TOK_ENVFILE , self._envFile)
        
        asqFile.writeFile(self._tmpFile, buffer)

        #~ print '//rawTaskHandler._updateTmpFile'
        #~ buffer = open(self._task.tmpfile, 'r')
        #~ try :
          #~ print ''.join( buffer.readlines() )
        #~ finally :
          #~ buffer.close


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _createEnvFile(self) :
        """\
          void _createEnvFile(void)
          
          Creates a temporary file in the most secure manner possible.
          The file may receive the new environment output
          (e.g. `$ SET > <envFile_path>`). Self._envFile is used to held
          the result of absolute path of the file.
        """
        asqBuild.logger.trace('rawTaskHandler._createEnvFile()')
        asqBuild.logger.incDepthTrace()
        try :
          
          self._envFile = asqPath.makeTmpFile(suffix='.txt', prefix='env_')
          assert os.path.isabs(self._envFile)
         
        finally :
          asqBuild.logger.decDepthTrace()


    def _deleteEnvFile(self) :
        """\
          void _deleteTmpFile(void)
          
          Deletes a previously created temporary file.
        """
        asqBuild.logger.trace('taskHandler._deleteEnvFile()')
        asqBuild.logger.incDepthTrace()
        try :
          
          if self._envFile :
            asqPath.delete(self._envFile, True)
         
        finally :
          asqBuild.logger.decDepthTrace()


    def _parseEnvFile(self) :
        """\
          void _parseEnvFile(void)
          
          Reads the envFile and updates the os.environ dictionary.
        """
        if -1 == asqWin.getWindowsPlatform() :
          raise asqBuildCore.taskError('TODO : rawTaskHandler._parseEnvFile()', self._task)
        else :
          result = dict()
          handle = open(self._envFile, 'rU')
          try :
            for line in handle :
              buffer = [ s.strip() for s in line.split('=', 1) ]
              result[ buffer[0] ] = buffer[1]
          finally :
            handle.close()
          
          return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __newCmdString(self) :
        """\
          string __newCmdString(void)
          
          Builds the command string to use the wrapper.
          
          return                      command line
        """
        if -1 == asqWin.getWindowsPlatform() :
          raise asqBuildCore.taskError('TODO : rawTaskHandler.__newCmdString()', self._task)
        else :
          return 'CALL "%s"' % (self._tmpFile, )


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # /class rawTaskHandler - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -