#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
Define a mail task; settings for reporting mail.

Require
  Python        2.2.2

Constants
  string        mailTaskType

Classes
  class         mailTaskItem(bool enabled)
                  extends py_netasq.building.core::taskItem
  class         mailTaskList(bool enabled)
                  extends py_netasq.building.core::taskList
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.21 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  getpass
import  socket
import  smtplib

from    email.MIMEMultipart     import MIMEMultipart
from    email.MIMEText          import MIMEText

from    py_netasq.commonlib     import asqMail, asqString, asqTypes

from    py_netasq               import building as asqBuild
from    py_netasq.building      import core     as asqBuildCore


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

mailTaskType = 'mailTask'

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class mailTask(asqBuildCore.task) :
    """\
      A mail task: send logfile to defined recipients, "failure" or
      "success" attributes will be used as mail subjects, if building
      process either failed or succeed. if "sender" is not set, then
      the "login name" of the current user will be used.
    """

    # class task type, nuff said :)
    classTaskType = mailTaskType

    # task type compatibility list
    compatibility = asqBuildCore.task.compatibility + (
        classTaskType,
      )

    # Properties defined for that task type and its descendant.
    # Note : attribute names should be "lower case" only.
    attributes    = (
        'taskname',             # task name (should held section name)
        'tasktype',             # task type as a string
        'enabled',              # is task active ?
        'workdir',              # task base directory - actually useless
        'comments',             # about this task
        
        'smtphost',             # host address
        'sender',               # mail sender
        #~ 'recipients',           # mail recipients
        'failure',              # mail subject "build failed"
        'success',              # mail subject "build success"
        'error_preview',        # last lines from the error log, '-1' means no limit
        
        'others'                # unrecognized attributes
      )

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __defaultSender(self, domain='netasq.com') :
        """\
          string __defaultSender(void)
          
          Returns the default sender address, build upon the name of the
          current user name.
          
          return                      default sender address
        """
        try :
          result = getpass.getuser()
        except ImportError, e :
          # an ImportError is raised on win98 boxes
          result = 'unknow'
        
        return '%s@%s' % (result, domain)


    def __connectSmtpHost(self) :
        """\
          bool __connectSmtpHost(void)
          
          Tries to establish a connection to the set smtp host; returns
          whether or not the connection was succesful. No SMTP session
          is kept open.
          
          return                      connection succesful ?
        """
        buffer = smtplib.SMTP()
        try :
          
          try :
            buffer.connect(self.smtphost)
          except smtplib.SMTPConnectError, e :
            return False
          except socket.error, e :
            return False
          else :
            buffer.quit()
            return True
          
        finally :
          del buffer


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _checkSmtpHost(self, status, name, label, vital=False) :
        """\
          void _checkSmtpHost(object status, string name, string label, bool vital)
          
          Helper function used by _attributeCheck(). Provided to reduce
          redundancy. If "vital" is set to True, then ANY abnormal setting
          will produce errors instead of warnings.
          
          param status                checkResult object
          param name                  attribute name
          param label                 attribute formal name
          param vital                 is attribute setting vital ?
                                        defaults to False
        """
        buffer = self.__dict__[name]
        if buffer is None :
          if vital :
            status.addMsgError('%-15s %s is not set' % (name, label))
          else :
            status.addMsgWarning('%-15s %s is not set' % (name, label))
        elif not asqTypes.isNonEmptyInstance(buffer, basestring) :
          status.addMsgError('%-15s %s is invalid' % (name, label))
        elif not self.__connectSmtpHost() :
          status.addMsgError('%-15s %s connection failed' % (name, label))


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _attributeCheck(self, status, name) :
        """\
          void _attributeCheck(object status, string name)
          
          Callback used by self.check(...)
          Checks the value of attribute; Appends error or warning
          messages to given check status.
          Should be overriden to set the behavior for children classes.
          
          param status                checkResult object
          param name                  attribute name
        """
        if 'smtphost' == name :
          self._checkSmtpHost(
              status, name, 'smtp host',
              vital=isinstance(self, asqBuildCore.taskItem)
            )
        elif 'failure' == name :
          self._checkValue(status, name, 'failure mail subject')
        elif 'success' == name :
          self._checkValue(status, name, 'success mail subject')
        else :
          super(mailTask, self)._attributeCheck(status, name)


    def _attributeNormalize(self, name) :
        """\
          void _attributeNormalize(string name)
          
          Callback used by self.normalize()
          Normalizes attribute : ensures that its type is correct.
          Should be overriden to set the behavior for children classes.
          
          param name                  attribute name
        """
        if 'sender' == name :
          if not asqTypes.isNonEmptyInstance(self.sender, basestring) :
            object.__setattr__(self, name, self.__defaultSender())
        elif 'error_preview' == name :
          if not asqTypes.isNonEmptyInstance(self.error_preview, int) :
            buffer = asqTypes.asInteger(self.error_preview, 10, -1)
            object.__setattr__(self, name, buffer)
        else :
          super(mailTask, self)._attributeNormalize(name)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def isValid(self) :
        """\
          bool isValid(void)
          
          Tells whether or not the task is "valid", id est, if mandatory
          attributes are set; an empty task is not valid, but an invalid
          task may not be empty ;)
          
          return                      is task valid
        """
        asqBuild.logger.trace('mailTask.isValid()')
        asqBuild.logger.incDepthTrace()
        try :
          
          if not super(mailTask, self).isValid() :
            return False
          else :
            return asqTypes.isNonEmptyInstance(self.smtphost, basestring)
          
        finally :
          asqBuild.logger.decDepthTrace()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # /class mailTask - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


class mailTaskItem(mailTask, asqBuildCore.taskItem) :
    """\
      A mail task: send logfile to defined recipients, "failure" or
      "success" attributes will be used as mail subjects, if building
      process either failed or succeed. if "sender" is not set, then
      the "login name" of the current user will be used.
      
      
      object            toMessage(bool succesful, string msgText)
    """
    
    ant_name    = 'mail'
    
    # Properties defined for that task type and its descendant.
    # Note : attribute names should be "lower case" only.
    # Properties defined for that task type and its descendant.
    # Note : attribute names should be "lower case" only.
    attributes  = (
        'taskname',             # task name (should held section name)
        'tasktype',             # task type as a string
        'enabled',              # is task active ?
        'workdir',              # task base directory - actually useless
        'comments',             # about this task
        
        'smtphost',             # host address
        'sender',               # mail sender
        'recipients',           # mail recipients
        'failure',              # mail subject "build failed"
        'success',              # mail subject "build success"
        'error_preview',        # last lines from the error log, '-1' means no limit
        
        'others'                # unrecognized attributes
      )

    # properties that could be overwritten by a "Merge" method
    mergeable   = (
        #~ 'workdir',
        'smtphost',
        'sender',
        'failure',
        'success',
        'error_preview',
      )

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def toMessage(self, succesful=False, mailtext='') :
        """\
          object toMessage(bool succesful, string msgText)
          
          Build and returns a MIMEText object according to task's settings.
          If "succesful" is set to True, then the 'success' string will be
          used as mail subject, else the 'failure' string is used.
          Task should be valid.
          
          param succesful             which subject to use
                                        defaults to False
          param mailtext              mail short text
                                        defaults to ''
          return                      a email.Message object
        """
        assert self.isValid()
        
        titles = { False : str(self.failure), True : str(self.success) }
        result = MIMEMultipart('mixed')
        
        asqMail.setupMessage(result, self.sender, self.recipients, titles[succesful])
        if mailtext : result.attach( MIMEText(mailtext) )
        
        return result


    def isValid(self) :
        """\
          bool isValid(void)
          
          Tells whether or not the task is "valid", id est, if mandatory
          attributes are set; an empty task is not valid, but an invalid
          task may not be empty ;)
          
          return                      is task valid
        """
        asqBuild.logger.trace('mailTaskItem.isValid()')
        asqBuild.logger.incDepthTrace()
        try :
          
          if not super(mailTaskItem, self).isValid() :
            return False
          else :
            return asqTypes.isNonEmptyInstance(self.recipients, basestring)
          
        finally :
          asqBuild.logger.decDepthTrace()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _attributeCheck(self, status, name) :
        """\
          void _attributeCheck(object status, string name)
          
          Callback used by self.check(...)
          Checks the value of attribute; Appends error or warning
          messages to given check status.
          Should be overriden to set the behavior for children classes.
          
          param status                checkResult object
          param name                  attribute name
        """
        if 'recipients' == name :
          self._checkValue(status, name, 'recipients list', vital=True)
        else :
          super(mailTaskItem, self)._attributeCheck(status, name)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _antAction(self) :
        """\
          str _antAction(void)
          
          Callback used by antDisplay()
          Returns a short string explaining the task's action.
          Should be overriden to set the behavior for children classes.
          Disabled and non valid tasks are already handled.
          
          return                      task's action
        """
        result = super(mailTaskItem, self)._antAction()
        
        # test if "result" is an empty string
        if not asqTypes.isNonEmptyInstance(result, basestring) :
          chunks = list()
          buffer = asqString.truncateString(self.recipients, 32, '...', ',')
          
          chunks.append('sending message to')
          chunks.append(buffer)
          result = ' '.join(chunks)
        
        return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _fillCmdLine(self, aCmdLine) :
        """\
          void _fillCmdLine(object aCmdLine)
          
          Callback used by toCmdLine()
          Build the task command line, task is assumed valid.
          Should be overriden to set the behavior for children classes.
          
          param aCmdLine              cmdLine object
        """
        asqBuild.logger.trace('mailTaskItem._fillCmdLine(aCmdLine)')
        asqBuild.logger.incDepthTrace()
        try :
          
          aCmdLine.append('mail')
          aCmdLine.append(self.recipients)
          
        finally :
          asqBuild.logger.decDepthTrace()


  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # /class mailTaskItem - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


class mailTaskList(mailTask, asqBuildCore.taskList) :
    """\
      A list of mailTaskItems, the list defines "default" attributes that
      each single item can override; if an item doesn't set an attribute,
      then the list's values will be used.
    """

    # task item class
    taskClass   = mailTaskItem

    # Properties defined for that task type and its descendant.
    # Note : attribute names should be "lower case" only.
    attributes    = (
        'taskname',             # task name (should held section name)
        'count',                # task item count
        'tasktype',             # task type as a string
        'enabled',              # is task active ?
        'workdir',              # task base directory - actually useless
        'comments',             # about this task
        
        'smtphost',             # host address
        'sender',               # mail sender
        #~ 'recipients',           # mail recipients
        'failure',              # mail subject "build failed"
        'success',              # mail subject "build success"
        'error_preview',        # last lines from the error log, '-1' means no limit
        
        'others'                # unrecognized attributes
      )

  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # /class mailTaskList - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -