#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
defines a global handler and a taskListHandler

Require
  Python        2.2.2

Classes
  class         globalTaskHandler(object aTask)
                  extends py_netasq.building.core::taskHandler
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.12 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

from    py_netasq               import building as asqBuild
from    py_netasq.building      import core     as asqBuildCore

from    factory                 import factory


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class globalTaskHandler(asqBuildCore.taskHandler) :
    """\
      A task handler which can handle ANY tasks execution.
    """

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __init__(self, aTask=None) :
        """\
          constructor globalTaskHandler(object aTask, object logger)
          
          Create a new object intended to ease command line execution.
          
          param aTask                 task to execute.
                                       defaults to None
          param logger                output logger
                                       defaults to None
        """
        self._objects = dict()
        # taskHandler.__init__ will assign 'aTask' to 'self.task'
        super(globalTaskHandler, self).__init__(aTask)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def run(self, aTask=None, raw=False, *arglist, **argdict) :
        """\
          var run(object aTask, bool raw, *arglist, **argdict)
          
          Execute inner task line, if "task" is given, it will replace
          the current taskItem instance; The shell status is returned,
          'None' if execution was succesful, any other value means
          failure. If "raw" is set to True, then no peculiar operation
          should be done upon the command's output.
          
          param aTask                 taskItem to execute.
                                       defaults to None
          param raw                   disable output formatting
                                        defaults to False
          param *arglist              extra parameters.
          param **argdict             extra named parameters.
          return                      shell status
        """
        if aTask is None :
          asqBuild.logger.trace('globalTaskHandler.run(aTask=None, raw=%s)', str(raw))
        else :
          asqBuild.logger.trace('globalTaskHandler.run(aTask, raw=%s)', str(raw))
        asqBuild.logger.incDepthTrace()
        try :
          
          if aTask is not None :
            self.task = aTask
          try :
            # because of _setTask, self.task could be None, thus
            # an AttributeError may be raised
            tkType = self.task.tasktype
          except AttributeError, e :
            return None
          else :
            handler = self._objects[tkType][1]
            return handler.run(self.task, raw, *arglist, **argdict)
          
        finally :
          asqBuild.logger.decDepthTrace()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _taskForType(self, aType) :
        """\
          object _taskForType(string aType)
          
          Returns the task item "holder" for the given 'aType'. If no
           task item does exist for that type, then a new one will be
           built (and its taskhandler in the same time).
          If given 'aType' is not recognised as a tasktype by the tasks
          factory, then None should be returned.
          
          param aType                 requested task's type
          return                      taskItem instance
        """
        try :
          # try to fetch task item, may raise KeyError
          result = self._objects[aType][0]
        except KeyError, e :
          # given type is missing from the '_objects' dictionary; both task
          # item and handler will be created accordingly.
          # If given 'aType' is not recognised, then the factory instance
          #  should return None as task item and task handler
          # note : 0 = taskItem, 1 = taskHandler
          self._objects[aType] = [None, None]
          self._objects[aType][0] = factory.buildTaskItem(aType, enabled=False)
          self._objects[aType][1] = factory.buildTaskHandler(aType)
          
          # do not forget the 'onProcessInput' and 'onProcessOutput' handler
          self._objects[aType][1].onEmitMsg       = self.onEmitMsg
          self._objects[aType][1].onProcessInput  = self.onProcessInput
          self._objects[aType][1].onProcessOutput = self.onProcessOutput
          
          result = self._objects[aType][0]
          
        return result


  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _getOutput(self) :
        """\
          list _getOutput(void)
          
          Getter - output property
          
          return                      output from the last command ran.
        """
        try :
          # self.task could be None, thus an AttributeError may be raised
          tkType = self.task.tasktype
          handlr = self._objects[tkType][1]
        except (AttributeError, KeyError), e :
          return super(globalTaskHandler, self)._getOutput()
        else :
          return handlr.output


    def _getTask(self) :
        """\
          object _getTask(void)
          
          Getter - task property
          
          return                      current taskItem value.
        """
        return super(globalTaskHandler, self)._getTask()


    def _getOnEmitMsg(self) :
        """\
          object _getOnEmitMsg(void)
          
          Getter - onEmitMsg property
          Returns handler bound to the emitMsg event
          
          return                      onEmitMsg handler
        """
        return super(globalTaskHandler, self)._getOnEmitMsg()


    def _getOnProcessInput(self) :
        """\
          var _getOnProcessInput(void)
          
          Getter - onProcessInput property
          Handler called upon each line of the command input.
        """
        return super(globalTaskHandler, self)._getOnProcessInput()


    def _getOnProcessOutput(self) :
        """\
          var _getOnProcessOutput(void)
          
          Getter - onProcessOutput property
          Handler called upon each line of the command output.
        """
        return super(globalTaskHandler, self)._getOnProcessOutput()


  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _setTask(self, value) :
        """\
          void _setTask(object value)
          
          Setter - task property.
          If no task item / task handler are defined for the given "value"
          type, then they are created, else those instances are re-used.
          
          param value                 new taskItem value.
        """
        try :
          
          # if value is None, may raise AttributeError
          tkType = value.tasktype
          # tkType is not a valid task type => result is None
          buffer = self._taskForType(tkType)
          
        except AttributeError, e :
          buffer = None
         
        if buffer is not None :
          buffer.assign(value)
          buffer.completePaths()
        
        # keep a reference as the current task item to run
        self._task = buffer


    def _setOnEmitMsg(self, value) :
        """\
          void _setOnEmitMsg(object value)
          
          Setter - onEmitMsg property
          Sets the handler bound to the emitMsg event
          
          param value                 event handler
        """
        super(globalTaskHandler, self)._setOnEmitMsg(value)
        
        for key in self._objects.iterkeys() :
          try :
            # task handler can be None, thus may raise AttributeError
            self._objects[key][1].onEmitMsg = value
          except AttributeError, e :
            pass


    def _setOnProcessInput(self, value) :
        """\
          void _setOnProcessInput(var value)
          
          Setter - onProcessInput property
          Handler called upon each line of the command input. Updates the
          'onProcessInput' handler for each taskHandlers.
        """
        super(globalTaskHandler, self)._setOnProcessInput(value)
        
        for key in self._objects.iterkeys() :
          try :
            # task handler can be None, thus may raise AttributeError
            self._objects[key][1].onProcessInput = value
          except AttributeError, e :
            pass


    def _setOnProcessOutput(self, value) :
        """\
          void _setOnProcessOutput(var value)
          
          Setter - onProcessOutput property
          Handler called upon each line of the command output. Updates the
          'onProcessOutput' handler for each taskHandlers.
        """
        super(globalTaskHandler, self)._setOnProcessOutput(value)
        
        for key in self._objects.iterkeys() :
          try :
            # task handler can be None, thus may raise AttributeError
            self._objects[key][1].onProcessOutput = value
          except AttributeError, e :
            pass


  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    output              = property(
        doc  = "retains the output from the last command ran",
        fget = _getOutput,
        fset = None,
        fdel = None
      )

    task                = property(
        doc  = "current task item to handle",
        fget = _getTask,
        fset = _setTask,
        fdel = None
      )

    onEmitMsg   = property(
        doc  = "trigerred each time a message is emitted",
        fget = _getOnEmitMsg,
        fset = _setOnEmitMsg,
        fdel = None
      )

    onProcessInput      = property(
        doc  = "Handler called upon each line of the command input.",
        fget = _getOnProcessInput,
        fset = _setOnProcessInput,
        fdel = None
      )

    onProcessOutput     = property(
        doc  = "Handler called upon each line of the command output.",
        fget = _getOnProcessOutput,
        fset = _setOnProcessOutput,
        fdel = None
      )


  # /class globalTaskHandler  - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -