#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
concentrates all task types and defines a task factory.

Require
  Python        2.2.2

Constants
  tuple         taskTypes
  string        cvsTaskType
  string        dccTaskType
  string        delTaskType
  string        rawTaskType
  string        resTaskType
  string        sf6TaskType
  string        sf7TaskType
  string        svnTaskType
  string        vrsTaskType
  string        mailTaskType

Functions
  var           taskItemClass(string aType)
  var           taskListClass(string aType)
  var           taskHandlerClass(string aType)

Classes
  class         factory(object)
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.2 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
#
# To handle a new task type, one should import the appropriate task
#  package, and then update the '_taskClasses' dictionnary.
#
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  sys

from    cvsTask                 import  cvsTaskType, cvsTaskItem, cvsTaskList, cvsTaskHandler
from    dccTask                 import  dccTaskType, dccTaskItem, dccTaskList, dccTaskHandler
from    delTask                 import  delTaskType, delTaskItem, delTaskList, delTaskHandler
from    rawTask                 import  rawTaskType, rawTaskItem, rawTaskList, rawTaskHandler
from    resTask                 import  resTaskType, resTaskItem, resTaskList, resTaskHandler
from    sf6Task                 import  sf6TaskType, sf6TaskItem, sf6TaskList, sf6TaskHandler
from    sf7Task                 import  sf7TaskType, sf7TaskItem, sf7TaskList, sf7TaskHandler
from    svnTask                 import  svnTaskType, svnTaskItem, svnTaskList, svnTaskHandler
from    vrsTask                 import  vrsTaskType, vrsTaskItem, vrsTaskList, vrsTaskHandler

from    mailTask                import  mailTaskType, mailTaskItem, mailTaskList, mailTaskHandler


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

# class ordered by task type; task item, task list and task handler
_taskClasses  = {
    # task type  : (0 : task item, 1 : task list, 2 : handler)
    cvsTaskType  : (cvsTaskItem, cvsTaskList, cvsTaskHandler),
    dccTaskType  : (dccTaskItem, dccTaskList, dccTaskHandler),
    delTaskType  : (delTaskItem, delTaskList, delTaskHandler),
    rawTaskType  : (rawTaskItem, rawTaskList, rawTaskHandler),
    resTaskType  : (resTaskItem, resTaskList, resTaskHandler),
    sf6TaskType  : (sf6TaskItem, sf6TaskList, sf6TaskHandler),
    sf7TaskType  : (sf7TaskItem, sf7TaskList, sf7TaskHandler),
    svnTaskType  : (svnTaskItem, svnTaskList, svnTaskHandler),
    vrsTaskType  : (vrsTaskItem, vrsTaskList, vrsTaskHandler),
    
    mailTaskType : (mailTaskItem, mailTaskList, mailTaskHandler)
  }


# classes index
__TASKITEM = 0
__TASKLIST = 1
__HANDLER  = 2

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

# whole task types list
taskTypes = tuple(_taskClasses.keys())

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

def taskItemClass(aType) :
    """\
      var taskItemClass(string aType)
      
      Returns the task item class for given task type "aType". None may
      be returned if there is no class matching the given criteria.
      
      param aType                 task type
      return                      a task item class, or None
    """
    try :
      result = _taskClasses[aType]
    except KeyError, e :
      buffer = "unknown task type %s" % (repr(aType), )
      # add previous traceback to exception
      raise ValueError(buffer), None, sys.exc_info()[2]
    else :
      return result[__TASKITEM]


def taskListClass(aType) :
    """\
      var taskListClass(string aType)
      
      Returns the task list class for given task type "aType". None may
      be returned if there is no class matching the given criteria.
      
      param aType                 task type
      return                      a task list class, or None
    """
    try :
      result = _taskClasses[aType]
    except KeyError, e :
      buffer = "unknown task type %s" % (repr(aType), )
      # add previous traceback to exception
      raise ValueError(buffer), None, sys.exc_info()[2]
    else :
      return result[__TASKLIST]


def taskHandlerClass(aType) :
    """\
      var taskHandlerClass(string aType)
      
      Returns the task handler class for given task type "aType". None
      may be returned if there is no class matching the given criteria.
      
      param aType                 task type
      return                      a task handler class, or None
    """
    try :
      result = _taskClasses[aType]
    except KeyError, e :
      buffer = "unknown task type %s" % (repr(aType), )
      # add previous traceback to exception
      raise ValueError(buffer), None, sys.exc_info()[2]
    else :
      return result[__HANDLER]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class factory(object) :
    """\
      Define a factory to create task items and command handlers according
      to a tasktype. This class should not be instanciated, since only
      classmethods are provided.
      
      object            taskItemFromDict(dict aDict, string defaultType)
      object            taskItemFromList(list aList, string defaultType)
      
      object            taskListFromDict(dict aDict, string defaultType)
      object            taskListFromList(list aList, string defaultType)
      
      object            buildTaskItem(string aType, *argList, **argDict)
      object            buildTaskList(string aType, *argList, **argDict)
      object            buildTaskHandler(string aType, *argList, **argDict)
    """

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def taskItemFromDict(cls, aDict, defaultType=None) :
        """\
          object taskItemFromDict(dict aDict, string defaultType)
          
          Create a task item from given dictionnary, task type is extracted
          from the dict; if no type is found, then 'defaultType' is assumed.
          If no task type is not recognised then None is returned.
          
          param aDict                 attributes dictionnary
          param defaultType           assumed tasktype, defaults to None
          return                      a new task item, or None
        """
        tkType = aDict.get('tasktype', defaultType)
        result = cls.buildTaskItem(tkType, enabled=False)
        if result is not None :
          result.fromDict(aDict, defaultType)
        return result


    def taskItemFromList(cls, aList, defaultType=None) :
        """\
          object taskItemFromList(list aList, string defaultType)
          
          Create a task item from given list, task type is extracted from
          the list; if no type is found, then 'defaultType' is assumed.
          If no task type is not recognised then None is returned.
          
          param aList                 list of attributes
          param defaultType           assumed tasktype, defaults to None
          return                      a new task item, or None
        """
        buffer = dict(aList)
        try :
          
          return cls.taskItemFromDict(buffer, defaultType)
          
        finally :
          del buffer


    def taskListFromDict(cls, aDict, defaultType=None) :
        """\
          object taskListFromDict(list aDict, string defaultType)
          
          Create a task list from given dictionnary, task type is extracted
          from the dict; if no type is found, then 'defaultType' is assumed.
          If no task type is not recognised then None is returned.
          
          param aDict                 attributes dictionnary
          param defaultType           assumed tasktype, defaults to None
          return                      a new task item, or None
        """
        tkType = aDict.get('tasktype', defaultType)
        result = cls.buildTaskList(tkType, enabled=False)
        if result is not None :
          result.fromDict(aDict, defaultType)
        return result


    def taskListFromList(cls, aList, defaultType=None) :
        """\
          object taskListFromList(list aList, string defaultType)
          
          Create a task list from given list, task type is extracted from
          the list; if no type is found, then 'defaultType' is assumed.
          If no task type is not recognised then None is returned.
          
          param aList                 list of attributes
          param defaultType           assumed tasktype, defaults to None
          return                      a new task item, or None
        """
        buffer = dict(aList)
        try :
          
          return cls.taskListFromDict(buffer, defaultType)
          
        finally :
          del buffer


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def buildTaskItem(cls, aType, *argList, **argDict) :
        """\
          object buildTaskItem(string aType, *argList, **argDict)
          
          Create a task item, according to given task type 'aType'. If task
          type is not recognised then None is returned.
          
          param aType                 task type
          param *argList              dynamic arg list
          param **argDict             dynamic arg dict
          return                      a new task item, or None
        """
        buffer = taskItemClass(aType)
        if callable(buffer) :
          return buffer(*argList, **argDict)
        else :
          return None


    def buildTaskList(cls, aType, *argList, **argDict) :
        """\
          object buildTaskList(string aType, *argList, **argDict)
          
          Create a task list, according to given task type 'aType'. If task
          type is not recognised then None is returned.
          
          param aType                 task type
          param *argList              dynamic arg list
          param **argDict             dynamic arg dict
          return                      a new task list, or None
        """
        buffer = taskListClass(aType)
        if callable(buffer) :
          return buffer(*argList, **argDict)
        else :
          return None


    def buildTaskHandler(cls, aType, *argList, **argDict) :
        """\
          object buildTaskHandler(string aType, *argList, **argDict)
          
          Create a task handler, according to given task type 'aType'. If
          task type is not recognised then None is returned.
          
          param aType                 task type
          param *argList              dynamic arg list
          param **argDict             dynamic arg dict
          return                      a new task handler, or None
        """
        buffer = taskHandlerClass(aType)
        if callable(buffer) :
          return buffer(*argList, **argDict)
        else :
          return None


  # ClassMethods  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    taskItemFromDict    = classmethod(taskItemFromDict)
    taskItemFromList    = classmethod(taskItemFromList)
    
    taskListFromDict    = classmethod(taskListFromDict)
    taskListFromList    = classmethod(taskListFromList)
    
    buildTaskItem       = classmethod(buildTaskItem)
    buildTaskList       = classmethod(buildTaskList)
    buildTaskHandler    = classmethod(buildTaskHandler)

  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # /class factory  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -