#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
A system safe delete task item used by the building utility.

Require
  Python        2.2

Constants
  string        delTaskType

Classes
  class         delTaskItem(bool enabled)
                  extends py_netasq.building.core::taskItem
  class         delTaskList(bool enabled)
                  extends py_netasq.building.core::taskList
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.17 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  glob

from    py_netasq.commonlib     import asqString, asqTypes

from    py_netasq               import building as asqBuild
from    py_netasq.building      import core     as asqBuildCore


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

delTaskType  = 'delTask'

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class delTask(asqBuildCore.task) :
    """\
      A delete task: remove file(s) matching the target pattern. Since
      no shell command do provide the wanted behavior, we will rely on
      python methods.
    """

    # class task type, nuff said :)
    classTaskType = delTaskType

    # task type compatibility list
    compatibility = asqBuildCore.task.compatibility + (
        classTaskType,
      )

    # Properties defined for that task type and its descendant.
    # Note : attribute names should be "lower case" only.
    attributes    = (
        'taskname',             # task name (should held section name)
        'tasktype',             # task type as a string
        'enabled',              # is task active ?
        'workdir',              # task base directory
        'comments',             # about this task
        
        #~ 'target',               # path patterns to remove
        #~ 'exclude',              # path patterns to keep
        
        'others'                # unrecognized attributes
      )

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # /class delTask  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


class delTaskItem(delTask, asqBuildCore.taskItem) :
    """\
      A delete task: remove file(s) matching the target pattern. Since
      no shell command do provide the wanted behavior, we will rely on
      python methods.
      the "target" property can be used as a "wildcard" pattern.
    """
    
    ant_name    = 'delete'
    
    # Properties defined for that task type and its descendant.
    # Note : attribute names should be "lower case" only.
    attributes  = (
        'taskname',             # task name (should held section name)
        'tasktype',             # task type as a string
        'enabled',              # is task active ?
        'workdir',              # task base directory
        'comments',             # about this task
        
        'target',               # path patterns to remove
        'exclude',              # path patterns to keep
        
        'others'                # unrecognized attributes
      )

    # properties that could be overwritten by a "Merge" method
    mergeable   = (
        'workdir',
      )

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def isValid(self) :
        """\
          bool isValid(void)
          
          Tells whether or not the task is "valid", id est, if mandatory
          attributes are set; an empty taskItem is not valid, but an
          invalid taskItem may not be empty ;)
          
          return                      is task valid
        """
        asqBuild.logger.trace('delTaskItem.isValid()')
        asqBuild.logger.incDepthTrace()
        try :
          
          if not super(delTaskItem, self).isValid() :
            return False
          else :
            return asqTypes.isNonEmptyInstance(self.target, basestring)
          
        finally :
          asqBuild.logger.decDepthTrace()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _checkGlob(self, status, name, label, vital=False) :
        """\
          void _checkGlob(object status, string name, string label, bool vital)
          
          Helper function used by _attributeCheck(). Provided to reduce
          redundancy. If "vital" is set to True, then ANY abnormal setting
          will produce errors instead of warnings.
          
          param status                checkResult object
          param name                  attribute name
          param label                 attribute formal name
          param vital                 is attribute setting vital ?
                                        defaults to False
        """
        buffer = self.__dict__[name]
        if buffer is None :
          if vital :
            status.addMsgError('%-15s %s is not set' % (name, label))
          else :
            status.addMsgWarning('%-15s %s is not set' % (name, label))
        elif not asqTypes.isNonEmptyInstance(buffer, basestring) :
          status.addMsgError('%-15s %s is invalid'  % (name, label))
        else :
          for item in buffer.split(';') :
            if not glob.glob(item) :
              status.addMsgWarning('%-15s no file names do match %s : %s' % (name, label, item))


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _attributeCheck(self, status, name) :
        """\
          void _attributeCheck(object status, string name)
          
          Callback used by self.check(...)
          Checks the value of attribute; Appends error or warning
          messages to given check status.
          Should be overriden to set the behavior for children classes.
          
          param status                checkResult object
          param name                  attribute name
        """
        if 'target' == name :
          self._checkGlob(status, name, 'target pattern', vital=True)
        elif 'exclude' == name :
          self._checkGlob(status, name, 'exclude pattern', vital=False)
        else :
          super(delTaskItem, self)._attributeCheck(status, name)


    def _attributeIsPath(self, name) :
        """\
          bool _attributeIsPath(string name)
          
          Callback used by self.completePaths(...)
          Returns True if attribute is a path, please note that the
          'workdir' attribute should NOT be considered as a path here.
          Should be overriden to set the behavior for children classes.
          
          param name                  attribute name
          return                      is attribute a path
        """
        if name in ('target', 'exclude') :
          return True
        else :
          return super(delTaskItem, self)._attributeIsPath(name)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _antAction(self) :
        """\
          str _antAction(void)
          
          Callback used by antDisplay()
          Returns a short string explaining the task's action.
          Should be overriden to set the behavior for children classes.
          Disabled and non valid tasks are already handled.
          
          return                      task's action
        """
        result = super(delTaskItem, self)._antAction()
        
        # test if "result" is an empty string
        if not asqTypes.isNonEmptyInstance(result, basestring) :
          chunks = list()
          buffer = asqString.shortenString(self.target, 36, '...', '\\')
          
          chunks.append('deleting')
          chunks.append('`%s`' % (buffer, ))
          result = ' '.join(chunks)
        
        return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _fillCmdLine(self, aCmdLine) :
        """\
          void _fillCmdLine(object aCmdLine)
          
          Callback used by toCmdLine()
          Build the task command line, task is assumed valid.
          Should be overriden to set the behavior for children classes.
          
          param aCmdLine              cmdLine object
        """
        asqBuild.logger.trace('delTaskItem._fillCmdLine(aCmdLine)')
        asqBuild.logger.incDepthTrace()
        try :
          
          # builds a dummy command line
          aCmdLine.append('del')
          aCmdLine.appendQuoted(self.target)
          
        finally :
          asqBuild.logger.decDepthTrace()


  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # /class delTaskItem  - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


class delTaskList(delTask, asqBuildCore.taskList) :
    """\
      A list of delTaskItems, the list defines "default" attributes that
      each single item can override; if an item doesn't set an attribute,
      then the list's values will be used.
    """

    # task item class
    taskClass   = delTaskItem

    # Properties defined for that task type and its descendant.
    # Note : attribute names should be "lower case" only.
    attributes    = (
        'taskname',             # task name (should held section name)
        'count',                # task item count
        'tasktype',             # task type as a string
        'enabled',              # is task active ?
        'workdir',              # task base directory
        'comments',             # about this task
        
        #~ 'target',               # path patterns to remove
        #~ 'exclude',              # path patterns to keep
        
        'others'                # unrecognized attributes
      )

  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # /class delTaskList  - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
  print __file__
  print __doc__

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -