#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
Define a delphi command line compiler task item used by the
building utility.

Require
  Python        2.2

Constants
  string        dccTaskType

Classes
  class         dccTaskItem(bool enabled)
                  extends py_netasq.building.core::taskItem
  class         dccTaskList(bool enabled)
                  extends py_netasq.building.core::taskList
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.18 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  os

from    py_netasq.commonlib     import asqCommand, asqString, asqTypes

from    py_netasq               import building as asqBuild
from    py_netasq.building      import core     as asqBuildCore


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

dccTaskType = 'dccTask'

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class dccTask(asqBuildCore.task) :
    """\
      A simple task used by the automated building system; this class
      is used to compile delphi projects.
    """

    # class task type, nuff said :)
    classTaskType = dccTaskType

    # task type compatibility list
    compatibility = asqBuildCore.task.compatibility + (
        classTaskType,
      )

    # Properties defined for that task type and its descendant.
    # Note : attribute names should be "lower case" only.
    attributes    = (
        'taskname',             # task name (should held section name)
        'tasktype',             # task type as a string
        'enabled',              # is task active ?
        'workdir',              # task base directory
        'comments',             # about this task
        
        'compiler',             # path to compiler executable
        #~ 'project',              # path to delphi project (*.dpr)
        'cmdextra',             # extra application parameters
        'bindir',               # binary (exe/dll) ouput directory
        'dcudir',               # dcu output directory
        'cfgfile',              # path to project configuration file
        
        'mad_patch',            # path to madExceptPatch utility
        'mad_mesfile',          # path to .mes file
        'mad_mapfile',          # path to .map file
        'mad_extra',            # extra parameters
        
        'others'                # unrecognized attributes
      )

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _attributeCheck(self, status, name) :
        """\
          void _attributeCheck(object status, string name)
          
          Callback used by self.check(...)
          Checks the value of attribute; Appends error or warning
          messages to given check status.
          Should be overriden to set the behavior for children classes.
          
          param status                checkResult object
          param name                  attribute name
        """
        if 'compiler' == name :
          self._checkFilePath(
              status, name, 'delphi compiler executable',
              vital=isinstance(self, asqBuildCore.taskItem)
            )
        elif 'bindir' == name :
          self._checkDirPath(status, name, 'binary output directory')
        elif 'dcudir' == name :
          self._checkDirPath(status, name, 'dcu output directory')
        elif 'cfgfile' == name :
          self._checkFilePath(status, name, 'project configuration file')
        elif 'mad_patch' == name :
          self._checkFilePath(status, name, 'madExceptPatch executable')
        elif 'mad_mesfile' == name :
          self._checkFilePath(status, name, 'madExcept setting file')
        else :
          super(dccTask, self)._attributeCheck(status, name)


    def _attributeIsPath(self, name) :
        """\
          bool _attributeIsPath(string name)
          
          Callback used by self.completePaths(...)
          Returns True if attribute is a path, please note that the
          'workdir' attribute should NOT be considered as a path here.
          Should be overriden to set the behavior for children classes.
          
          param name                  attribute name
          return                      is attribute a path
        """
        if name in ('compiler', 'bindir', 'dcudir', 'cfgfile') :
          return True
        elif name in ('mad_patch', 'mad_mesfile', 'mad_mapfile') :
          return True
        else :
          return super(dccTask, self)._attributeIsPath(name)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def isValid(self) :
        """\
          bool isValid(void)
          
          Tells whether or not the task is "valid", id est, if mandatory
          attributes are set; an empty task is not valid, but an invalid
          task may not be empty ;)
          
          return                      is task valid
        """
        asqBuild.logger.trace('dccTask.isValid()')
        asqBuild.logger.incDepthTrace()
        try :
          
          if not super(dccTask, self).isValid() :
            return False
          elif not asqTypes.isNonEmptyInstance(self.compiler, basestring) :
            return False
          else :
            return True
          
        finally :
          asqBuild.logger.decDepthTrace()


  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # /class dccTask  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


class dccTaskItem(dccTask, asqBuildCore.taskItem) :
    """\
      A simple task used by the automated building system; this class
      is used to compile delphi projects.
      
      
      object                    patchCommand(void)
    """

    ant_name    = 'dcc32'

    # Properties defined for that task type and its descendant.
    # Note : attribute names should be "lower case" only.
    attributes  = (
        'taskname',             # task name (should held section name)
        'tasktype',             # task type as a string
        'enabled',              # is task active ?
        'workdir',              # task base directory
        'comments',             # about this task
        
        'compiler',             # path to compiler executable
        'project',              # path to delphi project (*.dpr)
        'cmdextra',             # extra application parameters
        'bindir',               # binary (exe/dll) ouput directory
        'dcudir',               # dcu output directory
        'cfgfile',              # path to project configuration file
        
        'mad_patch',            # path to madExceptPatch utility
        'mad_mesfile',          # path to .mes file
        'mad_mapfile',          # path to .map file
        'mad_extra',            # extra parameters
        
        'others'                # unrecognized attributes
      )

    # properties that could be overwritten by a "Merge" method
    mergeable   = (
        'workdir',
        'compiler',
        'cmdextra',
        'bindir',
        'dcudir',
        'cfgfile',
        'mad_patch',
        'mad_mesfile',
        'mad_mapfile',
        'mad_extra',
      )

    # we don't know the file name extension of the resulting binary,
    # but we can at least guess it.
    bin_exts    = ('.exe', '.dll', '.bpl')

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def isValid(self) :
        """\
          bool isValid(void)
          
          Tells whether or not the task is "valid", id est, if mandatory
          attributes are set; an empty task is not valid, but an invalid
          task may not be empty ;)
          
          return                      is task valid
        """
        asqBuild.logger.trace('dccTaskItem.isValid()')
        asqBuild.logger.incDepthTrace()
        try :
          
          if not super(dccTaskItem, self).isValid() :
            return False
          else :
            return asqTypes.isNonEmptyInstance(self.project, basestring)
          
        finally :
          asqBuild.logger.decDepthTrace()


    def patchCmdLine(self) :
        """\
          object patchCommand(void)
          
          Returns a command line as an CmdLine instance.
          
          return                      CmdLine instance
        """
        result = asqCommand.cmdLine()
        binary = self.__findBinary()
        
        if not asqTypes.isNonEmptyInstance(binary, basestring) :
          return result
        elif not asqTypes.isNonEmptyInstance(self.mad_patch, basestring) :
          return result
        
        result.appendQuoted(self.mad_patch)
        result.appendQuoted(binary)
        
        if asqTypes.isNonEmptyInstance(self.mad_mesfile, basestring) :
          result.appendQuoted(self.mad_mesfile)
        if asqTypes.isNonEmptyInstance(self.mad_mapfile, basestring) :
          result.appendQuoted(self.mad_mapfile)
        if asqTypes.isNonEmptyInstance(self.mad_extra, basestring) :
          result.appendQuoted(self.mad_extra)
        
        return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _attributeCheck(self, status, name) :
        """\
          void _attributeCheck(object status, string name)
          
          Callback used by self.check(...)
          Checks the value of attribute; Appends error or warning
          messages to given check status.
          Should be overriden to set the behavior for children classes.
          
          param status                checkResult object
          param name                  attribute name
        """
        if 'project' == name :
          self._checkFilePath(status, name, 'delphi project file', vital=True)
        else :
          super(dccTaskItem, self)._attributeCheck(status, name)


    def _attributeIsPath(self, name) :
        """\
          bool _attributeIsPath(string name)
          
          Callback used by self.completePaths(...)
          Returns True if attribute is a path, please note that the
          'workdir' attribute should NOT be considered as a path here.
          Should be overriden to set the behavior for children classes.
          
          param name                  attribute name
          return                      is attribute a path
        """
        if name in ('project', ) :
          return True
        else :
          return super(dccTaskItem, self)._attributeIsPath(name)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def antDisplayPatch(self) :
        """\
          string antDisplayPatch(void)
          
          Returns a short string explaining the current action: here the
          mad except binary patching.
          
          return                      task's action
        """
        chunks = list()
        binary = self.__findBinary()
        binary = asqString.shortenString(binary, 36, '...', '\\')
        
        chunks.append(self._antMargin())
        #~ chunks.append(self._antName())
        chunks.append('patching')
        chunks.append('`%s`' % (binary, ))
        
        return ' '.join(chunks)


    def _antAction(self) :
        """\
          string _antAction(void)
          
          Callback used by antDisplay()
          Returns a short string explaining the task's action.
          Should be overriden to set the behavior for children classes.
          Disabled and non valid tasks are already handled.
          
          return                      task's action
        """
        result = super(dccTaskItem, self)._antAction()
        
        # test if "result" is an empty string
        if not asqTypes.isNonEmptyInstance(result, basestring) :
          chunks = list()
          buffer = asqString.shortenString(self.project, 36, '...', '\\')
          
          chunks.append('compiling')
          chunks.append('`%s`' % (buffer, ))
          result = ' '.join(chunks)
        
        return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _fillCmdLine(self, aCmdLine) :
        """\
          void _fillCmdLine(object aCmdLine)
          
          Callback used by toCmdLine()
          Build the task command line, task is assumed valid.
          Should be overriden to set the behavior for children classes.
          
          param aCmdLine              cmdLine object
        """
        asqBuild.logger.trace('dccTaskItem._fillCmdLine(aCmdLine)')
        asqBuild.logger.incDepthTrace()
        try :
          
          # path to dcc32.exe
          aCmdLine.appendQuoted(self.compiler)
          # project file *.dpr
          aCmdLine.appendQuoted(self.project)
          # binary output directory
          if asqTypes.isNonEmptyInstance(self.bindir, basestring) :
            aCmdLine.appendQuoted('-E%s' % (self.bindir,))
          # dcu output directory
          if asqTypes.isNonEmptyInstance(self.dcudir, basestring) :
            aCmdLine.appendQuoted('-N%s' % (self.dcudir,))
          # extra compiler parameters
          if asqTypes.isNonEmptyInstance(self.cmdextra, basestring) :
            aCmdLine.append(self.cmdextra)
          
        finally :
          asqBuild.logger.decDepthTrace()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __findBinary(self) :
        """\
          var __findBinary(void)
          
          Try to guess the path of the compiled binary file. If a file is
          actually found, then the path is returned, else the None value
          is returned.
          
          return                      binary full path, or None
        """
        if not asqTypes.isNonEmptyInstance(self.project, basestring) :
          return None
        
        buffer = os.path.splitext( os.path.basename(self.project) )[0]
        if self.bindir is None :
          buffer = os.path.join(os.getcwd(), buffer)
        else :
          buffer = os.path.join(self.bindir, buffer)
        for ext in self.bin_exts :
          result = '%s%s' % (buffer, ext)
          if os.path.isfile(result) :
            return result
        else :
          return None


  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # /class dccTaskItem  - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


class dccTaskList(dccTask, asqBuildCore.taskList) :
    """\
      A list of dccTaskItems, the list defines "default" attributes that
      each single item can override; if an item doesn't set an attribute,
      then the list's values will be used.
    """

    # task item class
    taskClass   = dccTaskItem

    # Properties defined for that task type and its descendant.
    # Note : attribute names should be "lower case" only.
    attributes    = (
        'taskname',             # task name (should held section name)
        'count',                # task item count
        'tasktype',             # task type as a string
        'enabled',              # is task active ?
        'workdir',              # task base directory
        'comments',             # about this task
        
        'compiler',             # path to compiler executable
        #~ 'project',              # path to setup Factory project (*.dcc)
        'cmdextra',             # extra application parameters
        'bindir',               # binary (exe/dll) ouput directory
        'dcudir',               # dcu output directory
        'cfgfile',              # path to project configuration file
        
        'mad_patch',            # path to madExceptPatch utility
        'mad_mesfile',          # path to .mes file
        'mad_mapfile',          # path to .map file
        'mad_extra',            # extra parameters
        
        'others'                # unrecognized attributes
      )

  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # /class dccTaskList  - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -