#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
A delphi building utility: Wrapper around delphi compiler,
with custom config file support.

Require
  Python        2.2

Classes
  class         dccTaskHandler(object aTask, object logger)
                  extends py_netasq.building.core::taskHandler
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.8 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  os
import  shutil

from    py_netasq.commonlib     import asqPath

from    py_netasq               import building as asqBuild
from    py_netasq.building      import core     as asqBuildCore

import  tasks


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

# About *.cfg FILE (taken from "DELPHI in a nutshell") :
#
# The compiler gets options from four places. It checks all four sources,
# in order, so LATER sources can override earlier ones:
#
# - the global configuration file, 'dcc32.cfg', is in Delphi's 'bin'
#  directory. You can modify this file to store options that apply to all
#  projects. For example, you might use -u to list directories where
#  Delphi should look for *.dcu files.
#
# - the local configuration file, 'dcc32.cfg', is in the CURRENT directory.
#
# - the project configuration file resides in the same directory as the
#  project source file. Its filename is the project filename, with the
#  '.cfg' extension.
#  Note that the IDE automatically creates a '.cfg' file for each project,
#  saving the project's options in that configuration file.
#
# - you can override the options in the configuration fileS by specifying
#  additional or different options on the command line. The options can
#  precede or follow the source filename.
#
# If a config file is provided, then we need to back up any local file found.

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class dccTaskHandler(asqBuildCore.taskHandler) :
    """\
      Wrapper around delphi compiler, with custom config file
      support. Settings have to be set through object's "task"
      (class dccTaskItem) attribute.
    """

    # task class
    taskClass = tasks.dccTaskItem

    # format string : "<project>.cfg"
    FMT_CFG_FILE = '%s.cfg'
    # format string : "_<project>_cfg.bak"
    FMT_BAK_FILE = '_%s_cfg.bak'

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __init__(self, aTask=None) :
        """\
          constructor taskHandler(object aTask, object logger)
          
          Create a new object intended to ease command line execution.
          
          param aTask                 task to execute.
                                       defaults to None
          param logger                output logger
                                       defaults to None
        """
        super(dccTaskHandler, self).__init__(aTask)
        # path/to/project/<project>.cfg
        self._cfgFile = None
        # path/to/project/_<project>_cfg.bak
        self._cfgBack = None


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def debug(self) :
        """\
          void debug(void)
          
          Prints out the main attributes values through the logging
          utility.
        """
        self._debugKeyVal('_cfgFile', self._cfgFile)
        self._debugKeyVal('_cfgBack', self._cfgBack)
        super(dccTaskHandler, self).debug()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _setup(self, *arglist, **argdict) :
        """\
          void _setup(*arglist, **argdict)
          
          Called right before the actual command execution; should be
          overidden to set up the execution environment.
          
          - Paths found in project's dpr are RELATIVE to that peculiar file,
           we have to execute the build command FROM THE PROJECT'S DIRECTORY.
          - If a custom config file (cfgFile) is supplied it must be placed
           as '<project>.cfg' in project's directory, overridding the
           project's one (if existing), and any 'dcc32.cfg' file.
          
          param *arglist              extra parameters.
          param **argdict             extra named parameters.
        """
        asqBuild.logger.trace('dccTaskHandler._setup()')
        asqBuild.logger.incDepthTrace()
        try :
          
          super(dccTaskHandler, self)._setup(*arglist, **argdict)
          # path resolutions
          prjPath, prjName = os.path.split(self._task.project)
          # config and project files must share the
          # same name (minus file extensions)
          prjName = os.path.splitext(prjName)[0]
          
          self._cfgFile = os.path.join(prjPath, self.FMT_CFG_FILE % (prjName, ))
          self._cfgBack = os.path.join(prjPath, self.FMT_BAK_FILE % (prjName, ))
          
          # go to the project source directory
          self._pushCwd(prjPath)
          # sets the config file
          self._backupCfgFile()
         
          # ensures that bin and dcu dirs do exist.
          if self._task.bindir is not None :
            asqPath.makeDir(self._task.bindir)
          if self._task.dcudir is not None :
            asqPath.makeDir(self._task.dcudir)
         
        finally :
          asqBuild.logger.decDepthTrace()


    def _teardown(self, *arglist, **argdict) :
        """\
          void _teardown(*arglist, **argdict)
          
          Called right after the actual command execution; should be
          overidden to clean up the execution environment.
          
          - delete copied custom config file (if any)
          - restore local config file (if any)
          - change back working directory.
          
          param *arglist              extra parameters.
          param **argdict             extra named parameters.
        """
        asqBuild.logger.trace('dccTaskHandler._teardown()')
        asqBuild.logger.incDepthTrace()
        try :
          
          self._restoreCfgFile()
          self._popCwd()
          super(dccTaskHandler, self)._teardown(*arglist, **argdict)
         
        finally :
          asqBuild.logger.decDepthTrace()


    def _execute(self, cmdStr, raw, *arglist, **argdict) :
        """\
          var _execute(string cmdStr, bool raw, *arglist, **argdict)
          
          Execute inner command line and return the shell status.
          Each line from stdout and stderr will be decoded and processed,
          i.e. lines will go through _decodeOutput() and _processOutput().
          
          param cmdStr                command as a string to execute.
          param raw                   disable output formatting
          param *arglist              extra parameters.
          param **argdict             extra named parameters.
          return                      shell status
        """
        asqBuild.logger.trace('dccTaskHandler._execute(%s, %s)', cmdStr, str(raw))
        asqBuild.logger.incDepthTrace()
        try :
          
          result = super(dccTaskHandler, self)._execute(cmdStr, raw, *arglist, **argdict)
          if result is None :
            # call mad except "patch.exe" over binary file
            buffer = str(self._task.patchCmdLine()).strip()
            if 0 < len(buffer) :
              self._emitMsg(self._task.antDisplayPatch())
              result = super(dccTaskHandler, self)._execute(buffer, raw, *arglist, **argdict)
          
          return result
         
        finally :
          asqBuild.logger.decDepthTrace()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _backupCfgFile(self) :
        """\
          void _backupCfgFile(void)
          
          If a custom config file is provided, it must be copied to
          project's directory; To avoid file over writting, existing local
          config file is renamed.
        """
        if self._task.cfgfile is None :
          return
        # maybe given custom cfg file is actually project's local cfg file !!
        elif self._cfgFile == self._task.cfgfile :
          self._cfgFile = None
          return
        # if a file already exists it will be renamed
        # existing file MUST NOT be overwritten
        if os.path.isfile(self._cfgFile) :
          os.rename(self._cfgFile, self._cfgBack)
        shutil.copy(self._task.cfgfile, self._cfgFile)


    def _restoreCfgFile(self) :
        """\
          void _restoreCfgFile(void)
          
          If a custom config file is provided, then it might have been
          copied to project's directory; It will be deleted, and backed up
          local config (if any) will be restored.
        """
        if self._task.cfgfile is None :
          return
        elif self._cfgFile is None :
          return
        # restore previous config file.
        if os.path.isfile(self._cfgFile) :
          os.remove(self._cfgFile)
        if os.path.isfile(self._cfgBack) :
          os.rename(self._cfgBack, self._cfgFile)


  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # /class dccTaskHandler - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -