#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
Define a "CVS Task item" list used by the building utility.

Require
  Python        2.2

Constants
  string        cvsTaskType

Classes
  class         cvsTaskItem(bool enabled)
                  extends py_netasq.building.core::taskItem
  class         cvsTaskList(bool enabled)
                  extends py_netasq.building.core::taskList
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.19 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  os

from    py_netasq.commonlib     import asqTypes

from    py_netasq               import building as asqBuild
from    py_netasq.building      import core     as asqBuildCore


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

cvsTaskType = 'cvsTask'

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class cvsTask(asqBuildCore.task) :
    """\
      A CVS task: a cvs command, which can be either 'checkout',
      'export' or 'update'.
    """

    # class task type, nuff said :)
    classTaskType = cvsTaskType

    # task type compatibility list
    compatibility = asqBuildCore.task.compatibility + (
        classTaskType,
      )

    # Properties defined for that task type and its descendant.
    # Note : attribute names should be "lower case" only.
    attributes    = (
        'taskname',             # task name (should held section name)
        'tasktype',             # task type as a string
        'enabled',              # is task active ?
        'workdir',              # task base directory
        'comments',             # about this task
        
        'client',               # cvs client (exe) to use
        #~ 'module',               # module in cvs tree
        'command',              # cvs command to perform
        'cmdextra',             # extra cvs CLIENT parameters
        'cvsroot',              # CVSROOT
        'cvsrsh',               # CVS_RSH environement value
        'destdir',              # retrieved files destination
        'notafter',             # revision date limit
        'tag',                  # module tag
        
        'others'                # unrecognized attributes
      )

    # CVS tasks can only RETRIEVE files
    allowed_commands = ('checkout', 'export', 'update')

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def isValid(self) :
        """\
          bool isValid(void)
          
          Tells whether or not the task is "valid", id est, if mandatory
          attributes are set; an empty taskItem is not valid, but an
          invalid taskItem may not be empty ;)
          
          return                      is task valid
        """
        asqBuild.logger.trace('cvsTask.isValid()')
        asqBuild.logger.incDepthTrace()
        try :
          
          if not super(cvsTask, self).isValid() :
            return False
          elif not asqTypes.isNonEmptyInstance(self.client, basestring) :
            return False
          elif not asqTypes.isNonEmptyInstance(self.command, basestring) :
            return False
          else :
            return (self.command in self.allowed_commands)
          
        finally :
          asqBuild.logger.decDepthTrace()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _checkCommand(self, status, name, label, vital=False) :
        """\
          void _checkCommand(object status, string name, string label, bool vital)
          
          Helper function used by _attributeCheck(). Provided to reduce
          redundancy. If "vital" is set to True, then ANY abnormal setting
          will produce errors instead of warnings.
          
          param status                checkResult object
          param name                  attribute name
          param label                 attribute formal name
          param vital                 is attribute setting vital ?
                                        defaults to False
        """
        buffer = self.__dict__[name]
        if buffer is None :
          if vital :
            status.addMsgError('%-15s %s is not set'   % (name, label))
          else :
            status.addMsgWarning('%-15s %s is not set'   % (name, label))
        elif not asqTypes.isNonEmptyInstance(buffer, basestring) :
          status.addMsgError('%-15s %s is invalid'  % (name, label))
        elif buffer not in self.allowed_commands :
          status.addMsgError('%-15s %s not allowed' % (name, label))


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _attributeCheck(self, status, name) :
        """\
          void _attributeCheck(object status, string name)
          
          Callback used by self.check(...)
          Checks the value of attribute; Appends error or warning
          messages to given check status.
          Should be overriden to set the behavior for children classes.
          
          param status                checkResult object
          param name                  attribute name
        """
        if 'client' == name :
          self._checkFilePath(
              status, name, 'cvs client executable',
              vital=isinstance(self, asqBuildCore.taskItem)
            )
        elif 'command' == name :
          self._checkCommand(
              status, name, 'cvs command',
              vital=isinstance(self, asqBuildCore.taskItem)
            )
        elif 'destdir' == name :
          self._checkDirPath(status, name, 'destination directory')
        else :
          super(cvsTask, self)._attributeCheck(status, name)


    def _attributeIsPath(self, name) :
        """\
          bool _attributeIsPath(string name)
          
          Callback used by self.completePaths(...)
          Returns True if attribute is a path, please note that the
          'workdir' attribute should NOT be considered as a path here.
          Should be overriden to set the behavior for children classes.
          
          param name                  attribute name
          return                      is attribute a path
        """
        if name in ('client', 'destdir', ) :
          return True
        else :
          return super(cvsTask, self)._attributeIsPath(name)


  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # /class csvTask  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


class cvsTaskItem(cvsTask, asqBuildCore.taskItem) :
    """\
      A CVS task: a cvs command, which can be either 'checkout',
      'export' or 'update'.
    """

    ant_name    = 'cvs'

    # Properties defined for that task type and its descendant.
    # Note : attribute names should be "lower case" only.
    attributes  = (
        'taskname',             # task name (should held section name)
        'tasktype',             # task type as a string
        'enabled',              # is task active ?
        'workdir',              # task base directory
        'comments',             # about this task
        
        'client',               # cvs client (exe) to use
        'module',               # module in cvs tree
        'command',              # cvs command to perform
        'cmdextra',             # extra cvs CLIENT parameters
        'cvsroot',              # CVSROOT
        'cvsrsh',               # CVS_RSH environement value
        'destdir',              # retrieved files destination
        'notafter',             # revision date limit
        'tag',                  # module tag
        
        'others'                # unrecognized attributes
      )

    # properties that could be overwritten by a "Merge" method
    mergeable   = (
        'workdir',
        'client',
        'command',
        'cmdextra',
        'cvsroot',
        'cvsrsh',
        'destdir',
        'notafter',
        'tag',
      )

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def isValid(self) :
        """\
          bool isValid(void)
          
          Tells whether or not the task is "valid", id est, if mandatory
          attributes are set; an empty taskItem is not valid, but an
          invalid taskItem may not be empty ;)
          
          return                      is task valid
        """
        asqBuild.logger.trace('cvsTaskItem.isValid()')
        asqBuild.logger.incDepthTrace()
        try :
          
          if not super(cvsTaskItem, self).isValid() :
            return False
          else :
            return asqTypes.isNonEmptyInstance(self.module, basestring)
          
        finally :
          asqBuild.logger.decDepthTrace()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _attributeCheck(self, status, name) :
        """\
          void _attributeCheck(object status, string name)
          
          Callback used by self.check(...)
          Checks the value of attribute; Appends error or warning
          messages to given check status.
          Should be overriden to set the behavior for children classes.
          
          param status                checkResult object
          param name                  attribute name
        """
        if 'module' == name :
          self._checkValue(status, name, 'cvs module', vital=True)
        else :
          super(cvsTaskItem, self)._attributeCheck(status, name)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _antAction(self) :
        """\
          str _antAction(void)
          
          Callback used by antDisplay()
          Returns a short string explaining the task's action.
          Should be overriden to set the behavior for children classes.
          Disabled and non valid tasks are already handled.
          
          return                      task's action
        """
        result = super(cvsTaskItem, self)._antAction()
        
        # test if "result" is an empty string
        if not asqTypes.isNonEmptyInstance(result, basestring) :
          chunks = list()
          
          chunks.append(self.command)
          chunks.append('`%s`' % (self.module, ))
          if asqTypes.isNonEmptyInstance(self.tag, basestring) :
            chunks.append('-r')
            chunks.append(self.tag)
          result = ' '.join(chunks)
        
        return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _fillCmdLine(self, aCmdLine) :
        """\
          void _fillCmdLine(object aCmdLine)
          
          Callback used by toCmdLine()
          Build the task command line, task is assumed valid.
          Should be overriden to set the behavior for children classes.
          
          param aCmdLine              cmdLine object
        """
        asqBuild.logger.trace('cvsTaskItem._fillCmdLine(aCmdLine)')
        asqBuild.logger.incDepthTrace()
        try :
          
          # cvs executable
          aCmdLine.appendQuoted(self.client)
          if asqTypes.isNonEmptyInstance(self.cvsroot, basestring) :
            # fully qualified cvs root
            aCmdLine.append('-d')
            aCmdLine.appendQuoted(self.cvsroot)
          if asqTypes.isNonEmptyInstance(self.cmdextra, basestring) :
            # cvs extra parameters
            aCmdLine.append(self.cmdextra)
          # cvs command
          aCmdLine.append(self.command)
          if asqTypes.isNonEmptyInstance(self.notafter, basestring) :
            # retrieve revisions upto 'notAfter' date (not after)
            aCmdLine.append('-D')
            aCmdLine.appendQuoted(self.notafter)
          if asqTypes.isNonEmptyInstance(self.tag, basestring) :
            aCmdLine.append('-r')
            aCmdLine.append(self.tag)
          elif self.command in ('update', 'checkout') :
            # reset sticky tag
            aCmdLine.append('-A')
          
          if 'update' != self.command :
            aCmdLine.appendQuoted(self.module)
          else :
            # create directory but skip empty ones
            # module name should not be present
            aCmdLine.append('-dP')
          
        finally :
          asqBuild.logger.decDepthTrace()


  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # /class cvsTaskItem  - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


class cvsTaskList(cvsTask, asqBuildCore.taskList) :
    """\
      A list of cvsTaskItems, the list defines "default" attributes that
      each single item can override; if an item doesn't set an attribute,
      then the list's values will be used.
    """

    # task item class
    taskClass   = cvsTaskItem

    # Properties defined for that task type and its descendant.
    # Note : attribute names should be "lower case" only.
    attributes    = (
        'taskname',             # task name (should held section name)
        'count',                # task item count
        'tasktype',             # task type as a string
        'enabled',              # is task active ?
        'workdir',              # task base directory
        'comments',             # about this task
        
        'client',               # cvs client (exe) to use
        #~ 'module',               # module in cvs tree
        'command',              # cvs command to perform
        'cmdextra',             # extra cvs CLIENT parameters
        'cvsroot',              # CVSROOT
        'cvsrsh',               # CVS_RSH environement value
        'destdir',              # retrieved files destination
        'notafter',             # revision date limit
        'tag',                  # module tag
        
        'others'                # unrecognized attributes
      )

  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # /class cvsTaskList  - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -