#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
Define a "build file" parsing class used by the building utility.

Require
  Python        2.2

Classes
  class         fileParser(dict defaults)
                  extends py_netasq.commonlib.asqIniParser::asqIniParser
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.40 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  ConfigParser
import  logging
import  os
import  sys

from    py_netasq.commonlib     import asqIniParser, asqLogging
from    py_netasq.commonlib     import asqConvert, asqMapping, asqPath
from    py_netasq.commonlib     import asqSequence, asqTypes

from    py_netasq               import building as asqBuild
from    py_netasq.building      import core     as asqBuildCore
from    py_netasq.building      import tasks    as asqBuildTasks


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

LOGGER_NAME             = 'BuildLogger'

# minimum version expected
VERSION_MIN             = ('1', '0')
# maximum version expected
VERSION_MAX             = ('5', '0')
# handled versions range
VERSION_RANGE           = ( VERSION_MIN , VERSION_MAX )


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

# token definitions - section headers
SECT_CONFIG             = "Config"
SECT_DEFAULT            = "DEFAULT"
SECT_ENVIRONMENT        = "ENVIRONMENT"
SECT_FILEVERSION        = "FileVersion"
SECT_FINALIZE           = "Finalize"
SECT_INITIALIZE         = "Initialize"
SECT_LOGGING            = "Logging"
SECT_MAILING            = "Mailing"

# token definitions - section options
TOKN_ENABLED            = "enabled"
TOKN_LOCK_FILE          = "lock_file"
TOKN_LOG_FILE           = "filename"
TOKN_LOG_HANDLER        = "handler"
TOKN_LOG_MAXSIZE        = "maxsize"
TOKN_LOG_MAXFILE        = "maxfile"
TOKN_TASKS_ORDER        = "tasks_order"
TOKN_TASKTYPE           = "tasktype"
TOKN_VERSION            = "version"



SECT_FILEVERSION_DEFAULTS = {
    TOKN_VERSION        : '2.0',
  }

SECT_CONFIG_DEFAULTS      = {
    TOKN_TASKS_ORDER    : 'CvsTasks, VrsTasks, ResTasks, DccTasks, Sf6Tasks',
    TOKN_LOCK_FILE      : '',
  }

SECT_LOGGING_DEFAULTS     = {
    TOKN_ENABLED        : 'False',
    TOKN_LOG_FILE       : '',
    TOKN_LOG_HANDLER    : 'RotatingFileHandler',
    TOKN_LOG_MAXSIZE    : '1M',
    TOKN_LOG_MAXFILE    : '1',
  }


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class _sectionProxy(object) :
    """\
      Defines a "proxy object" which gathers all options from a single
      *.ini section. read/write operations goes through that object.
      
      
      constructor       _sectionProxy(
                            object parser, string section, dict defaults
                          )
      
      void              debug(void)
    """

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __init__(self, parser, section, defaults={}) :
        """\
          constructor _sectionProxy(object parser, string section, dict defaults)
          
          '_sectionProxy' contructor, sets the vital properties. When a
          requested option is not found in the config parser data set, then
          the 'defaults' dictionary is checked; if option is missing from
          the dict, then a ConfigParser.NoOptionError is raised.
          
          param parser                file parser object
          param section               section name
          param defaults              default values
        """
        if parser is None :
          raise TypeError('A valid parser object must be provided')
        if not asqTypes.isNonEmptyInstance(section, basestring) :
          raise TypeError('A valid section name must be provided')
        
        self._parser    = parser
        self._section   = section
        self._defaults  = defaults.copy()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __getattr__(self, name) :
        """\
          var __getattr__(string name)
         
          Special method - Called when an attribute lookup has not found
          the attribute in the usual places (i.e. it is not an instance
          attribute nor is it found in the class tree for self). name is
          the attribute name. This method should return the (computed)
          attribute value or raise an AttributeError exception.
          
          param name                  requested attribute
          return                      attribute value
        """
        asqBuild.logger.trace('_sectionProxy.__getattr__(name=%s)', repr(name)) 
        asqBuild.logger.incDepthTrace()
        try :
          
          try :
            result = self._parser.get(self._section, name)
          except (ConfigParser.NoSectionError, ConfigParser.NoOptionError), e :
            if name in self._defaults :
              return self._defaults[name]
            else :
              raise ConfigParser.NoOptionError(), None, sys.exc_info()[2]
          else :
            return result
          
        finally :
          asqBuild.logger.decDepthTrace()


    def __setattr__(self, name, value) :
        """\
          void __setattr__(string name, var value)
        
          Called when an attribute assignment is attempted. This is called
          instead of the normal mechanism (i.e. store the value in the
          instance dictionary). name is the attribute name, value is the
          value to be assigned to it.
        
          param name                  attribute to set
          param value                 attribute value
        """
        asqBuild.logger.trace('_sectionProxy.__setattr__(name=%s, value)', repr(name)) 
        asqBuild.logger.incDepthTrace()
        try :
          
          if name.startswith('_') :
            super(_sectionProxy, self).__setattr__(name, value)
          else :
            # ensures that section actually exists within parser dataset
            self._parser.addSection(self._section)
            self._parser.setOption(self._section, name, value, '')
          
        finally :
          asqBuild.logger.decDepthTrace()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _debugKeyVal(self, key, val) :
        """\
          void _debugKeyVal(string key, var val)
          
          Formats and prints out the given key/value duo. Ensures
          display consistency.
          
          param key                   value name
          param val                   value to display
        """
        asqBuild.logger.debug('%-15s %s', key, repr(val))


    def debug(self) :
        """\
          void debug(void)
          
          Prints out the main attributes values through the logging
          utility.
        """
        values = self.items(raw=True)
        
        #~ asqBuild.logger.debug('[%s]', self._section)
        for key in asqSequence.sort(None, values.keys()) :
          self._debugKeyVal(key, values[key])


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def items(self, raw=True) :
        """\
          dict items(bool raw)
          
          Return a dictionary { name: value, } of all options found in
          current section, plus default values expected in the section.
          
          param raw                   no interpolation,
                                        defaults to True
          return                      section's options as a dict
        """
        result = self._defaults
        
        try :
          buffer = self._parser.getSectionDict(self._section, full=False, raw=raw)
        except ConfigParser.NoSectionError, e :
          pass
        else :
          result.update(buffer)
        
        return result


  # /class _sectionProxy  - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


class fileParser(asqIniParser.asqIniParser) :
    """\
      A custom ConfigParser descendant. Creates taskItems and taskLists
      from the parsed *.ini sections. 'config' and 'fileVersion' properties
      are available as '_sectionProxy' instances.
      
      
      constructor       fileParser(dict defaults)
      
      void              debug(void)
      
      void              clear(void)
      void              clearSection(string section)
      
      property list     fileComments    list of comments
      property list     ignored         ignored sections list
      
      property object   logger          logger bound to the file
      
      
      property dict     settings        list of settings sections
      property dict     taskLists       list of taskLists
      property dict     taskItems       list of taskItems
    """


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def projectHelp(cls, filename, display) :
        """\
          static void projectHelp(string filename, object display)
          
          Displays comments for each task lists and items from the
          configuration file 'filename'.
          
          param filename              config file path
          param display               logger object
        """
        filename = asqPath.normalize(os.path.abspath(filename))
        aParser  = fileParser(None)
        
        aParser.readFile(filename)
        
        for item in aParser.fileComments :
          display.log(logging.INFO, item)


    # set class methods
    projectHelp = classmethod(projectHelp)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __init__(self, defaults=None) :
        """\
          constructor fileParser(dict defaults)
          
          Create a new build file parser, 'defaults' dictitonary will be
          used to set the "DEFAULT" section's options; those options can be
          used as variables.
          
          param defaults              DEFAULT options, defaults to None
        """
        super(fileParser, self).__init__(defaults)
        
        # as logger is not set, the property getter will return a new one
        # set logger.propagate to False : Do not pass messages to ancestors
        self.logger.propagate = False
        # inner logging handler instance
        self._logHandler = None
        # path to lock file, only one build may be done at once
        self._lockPath   = None
        
        # os specific environment section
        self._environment= None
        
        # sections that can't be handled as task objects
        self._settings   = dict()
        # list of taskItems (ex: Mailing, ...) instances
        self._taskItems  = dict()
        # list of taskLists (ex: Initialize, ...) instances
        self._taskLists  = dict()
        # list of ignored sections names
        self._ignored    = list()
        
        # default values for [Config], [FileVersion] and [Logging] sections
        self._settings[SECT_CONFIG      ] = _sectionProxy(self, SECT_CONFIG      , SECT_CONFIG_DEFAULTS)
        self._settings[SECT_FILEVERSION ] = _sectionProxy(self, SECT_FILEVERSION , SECT_FILEVERSION_DEFAULTS)
        self._settings[SECT_LOGGING     ] = _sectionProxy(self, SECT_LOGGING     , SECT_LOGGING_DEFAULTS)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _debugKeyVal(self, key, val) :
        """\
          void _debugKeyVal(string key, var val)
          
          Formats and prints out the given key/value duo. Ensures
          display consistency.
          
          param key                   value name
          param val                   value to display
        """
        asqBuild.logger.debug('%-15s %s', key, repr(val))


    def debug(self) :
        """\
          void debug(void)
          
          Prints out the main attributes values through the logging
          utility.
        """
        self._debugKeyVal('ignored', self.ignored)
        asqBuild.logger.debug('--')
        asqBuild.logger.debug('')
        
        asqBuild.logger.debug('[%s]', ConfigParser.DEFAULTSECT)
        for (key, val) in self.defaults().iteritems() :
          self._debugKeyVal(key, val)
        asqBuild.logger.debug('')
        asqBuild.logger.debug('')
        
        if self._environment is not None :
          asqBuild.logger.debug('[%s]', SECT_ENVIRONMENT)
          for (key, val) in self.environment.iteritems() :
            self._debugKeyVal(key, val)
          asqBuild.logger.debug('')
          asqBuild.logger.debug('')
        
        for section in self._settings.iterkeys() :
          try :
            buffer = self._settings[section]
          except KeyError, e :
            continue
          else :
            asqBuild.logger.debug('[%s]', section)
            buffer.debug()
            asqBuild.logger.debug('')
            asqBuild.logger.debug('')
        
        for section in self._taskItems.iterkeys() :
          try :
            buffer = self._taskItems[section]
          except KeyError, e :
            continue
          else :
            asqBuild.logger.debug('[%s]', section)
            buffer.debug()
            asqBuild.logger.debug('')
            asqBuild.logger.debug('')
        
        for section in self._getTasksOrderList() :
          try :
            buffer = self._taskLists[section]
          except KeyError, e :
            continue
          else :
            asqBuild.logger.debug('[%s]', section)
            buffer.debug()
            asqBuild.logger.debug('')
            asqBuild.logger.debug('')


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def optionxform(self, optionstr) :
        """\
          string optionxform(string optionstr)
        
          Transforms the option name 'optionstr' as found in an input
          file or as passed in by client code to the form that should
          be used in the internal structures.
          
          If the first character of the given string is not an alphabetic
          char, then the given string is simply returned, else the
          parent implementation is used.
          
          The default implementation returns a lower-case version of
          'optionstr'
          
          param optionstr             option name string found in file
          return                      option name to use internally
        """
        if 0 == len(optionstr) :
          return optionstr
        elif not optionstr[0].isalpha() :
          return optionstr
        else :
          return super(fileParser, self).optionxform(optionstr)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def clear(self) :
        """\
          void clear(void)
          
          Empty parser dataset, all options and sections are trashed.
        """
        # super.clear() automatically resets the _settings items
        super(fileParser, self).clear()
        self._taskItems.clear()
        self._taskLists.clear()
        self._environment = None
        self._ignored[:]  = []


    def clearSection(self, section=None) :
        """\
          void clearSection(string section)
          
          Empty a TASK section (items or list), or all sections if no
          section name is provided. If a 'section' is given, but does
          not exist then nothing happens.
          Section will do remain, only options will be trashed.
          
          param section               section name, defaults to None
        """
        # should also clear the items of the 'settings' dictionary
        super(fileParser, self).clearSection(section)
        
        if section is None :
          for key in self._taskItems.iterkeys() :
            self._taskItems[key].reset()
          for key in self._taskLists.iterkeys() :
            self._taskLists[key].reset()
        elif section in self._taskItems :
          self._taskItems[section].reset()
        elif section in self._taskLists :
          self._taskLists[section].reset()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def updateDefaults(self, aDict, default='') :
        """\
          void updateDefaults(dict aDict, var default)
          
          Update instance wide defaults values with provided 'aDict'
          dictionary; if a value of given dictionary is found to be None,
          then it is replaced with the 'default' value.
          
          e.g. :
            defaults()
            => foo = bar
               ham = ha
          
            aDict = { 'foo': 'oof', 0: None, 1: 'mr' }
            updateDefaults(aDict, '<none>')
            => foo = oof
               ham = ha
               0   = <none>
               1   = mr
          
          param aDict                 new value
          param default               default value, defaults to ''
        """
        super(fileParser, self).updateDefaults(aDict, default)
        # everything should be updated
        self._handleDataSet()


    def updateSections(self, section, aDict, default='') :
        """\
          void updateSection(string section, dict aDict, var default)
          
          Update 'section' values with provided 'aDict' dictionary;
          if a value of given dictionary is found to be None, then
          it is replaced with the 'default' value.
          
          param section               section name
          param aDict                 new value
          param default               default value, defaults to ''
        """
        super(fileParser, self).updateSections(section, aDict, default)
        # everything should be updated
        self._handleDataSet()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _read(self, fp, fpname):
        """\
          void _read(object fp, string fpname)
          
          Parse a sectioned setup file.
          The sections in setup file contains a title line at the top,
          indicated by a name in square brackets (`[]'), plus key/value
          options lines, indicated by `name: value' format lines.
          Continuations are represented by an embedded newline then
          leading whitespace.  Blank lines, lines beginning with a '#',
          and just about everything else are ignored.
          
          param fp                    file object
          param fpname                file name
        """
        asqBuild.logger.trace('fileParser._read(fp, fpname=%s)', repr(fpname)) 
        asqBuild.logger.incDepthTrace()
        try :
          
          super(fileParser, self)._read(fp, fpname)
          self._handleDataSet()
          
        finally :
          asqBuild.logger.decDepthTrace()


    def _handleDataSet(self) :
        """\
          void _handleDataSet(void)
         
          Handles the setup file dataset, normalizes dataset and
          extracts task items and lists.
          This method is automatically called whenever file is read
          (through '_read()' or 'readFile()').
        """
        asqBuild.logger.trace('fileParser._handleDataSet(void)') 
        asqBuild.logger.incDepthTrace()
        try :
          
          # no need to consider the sections from the settings dictionary
          # thanks to the _sectionProxy class, the dataset is DIRECTLY read
          self._taskLists.clear()
          self._taskItems.clear()
          
          if self._testFileVersion() :
            self._environment = self._extractEnvironment(False)
            self._taskLists   = self._extractTaskLists(False)
            self._taskItems   = self._extractTaskItems(False)
          
          # updates the ignored section list
          self._updateIgnored()
          
        finally :
          asqBuild.logger.decDepthTrace()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _testFileVersion(self) :
        """\
          bool _testFileVersion(void)
          
          Tells wether or not the file version is in the allowed versions
          range.
          
          return                      True if file version is accepted
        """
        buffer = tuple(self._settings[SECT_FILEVERSION].version.split('.'))
        return VERSION_RANGE[0] <= buffer <= VERSION_RANGE[1]


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _updateIgnored(self) :
        """\
          void _updateIgnored(void)
          
          Updates the 'ignored' sections property.
        """
        parsed = list()
        parsed.extend(self._settings.keys())
        parsed.extend(self._taskItems.keys())
        if self._environment is not None :
          parsed.append(SECT_ENVIRONMENT)
        
        for key in self._taskLists.iterkeys() :
          parsed.append(key)
          parsed.extend([ x.taskname for x in self._taskLists[key] ])
        
        # DEFAULT is not included in the sections() list
        self._ignored[:] = asqSequence.difference(self.sections(), parsed)
        self._ignored.sort()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _extractEnvironment(self, raw=False) :
        """\
          dict _extractEnvironment(bool raw)
          
          Returns the 'ENVIRONMENT' section from the dataset. If that
          section is actually missing, then None may be returned.
          
          param raw                   extract with no interpolation,
                                        defaults to False
          return                      environment dictionary
        """
        try :
          return self.getSectionDict(SECT_ENVIRONMENT, False, raw)
        except ConfigParser.NoSectionError, e :
          return None


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _extractTaskLists(self, raw=False) :
        """\
          dict _extractTaskLists(bool raw)
          
          Set the 'taskLists' dictionary according to the task lists
          actually found in the current dataset.
          Parses 'Initialize' and 'Finalize' sections, plus those listed
          in the 'Config.tasks_order'.
          
          param raw                   extract with no interpolation,
                                        defaults to False
          return                      task lists dictionary
        """
        result = dict()
        
        result[SECT_INITIALIZE] = self._parseTaskList(
            SECT_INITIALIZE, asqBuildTasks.rawTaskType, raw
          )
        result[SECT_FINALIZE  ] = self._parseTaskList(
            SECT_FINALIZE  , asqBuildTasks.rawTaskType, raw
          )
        for section in self._getTasksOrderList() :
          tkType = self._guessTaskType(section)
          result[section] = self._parseTaskList(section, tkType, raw)
        
        # remove None items from the dictionary
        return asqMapping.filterDict(None, result)


    def _extractTaskItems(self, raw=False) :
        """\
          dict _extractTaskItems(bool raw)
          
          Set the 'taskLists' dictionary according to the task lists
          actually found in the current dataset.
          Parses 'Mailing' section.
          
          param raw                   extract with no interpolation,
                                        defaults to False
          return                      task items dictionary
        """
        result = dict()
        
        result[SECT_MAILING] = self._parseTaskItem(
            SECT_MAILING, asqBuildTasks.mailTaskType, raw
          )
        
        # remove None items from the dictionary
        return asqMapping.filterDict(None, result)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _parseTaskItem(self, section, defaultType=None, raw=False) :
        """\
          object _parseTaskItem(string section, string defaultType, bool raw)
          
          Builds a task item object from the given section; None may be
          returned if no valid task type is found (or assumed).
          
          param section               section name
          param defaultType           assumed task type, defaults to None
          param raw                   extract with no interpolation,
                                        defaults to False
          return                      a taskItem object, or None
        """
        try :
          buffer = self.getSectionDict(section, full=False, raw=raw)
        except ConfigParser.NoSectionError, e :
          result = None
        else :
          result = asqBuildTasks.factory.taskItemFromDict(buffer, defaultType)
          result.taskname = section
        return result


    def _parseTaskList(self, section, defaultType=None, raw=False) :
        """\
          object _parseTaskList(string section, string defaultType, bool raw)
          
          Builds a task list object from the given section; None may be
          returned if no valid task type is found (or assumed). Sub-tasks
          are added as task list's items, and are NOT merged with the
          task list stub !
          
          param section               section name
          param defaultType           assumed task type, defaults to None
          param raw                   extract with no interpolation,
                                        defaults to False
          return                      a taskList object, or None
        """
        result = self._parseTaskListStub(section, defaultType, raw)
        if not isinstance(result, asqBuildCore.taskList) :
          result = None
        else :
          length = len(result)
          header = self._guessTaskListItemsHeader(section)
          result[:] = self._parseTaskListItems(header, defaultType, length, raw)
        return result


    def _parseTaskListStub(self, section, defaultType, raw) :
        """\
          object _parseTaskListStub(string section, string defaultType, bool raw)
          
          Builds a task list stub object from the given section; None may
          be returned if no valid task type is found (or assumed). Sub-tasks
          are not considered !
          
          param section               section name
          param defaultType           assumed task type
          param raw                   extract with no interpolation,
          return                      a taskList object, or None
        """
        try :
          buffer = self.getSectionDict(section, full=False, raw=raw)
        except ConfigParser.NoSectionError, e :
          result = None
        else :
          result = asqBuildTasks.factory.taskListFromDict(buffer, defaultType)
          result.taskname = section
        return result


    def _parseTaskListItems(self, section, defaultType, limit, raw) :
        """\
          list _parseTaskListItems(string section, string defaultType, int limit, bool raw)
          
          Extracts task items found under the given 'section' name (which
          should be something like "FooTasks_%d"). The given 'defaultType'
          is used to parse the items. If 'limit' equals -1 then all section
          matching the 'section' name format are considered.
          
          param section               section name
          param defaultType           assumed task type
          param limit                 max items count to parse
          param raw                   extract with no interpolation,
          return                      a list of task items
        """
        result = list()
        
        if 0 < limit :
          for i in xrange(limit) :
            header = section % (i, )
            result.append( self._parseTaskItem(header, defaultType, raw) )
        else :
          # offset <= 0, loop as long as matching sections are found
          offset = 0
          header = section % (offset, )
          while self.has_section(header) :
            result.append( self._parseTaskItem(header, defaultType, raw) )
            offset = 1 + offset
            header = section % (offset, )
        
        return filter(None, result)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _getLockFilePath(self) :
        """\
          string _getLockFilePath(void)
          
          Get the lock file's name from the 'Config' section; if actually
          none is set (empty string), then a file name is build upon the
          current file's name.
          e.g. self.filename = 'AdminSuite.ini'   =>   'AdminSuite.lock'
          
          The result should be a _full_ qualified path.
          
          return                      lock file path
        """
        # shortcut to the 'Config' section settings
        cfgSect = self._settings[SECT_CONFIG]
        
        # default lock file name, use current file
        result  = os.path.splitext(self.filename)[0] + '.lock'
        buffer  = getattr(cfgSect, TOKN_LOCK_FILE, '').strip()
        if 0 < len(buffer) : result = buffer
        
        return asqPath.absolute(result, self.dirname)


    def _createLockFile(self, filepath, content='') :
        """\
          boolean _createLockFile(string filepath, string content)
          
          Tries to create a lock file; may fail (and return False), if
          the file already exists, or if it can't be created.
          
          param filepath              lock file to create
          param content               lock file content
                                        defaults to the empty string
          return                      True if operation suceeded
        """
        assert asqTypes.isNonEmptyInstance(filepath, basestring)  , \
                'wrong filepath value'
        
        if os.path.exists(filepath) :
          return False
        
        try :
          handle = open(filepath, 'wb')
        except IOError, e :
          return False
        else :
          try :
            handle.write(content)
          finally :
            handle.close()
        
        return True


    def _removeLockFile(self, filepath) :
        """\
          boolean _removeLockFile(string filepath)
          
          Tries to delete the lock file from the file system. Returns False
          if the file couldn't be removed; if the file did not exist in the
          first place, then the operation will be considered succesful.
          
          param filepath              lock file to delete
          return                      True if operation suceeded
        """
        assert asqTypes.isNonEmptyInstance(filepath, basestring)  , \
                'wrong filepath value'
        
        try :
          asqPath.delete(filepath, force=True)
        except OSError, e :
          pass # mute exception
        
        return (not os.path.exists(filepath))


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _getLogFilePath(self) :
        """\
          string _getLogFilePath(void)
          
          Get the logfile's name from the 'Logging' section; if actually
          none is set (empty string), then a file name is build upon the
          current file's name.
          e.g. self.filename = 'AdminSuite.ini'   =>   'AdminSuite.log'
          
          The result should be a full qualified path.
          
          return                      log file path
        """
        # shortcut to the 'Logging' section settings
        logSect = self._settings[SECT_LOGGING]
        
        # default logging file name, use current file
        result  = os.path.splitext(self.filename)[0] + '.log'
        buffer  = getattr(logSect, TOKN_LOG_FILE, '').strip()
        if 0 < len(buffer) : result = buffer
        
        return asqPath.absolute(result, self.dirname)


    def _buildLogHandler(self) :
        """\
          object _buildLogHandler(void)
          
          Build a logging handler instance according to the settings from
          the [Logging] section. For now, only rotating log files are
          supported.
          
          return                      logging handler object
        """
        # shortcut to the 'Logging' section settings
        logSect = self._settings[SECT_LOGGING]
        
        logfile = self._getLogFilePath()
        buffer  = getattr(logSect, TOKN_LOG_MAXSIZE, '1M')
        maxsize = asqConvert.humanToByte(buffer)
        buffer  = getattr(logSect, TOKN_LOG_MAXFILE, '3')
        maxfile = asqTypes.asInteger(buffer, 10, 3)
        # For now, no need to check the TOKN_LOG_HANDLER property....
        result  = asqLogging.RotatingFileHandler(logfile, 'a', maxsize, maxfile)
        
        result.setFormatter( logging.Formatter() )
        
        #~ result.setFormatter(logging.Formatter(
            #~ fmt='%(asctime)s  %(message)s', datefmt='%H:%M:%S'
          #~ ))
        
        return result


    def _removeLogHandler(self) :
        """\
          void _removeLogHandler(void)
          
          Removes current inner logHandler from the file's logger.
        """
        # trash previous handler
        if self._logHandler is not None :
          self.logger.removeHandler(self._logHandler)


    def _appendLogHandler(self) :
        """\
          void _appendLogHandler(void)
          
          Refresh the logger configuration, previous handler is removed
          and replaced with a new one build from the file's settings.
        """
        # shortcut to the 'Logging' section settings
        logSect = self._settings[SECT_LOGGING]
        
        self._removeLogHandler()
        if asqTypes.asBoolean(logSect.enabled) :
          self._logHandler = self._buildLogHandler()
        else :
          # use dummy handler, avoid logging warning message
          self._logHandler = asqLogging.SilentHandler()
        self.logger.addHandler(self._logHandler)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _taskItemsComments(self) :
        """\
          list _taskItemsComments(void)
          
          Gathers comments properties from the single taskItems defined
          in the build config. file.
          
          return                      comments strings
        """
        result = list()
        
        if SECT_MAILING in self.taskItems :
          buffer = self.taskItems[SECT_MAILING].comments
          if asqTypes.isNonEmptyInstance(buffer, basestring) :
            result.append('%-15s %s' % (SECT_MAILING, buffer))
        
        return result


    def _taskListsComments(self) :
        """\
          list _taskListsComments(void)
          
          Gathers comments properties from the taskLists defined
          in the build config. file.
          
          return                      comments strings
        """
        result = list()
        
        if SECT_INITIALIZE in self.taskLists :
          buffer = self.taskLists[SECT_INITIALIZE].comments
          if asqTypes.isNonEmptyInstance(buffer, basestring) :
            result.append('%-15s %s' % (SECT_INITIALIZE, buffer))
          
        if SECT_FINALIZE in self.taskLists :
          buffer = self.taskLists[SECT_FINALIZE].comments
          if asqTypes.isNonEmptyInstance(buffer, basestring) :
            result.append('%-15s %s' % (SECT_FINALIZE, buffer))
          
        for secName in self._getTasksOrderList() :
          buffer = self.taskLists[secName].comments
          if asqTypes.isNonEmptyInstance(buffer, basestring) :
            result.append('%-15s %s' % (secName, buffer))
          #~ for item in self.taskLists[secName] :
            #~ buffer = item.comments
            #~ if asqTypes.isNonEmptyInstance(buffer, basestring) :
              #~ result.append('%-15s %s' % (item.taskname, buffer))
          
        return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _guessTaskType(self, section) :
        """\
          string __guessTaskType(string section)
          
          Guess task type from the "section" header. Returns None
          if no task type can be guessed.
          
          param section               task item header
          return                      default task item type
        """
        header = section.lower()
        buffer = [ x.lower() for x in asqBuildTasks.taskTypes ]
        
        for i in xrange(len(buffer)) :
          if header.startswith(buffer[i]) :
            return asqBuildTasks.taskTypes[i]
        else :
          return None


    def _guessTaskListItemsHeader(self, section) :
        """\
          string _guessTaskListItemsHeader(string section)
          
          Guess header format for children items of given 'section'.
          e.g.
            _guessTaskListItemsHeader('Initialize')     => 'Initialize_%d'
            _guessTaskListItemsHeader('CvsTasks')       => 'CvsTask_%d'
          
          param section               task list stub section name
          return                      format for task list items headers
        """
        if not section :
          return '%d'
        elif section.endswith('s') :
          return '%s_%%d' % (section[:-1], )
        else :
          return '%s_%%d' % (section, )


    def _getTasksOrderList(self) :
        """\
          list _getTasksOrderList(void)
          
          Returns as a list of strings, the custom task sections order
          defined in the [Config] section.
          
          return                      tasks order as a list
        """
        # shortcut to the 'Config' section settings
        cfgSect = self._settings[SECT_CONFIG]
        
        result  = getattr(cfgSect, TOKN_TASKS_ORDER, '')
        # explode string, remove empty chunks and duplicates
        result  = [ x.strip() for x in result.split(',') ]
        
        return asqSequence.unique(filter(None, result), ordered=True)


  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _getFileComments(self) :
        """\
          list _getFileComments(void)
          
          Getter - fileComments property
          Gathers tasks comments from build file
          
          return                      list of comments
        """
        result = list()
        
        result.extend( self._taskItemsComments() )
        result.extend( self._taskListsComments() )
        
        return result


    def _getIgnored(self) :
        """\
          list _getIgnored(void)
          
          Getter - ignored property
          Returns the list of ignored sections names.
          
          return                      ignored sections list
        """
        return self._ignored


    def _getLogger(self) :
        """\
          object _getLogger(void)
          
          Getter - logger property
          Return the logger instance build from the current file's
          settings. If no file has been read, then None may be returned.
          
          return                      logger object
        """
        return logging.getLogger(LOGGER_NAME)


    def _getEnvironment(self) :
        """\
          dict _getEnvironment(void)
          
          Getter - environment property
          Return the OS specific section ENVIRONMENT
          
          return                      ENVIRONMENT section
        """
        return self._environment


    def _getSettings(self) :
        """\
          dict _getSettings(void)
          
          Getter - settings property
          Return the sections that can't be handled as task objects
          
          return                      "other" sections dictionary
        """
        return self._settings


    def _getTaskItems(self) :
        """\
          dict _getTaskItems(void)
          
          Getter - taskItems property.
          Returns the task items dictionary extracted from the current dataset.
          
          return                      task items dictionary
        """
        return self._taskItems


    def _getTaskLists(self) :
        """\
          dict _getTaskLists(void)
          
          Getter - taskLists property.
          Returns the task lists dictionary extracted from the current dataset.
          
          return                      task lists dictionary
        """
        return self._taskLists


  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    fileComments        = property(
        doc  = "tasks comments list",
        fget = _getFileComments,
        fset = None,
        fdel = None
      )

    ignored             = property(
        doc  = "ignored sections names list",
        fget = _getIgnored,
        fset = None,
        fdel = None
      )

    logger              = property(
        doc  = "logger bound to the file",
        fget = _getLogger,
        fset = None,
        fdel = None
      )

    environment         = property(
        doc  = "OS specific environment values",
        fget = _getEnvironment,
        fset = None,
        fdel = None
      )

    settings            = property(
        doc  = "settings sections dictionary",
        fget = _getSettings,
        fset = None,
        fdel = None
      )

    taskItems           = property(
        doc  = "task items dictionary",
        fget = _getTaskItems,
        fset = None,
        fdel = None
      )

    taskLists           = property(
        doc  = "task lists dictionary",
        fget = _getTaskLists,
        fset = None,
        fdel = None
      )


  # /class fileParser - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -