#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
basic unit introduction

Require
  Python        2.2

Classes
  class         fileHandler(dict defaults)
                  extends py_netasq.building.parsing.iniChecker::fileChecker
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.40 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  ConfigParser
import  logging
import  os
import  re
import  sys
import  time

from    py_netasq.commonlib     import asqPath, asqString, asqTypes
from    py_netasq.commonlib     import asqConvert

from    py_netasq               import building as asqBuild
from    py_netasq.building      import core     as asqBuildCore
from    py_netasq.building      import tasks    as asqBuildTasks

import  iniParser
import  iniChecker

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

# from 0 (silent) to 2 (lot of noise)
VERBOSITY_RANGE = range(3)

VERBOSITY_SILENT  = 0
VERBOSITY_QUIET   = 1   # triggers ant displays
VERBOSITY_NORMAL  = 2   # display full log


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class fileHandler(iniChecker.fileChecker) :
    """\
      A build file handler, derivated from file checker. Loads and 
      performs the tasks found in the build file.
      
      
      constructor fileHandler(dict defaults)
    """

    # for each task, the time when the command is executed is recorded
    # dd/mm hh:mm:ss (ex : '[30/01 16:04:15]')
    log_timeFmt = '[%d/%m - %H:%M:%S]'


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def handleFile(cls, filename, display=None, verbosity=VERBOSITY_QUIET, sections=None, defaults=None) :
        """\
          static var handleFile(string filename, object display, int verbosity, list sections, dict defaults)
          
          Opens, parses and executes each requested tasks, as given by
           'sections'. The "global" execution result is then returned.
          
          param filename              config file path
          param display               logger object
                                        Defaults to None
          param verbosity             verbosity level
                                        Defaults to VERBOSITY_QUIET
          param sections              sections to consider
                                        Defaults to None
          param defaults              instance wide default values
                                        Defaults to None
        """
        filename = asqPath.normalize(os.path.abspath(filename))
        aHandler = fileHandler(None)
        
        aHandler.display   = display
        aHandler.verbosity = verbosity
        
        aHandler.readFile(filename)
        if asqTypes.isNonEmptyInstance(defaults, dict) :
          aHandler.updateDefaults(defaults, '')
        
        aHandler._appendLogHandler()
        try :
          aHandler._recordHeader()
          aHandler._recordDefaults()
          
          result = aHandler._runFileSections(sections)
          aHandler._recordFooter(0 == result, aHandler._errmsg)
          
        finally :
          aHandler._removeLogHandler()
        
        return result


    # set class methods
    handleFile  = classmethod(handleFile)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __init__(self, defaults=None) :
        """\
          constructor fileHandler(dict defaults)
          
          Create a new build file handler, 'defaults' dictitonary will be
          used to set the "DEFAULT" section's options; those options can be
          used as variables.
          
          param defaults              DEFAULT options
                                        Defaults to None
        """
        super(fileHandler, self).__init__(defaults)
        
        self._verbosity = VERBOSITY_NORMAL
        # error string (compiler failure ...)
        self._errmsg    = None
        # logger object used for direct display
        self._display   = None
        # time elapsed for the last execution
        self._elapsed   = 0
        
        self._handler   = asqBuildTasks.globalTaskHandler()
        
        self._handler.onEmitMsg       = self.onEmitMsg
        self._handler.onProcessInput  = self.onProcessInput
        self._handler.onProcessOutput = self.onProcessOutput


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def onEmitMsg(self, message) :
        """\
          void onEmitMsg(string message)
          
          Called whenever the global task handler emits a message.
          This is actually used for the "quiet" verbose mode.
          
          Note : this function does also allow list of strings to be
          passed as `message` argument.
          
          param message               message sent from handler
        """
        if VERBOSITY_QUIET != self.verbosity :
          pass
        elif self._display is None :
          pass
        else :
          if isinstance(message, basestring) :
            self._display.log(logging.INFO, message)
          elif isinstance(message, list) :
            for msgItem in message :
              self._display.log(logging.INFO, msgItem)


    def onProcessInput(self, line, raw, *arglist, **argdict) :
        """\
          void onProcessInput(string line, bool raw, *arglist, **argdict)
          
          Called as the command input is processed line by line.
          Here, the 'line' is simply sent to the logging utility.
          
          param line                  line from command output.
          param raw                   disable output formatting
          param *arglist              extra parameters list.
          param **argdict             extra named parameters.
        """
        buffer = list()
        
        # prepends blank line before command string
        buffer.append('')
        
        # :TRICKY: 'taskname' was set by _runTaskItem(), see below
        #  the task name is assumed to never be longer than ~60 chars
        if 'taskname' not in argdict :
          buffer.append( time.strftime(self.log_timeFmt) +' '+ '<Task_X>' )
        else :
          buffer.append( time.strftime(self.log_timeFmt) +' '+ argdict['taskname'] )
        
        # adding command string (could be a faked one)
        buffer.append( asqString.wrapString('$ '+ line, 78, '  ') )
        # appends blank line after command string
        buffer.append('')
        
        buffer = '\n'.join(buffer)
        
        #~ buffer = time.strftime(self.log_timeFmt) +' '+ line
        #~ buffer = asqString.wrapString(buffer, 78, '  ')
        
        self.logger.log(logging.INFO, buffer)
        
        if VERBOSITY_NORMAL != self.verbosity :
          pass
        elif self._display is None :
          pass
        else :
          self._display.log(logging.INFO, buffer)
        
        del buffer


    def onProcessOutput(self, line, raw, *arglist, **argdict) :
        """\
          void onProcessOutput(string line, bool raw, *arglist, **argdict)
          
          Called as the command output (as a task is being executed) is
          processed line by line.
          Here, the 'line' is simply sent to the logging utility.
          
          param line                  line from command output.
          param raw                   disable output formatting
          param *arglist              extra parameters list.
          param **argdict             extra named parameters.
        """
        buffer = asqString.wrapString(line, 78, '  ')
        self.logger.log(logging.INFO, buffer)
        
        # if VERBOSITY_NORMAL and a display is set, then use it
        if VERBOSITY_NORMAL != self.verbosity :
          pass
        elif self._display is None :
          pass
        else :
          self._display.log(logging.INFO, buffer)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _runFileSections(self, allowed) :
        """\
          var _runFileSections(list allowed)
          
          Executes tasks from the parsed sections according to the given
          list of 'allowed' sections. If a task fails do execute, then it's
          shell error code will be returned.
          
          param allowed               list of allowed sections
          return                      execution status code
        """
        if allowed is None :
          pass
        elif not isinstance(allowed, list) :
          raise TypeError("type mismatch `allowed` parameter, expecting list")
        else :
          allowed[:] = [ x.lower() for x in allowed ]
        
        result = -1
        moment = time.clock()
        locker = None
        
        if not self._testFileVersion() :
          minver = '.'.join(iniParser.VERSION_MIN)
          maxver = '.'.join(iniParser.VERSION_MAX)
          buffer = list()
          buffer.append('%s\n' % (iniParser.SECT_FILEVERSION, ))
          buffer.append('%s value out of range [%s , %s]' % (iniParser.TOKN_VERSION, minver, maxver))
          self._errmsg = ''.join(buffer)
          # can't send mail, mailing task (and any other tasks) are ignored
        else :
          locker = self._getLockFilePath()
          
          buffer = list()
          buffer.append(time.asctime())
          buffer.append(self.fullpath)
          
          if not self._createLockFile(locker, '\n'.join(buffer)) :
            assert -1 == result
            self._errmsg = 'A lock file couldn\'t be created : `%s`' % (locker, )
            
          else :
            try :
              try :
                self._applyEnvironment()
                self._runTaskLists(allowed)
                
              except asqBuildCore.taskError, e :
                assert -1 == result
                self._elapsed = time.clock() - moment
                if e.task is None :
                  self._errmsg = e.message
                else :
                  self._errmsg = '%s\n%s' % (e.task.taskname, e.message)
                
              except asqBuildCore.taskHandlerError, e :
                result = e.status
                self._elapsed = time.clock() - moment
                if e.task is None :
                  self._errmsg = '\n'.join(e.output)
                else :
                  buffer = list()
                  buffer.append('%s\n'   % (e.task.taskname, ))
                  buffer.append('%s\n\n' % (str(e.task.toCmdLine()).strip(), ))
                  
                  # shorten log, add only the last lines
                  if iniParser.SECT_MAILING not in self.taskItems :
                    buffer.append('\n'.join(e.output))
                  else :
                    loglen = self.taskItems[iniParser.SECT_MAILING].error_preview
                    if 1 > loglen :
                      buffer.append('\n'.join(e.output))
                    elif (1 + loglen) > len(e.output) :
                      buffer.append('\n'.join(e.output))
                    else :
                      buffer.append('(...)\n')
                      buffer.append('\n'.join(e.output[-loglen : ]))
                  self._errmsg = ''.join(buffer)
                
              else :
                result = 0
                self._elapsed = time.clock() - moment
                self._errmsg  = None
              
            finally :
              self._removeLockFile(locker)
            
          if 0 != result :
            self._sendMail(allowed, False, self._errmsg)
          else :
            result = self._sendMail(allowed, True, 'Build Success')
        
        return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _runTaskLists(self, allowed) :
        """\
          void _runTaskLists(list allowed)
          
          Loops through the 'taskLists' dictionary and execute each items.
          
          param allowed               list of allowed sections
        """
        if iniParser.SECT_INITIALIZE not in self.taskLists :
          pass
        elif self._isSectionAllowed(iniParser.SECT_INITIALIZE, allowed) :
          self._runTaskList( self.taskLists[iniParser.SECT_INITIALIZE] )
        
        try :
          for secName in self._getTasksOrderList() :
            if not self._isSectionAllowed(secName, allowed) :
              continue
            else :
              self._runTaskList( self.taskLists[secName] )
          
        finally :
          if iniParser.SECT_FINALIZE not in self.taskLists :
            pass
          elif self._isSectionAllowed(iniParser.SECT_FINALIZE, allowed) :
            self._runTaskList( self.taskLists[iniParser.SECT_FINALIZE] )


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _runTaskList(self, aList, **argdict) :
        """\
          void _runTaskList(object aList, **argdict)
          
          Loops through the given list of tasks 'aList', merging each item
          with the list, completing paths, and then running it with the
          inner global task handler 'handler'. The extra named parameters
          are directly passed to the 'run()' method from inner taskHandler.
          
          param aList                 taskList instance
          param **argdict             extra named parameters.
        """
        if not isinstance(aList, asqBuildCore.taskList) :
          raise TypeError("type mismatch `aList` parameter, expecting taskList")
        
        asqBuild.logger.trace('fileHandler._handleTaskList(aList=%s)' % (aList.taskname, ))
        asqBuild.logger.incDepthTrace()
        try :
          
          if VERBOSITY_QUIET != self.verbosity :
            pass
          elif aList.enabled :
            self.toDisplay(logging.INFO, '')
            self.toDisplay(logging.INFO, '%s :' % (aList.taskname, ))
          else :
            self.toDisplay(logging.INFO, '')
            self.toDisplay(logging.INFO, '%s : Disabled' % (aList.taskname, ))
          
          if aList.enabled :
            for ersatz in aList.iterMerged() :
              try :
                ersatz.completePaths(self.dirname)
                self._runTaskItem(ersatz, **argdict)
                
              finally :
                del ersatz
          
        finally :
          asqBuild.logger.decDepthTrace()


    def _runTaskItem(self, aItem, **argdict) :
        """\
          void _runTaskItem(object aItem, **argdict)
          
          Runs given task item 'aItem'. The extra named parameters are
          directly passed to the 'run()' method from inner taskHandler.
          
          param aItem                 taskItem instance
          param **argdict             extra named parameters.
        """
        if not isinstance(aItem, asqBuildCore.taskItem) :
          raise TypeError("type mismatch `aItem` parameter, expecting taskItem")
        
        # adding extra value for processInput
        argdict['taskname'] = aItem.taskname
        
        asqBuild.logger.trace('fileHandler._handleTaskItem(aItem=%s)' % (aItem.taskname, ))
        asqBuild.logger.incDepthTrace()
        try :
          
          self.taskHandler.run(aItem, **argdict)
          
        finally :
          asqBuild.logger.decDepthTrace()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _applyEnvironment(self) :
        """\
          void _applyEnvironment(void)
          
          Parses the ENVIRONMENT section, and made change to the OS
          environment (setting variables).
        """
        def __getEnvValue(match) :
            # match.group(0) stands for the whole match
            # match.group(1) stands for the varname
            result = os.getenv(match.group(1))
            if result is None :
              return match.group(0)
            else :
              return result
        
        if self._environment is None :
          return # no environment definitions ?
        elif bool(self._environment) :
          regexp = re.compile('\$([a-z][a-z0-9_]*)', re.I)
          
          #  FOO = hello
          #  BAR = $FOO world
          #
          # Because there is no key order, BAR may be handled before FOO
          # => The outer loop ensures that ALL variables will be extended
          for i in range(len(self.environment)) :
            for key in self.environment.iterkeys() :
              self._environment[key] = regexp.sub(__getEnvValue, self._environment[key])
              os.environ[key] = self._environment[key]


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _sendMail(self, allowed, success, errmsg=None) :
        """\
          void _sendMail(list allowed, bool success, string errmsg)
          
          Sends mail using the settings from the configuration file.
          If 'success' is set to True, then a "build succesful" mail will
          be sent to the defined recipients.
          The 'message' will be used as the mail text content.
          
          param allowed               list of allowed sections
          param sucess                build state
          param message               mail content
        """
        if not self._isSectionAllowed(iniParser.SECT_MAILING, allowed) :
          return 0
        elif iniParser.SECT_MAILING not in self.taskItems :
          return 0
        else :
          buffer = list()
          
          buffer.append(self.fullpath)
          
          if  success : 
            buffer.append('')
            buffer.append('Build Success')
          else :
            buffer.append('')
            buffer.append('Build Failure')
            
            if errmsg :
              buffer.append('')
              for item in errmsg.strip().split('\n') :
                buffer.append( asqString.wrapString(item, 76, '  ') )
          
          buffer.append('')
          buffer.append('Total time: %s' % (asqConvert.secondToHuman(self._elapsed), ))
          
          tkItem = self.taskItems[iniParser.SECT_MAILING]
          if VERBOSITY_QUIET == self.verbosity :
            self.toDisplay(logging.INFO, '')
            self.toDisplay(logging.INFO, '%s :' % (tkItem.taskname, ))
          
          try :
            self._runTaskItem(tkItem, succesful=success, mailtext='\n'.join(buffer))
          
          except asqBuildCore.taskError, e :
            buffer = list()
            if self._errmsg is not None :
              buffer.append(self._errmsg)
            if e.task is None :
              buffer.append(e.message)
            else :
              buffer.append('%s\n%s' % (e.task.taskname, e.message))
            self._errmsg = '\n\n'.join(buffer)
            return -1
            
          except asqBuildCore.taskHandlerError, e :
            buffer = list()
            if self._errmsg is not None :
              buffer.append(self._errmsg)
            if e.task is None :
              buffer.append('\n'.join(e.output))
            else :
              buffer.append('%s\n%s' % (e.task.taskname, '\n'.join(e.output)))
            self._errmsg = '\n\n'.join(buffer)
            return -1
          
          else :
            return 0


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def toDisplay(self, level, message) :
        """\
          void toDisplay(int level, string message)
          
          Prints given 'message' with given 'level' to the display logger,
          If that one is actually set, or verbosity is not set to silent.
          
          param level                 message logging level
          param message               message string
        """
        if VERBOSITY_SILENT == self.verbosity :
          pass
        elif self._display is not None :
          self._display.log(level, message)


    def printOut(self, toLogger, toDisplay, level, message) :
        """\
          void printOut(bool toLogger, bool toDisplay, int level, string message)
          
          Prints given 'message' with given 'level' to the execution logger
          and/or to the display logger.
          
          param toLogger              allow message to be sent to logger
          param toDisplay             allow message to be sent to display
          param level                 message logging level
          param message               message string
        """
        if toDisplay :
          self.toDisplay(level, message)
        if toLogger :
          self.logger.log(level, message)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _recordDefaults(self) :
        """\
          void _recordDefaults(void)
        """
        self.printOut(True, True, logging.INFO, '')
        self.printOut(True, True, logging.INFO, 'Defaults :')
        
        buffer = self.defaults(raw=False)
        indexs = buffer.keys()
        indexs.sort()
        
        for key in indexs :
          if '__name__' == key :
            continue
          
          if not asqTypes.isNonEmptyInstance(buffer[key], basestring) :
            val = ''
          else :
            val = buffer[key].replace('\n', '\n\t')
          
          self.printOut(True, True, logging.INFO, '  %-18s %s' % (key, val))


    def _recordHeader(self) :
        """\
          void _recordHeader(void)
          
          Prints out a header for the current file being processing to the
          execution and display loggers.
        """
        self.printOut(True, True, logging.INFO, time.asctime())
        
        buffer = asqString.shortenString(self.fullpath, 32, '...', '\\')
        self.printOut(True, False, logging.INFO, 'Processing file : `%s`' % (self.fullpath, ))
        self.printOut(False, True, logging.INFO, 'Processing file : `%s`' % (buffer, ))


    def _recordFooter(self, success, errmsg) :
        """\
          void _recordFooter(bool success, string errmsg)
          
          Prints out a header for the current file being processing to the
          execution and display loggers.
          
          param sucess                build state
          param errmsg                build error message
        """
        self.printOut(True, True, logging.INFO, '')
        
        if success :
          self.printOut(True, True, logging.INFO, 'Build Success')
        else :
          self.printOut(True, True, logging.INFO, 'Build Failure')
          
          if errmsg :
            self.printOut(False, True, logging.INFO, '')
            for line in errmsg.split('\n') :
              buffer = asqString.wrapString(line, 70, '')
              self.printOut(False, True, logging.INFO, buffer)
        
        self.printOut(True, True, logging.INFO, '')
        self.printOut(True, True, logging.INFO, 'Total time: %s' % (asqConvert.secondToHuman(self._elapsed), ))
        
        self.printOut(True, False, logging.INFO, '')
        self.printOut(True, False, logging.INFO, ' -' * 33)
        
        self.printOut(True, True, logging.INFO, '')


  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _getDisplay(self) :
        """\
          object _getDisplay(void)
          
          Getter - display property
          Output display, logger object
          
          return                      logger instance
        """
        return self._display


    def _getVerbosity(self) :
        """\
          int _getVerbosity(void)
          
          Getter - verbosity property
          verbosity level
          
          return                      verbosity level
        """
        return self._verbosity


    def _getTaskHanlder(self) :
        """\
          object _getTaskHanlder(void)
          
          Getter - taskHandler property.
          Returns the globalTaskHandler instance
          
          return                      globalTaskHandler
        """
        return self._handler


  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _setDisplay(self, value) :
        """\
          void _setDisplay(object value)
          
          Setter - display property
          Output display, logger object
          
          param value                 logger instance
        """
        if isinstance(value, logging.Logger) :
          self._display = value
        else :
          self._display = None


    def _setVerbosity(self, value) :
        """\
          void _setVerbosity(int value)
          
          Setter - verbosity property
          verbosity level
          
          param value                 new verbosity level
        """
        minVal = min(VERBOSITY_RANGE)
        maxVal = max(VERBOSITY_RANGE)
        
        self._verbosity = max(minVal, min(maxVal, value))


  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    display     = property(
        doc  = "direct display",
        fget = _getDisplay,
        fset = _setDisplay,
        fdel = None
      )

    taskHandler = property(
        doc  = "globalTaskHandler instance",
        fget = _getTaskHanlder,
        fset = None,
        fdel = None
      )

    verbosity   = property(
        doc  = "verbosity level",
        fget = _getVerbosity,
        fset = _setVerbosity,
        fdel = None
      )


  # /class fileHandler  - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -