#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
basic unit introduction

Require
  Python        2.2

Classes
  class         buildFileEditor(dict defaults)
                  extends py_netasq.building.parsing.iniParser::fileParser
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.5 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  ConfigParser
import  sys

from    py_netasq.commonlib     import asqMapping
from    py_netasq.commonlib     import asqSequence

from    py_netasq               import building as asqBuild
from    py_netasq.building      import core     as asqBuildCore
from    py_netasq.building      import tasks    as asqBuildTasks

import  iniParser

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class buildFileEditor(iniParser.fileParser) :
    """\
      Editor writes down the current dataset in a structured way.
      
      constructor buildFileEditor(void)
    """

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __init__(self, defaults=None) :
        """\
          constructor buildFileEditor(dict defaults)
          
          Create a new build file parser, 'defaults' dictitonary will be
          used to set the "DEFAULT" section's options; those options can be
          used as variables.
          
          param defaults              DEFAULT options, defaults to None
        """
        super(buildFileEditor, self).__init__(defaults)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def write(self, file) :
        """\
          void write(object file)
          
          Write a representation of the configuration to the specified file
          object. This representation can be parsed by a future read() call.
          
          Because ConfigParser does not give a friggin damn about sections
          order, we have to do the dirty work by ourselves...
          
          param file                  file descriptor
        """
        writtn = list()
        buffer = list()
        
        #~ buffer.extend( self._getSettingsLines(writtn) )
        buffer.extend( self._getTasksLines(writtn) )
        
        #~ buffer.extend( self._getOthersLines(writtn) )
        
        #~ # empty target file
        #~ # > seek(offset=0, whence=0)
        #~ # > truncate(size=0)
        #~ file.seek(0, 0)
        #~ file.truncate(0)
        #~ file.write("\n".join(buffer))
        
        sys.stdout.write("\n".join(buffer))


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _sortSectionValues(self, values) :
        """\
          list _sortSectionValues(var options)
          
          Returns the sorted list [(key, val), ...] for given 'values'
          mapping. Dictionnaries don't keep key order.
          
          param values                mapping to sort
          return                      sorted mapping
        """
        def _sortCallback(a, b) : return cmp(a[0], b[0])
        
        if isinstance(values, dict) :
          result = values.items()
        else :
          result = values
        result.sort(_sortCallback)
        return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _getSettingsLines(self, written) :
        """\
          list _getSettingsLines(list written)
          
          Formats the generic settings (FileVersion, DEFAULT and Config
          sections) as a "ready to display" list of strings.
          Each section name is added to the 'written' list, that list is
          used to avoid section duplicates.
          
          param written               section already written
          return                      writeable strings list
        """
        result = list()
        
        # [FILEVERSION]
        if iniParser.SECT_FILEVERSION not in written :
          values = self._fileVersion.items(raw=True)
          buffer = self._getSectionLines(
              iniParser.SECT_FILEVERSION,
              self._sortSectionValues(values),
              default=''
            )
          result.extend(buffer)
          result.append('')
          written.append(iniParser.SECT_FILEVERSION)
        
        # [DEFAULT]
        if iniParser.SECT_DEFAULT not in written :
          values = self.defaults(raw=True)
          buffer = self._getSectionLines(
              iniParser.SECT_DEFAULT,
              self._sortSectionValues(values),
              default=''
            )
          result.extend(buffer)
          result.append('')
          written.append(iniParser.SECT_DEFAULT)
        
        # [CONFIG]
        if iniParser.SECT_CONFIG not in written :
          values = self._config.items(raw=True)
          buffer = self._getSectionLines(
              iniParser.SECT_CONFIG,
              self._sortSectionValues(values),
              default=''
            )
          result.extend(buffer)
          result.append('')
          written.append(iniParser.SECT_CONFIG)
        
        return result


    def _getTasksLines(self, written) :
        """\
          list _getTasksLines(list written)
          
          Formats the tasks (Mailing, Initialize, Finalize and user defined
          sections) as a "ready to display" list of strings.
          Each section name is added to the 'written' list, that list is
          used to avoid section duplicates.
          
          param written               section already written
          return                      writeable strings list
        """
        result = list()
        
        written.append(iniParser.SECT_MAILING)
        written.append(iniParser.SECT_INITIALIZE)
        written.append(iniParser.SECT_FINALIZE)
        
        # [MAILING]
        buffer = self._taskItemLines(
            iniParser.SECT_MAILING, asqBuildTasks.mailTaskType
          )
        if buffer :
          result.extend(buffer)
          result.append('')
        # [INITIALIZE]
        buffer = self._taskListLines(iniParser.SECT_INITIALIZE)
        if buffer :
          result.extend(buffer)
          result.append('')
        # [FINALIZE]
        buffer = self._taskListLines(iniParser.SECT_FINALIZE)
        if buffer :
          result.extend(buffer)
          result.append('')
        # User defined [FooTasks], [FooTask_x]
        
        tasksList = self._getTasksOrderList()
        section = tasksList[0]
        written.append(section)
        buffer = self._taskListLines(section)
        if buffer :
          result.extend(buffer)
          result.append('')
        
        
        #~ for section in self._getTasksOrderList() :
          #~ written.append(section)
          #~ buffer = self._taskListLines(section)
          #~ if buffer :
            #~ result.extend(buffer)
            #~ result.append('')
        
        return result


    def _getOthersLines(self, written) :
        """\
          list _getOthersLines(list written)
          
          Formats remaining sections as a "ready to display" list of strings.
          Each section name is added to the 'written' list, that list is
          used to avoid section duplicates.
          
          param written               section already written
          return                      writeable strings list
        """
        result = list()
        
        remain = asqSequence.difference(self._tasks.keys(), written)
        for section in remain :
          aTask = self._tasks[section]
          if isinstance(aTask, asqBuildCore.taskList) :
            buffer = self._taskListLines(section)
          elif isinstance(aTask, asqBuildCore.taskItem) :
            buffer = self._taskItemLines(section, None)
          else : # skip - object is lost
            continue
          written.append(section)
          if buffer :
            result.extend(buffer)
            result.append('')
        
        remain = asqSequence.difference(self.sections(), written)
        for section in remain :
          values = self.getSectionDict(section, full=False, raw=True)
          buffer = self._getSectionLines(section, values, default='')
          written.append(section)
          result.extend(buffer)
          result.append('')
        
        return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _taskItemLines(self, section, defaultType=None) :
        """\
          list _taskItemLines(string section, string defaultType)
          
          param section               section name
          param defaultType           assumed task item type
          return                      writeable strings list
        """
        result = list()
        tkItem = self._tasks.get(section, None)
        
        if tkItem is not None :
          buffer = self._taskLines(section, tkItem, True, defaultType)
          result.extend(buffer)
        
        return result


    def _taskListLines(self, section) :
        """\
          list _taskListLines(string section)
          
          param section               section name
          return                      writeable strings list
        """
        result = list()
        header = self._guessTaskListItemsHeader(section)
        tkList = self._tasks.get(section, None)
        
        if tkList is not None :
          buffer = self._taskLines(
              section, tkList, filtered=False, defaultType=None
            )
          result.extend(buffer)
          for index in xrange(len(tkList)) :
            item = tkList[index]
            name = header % (index, )
            buffer = self._taskLines(name, item, True, tkList.tasktype)
            result.extend(buffer)
        
        return result


    def _taskLines(self, section, aTask, filtered=True, defaultType=None) :
        """\
          list _taskLines(string section, object aTask, bool filtered, string defaultType)
          
          
          param section               section name
          param aTask                 task (item or list)
          param filtered              display or not empty values
          param defaultType           assumed task type
          return                      writeable strings list
        """
        result = list()
        values = aTask.toList(filtered)
        
        for i in xrange(len(values)-1, -1, -1) :
          if 'taskname' == values[i][0] :
            del values[i]
          elif 'tasktype' == values[i][0] :
            if defaultType == values[i][1] :
              del values[i]
          elif 'workdir' == values[i][0] :
            if '.' == values[i][1] :
              del values[i]
        
        result.append('[%s]' % (section, ))
        for (key, val) in values :
          key = key.ljust(self._ljustify)
          result.append('%s = %s' % (key, val, ))
        result.append('')
        
        return result


  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # /class buildFileEditor  - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -