#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
Define a "build file" checking (based upon the build file parser) class
used by the building utility.

Require
  Python        2.2

Classes
  class         fileChecker(dict defaults)
                  extends py_netasq.building.parsing.iniParser::fileParser
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.16 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  ConfigParser
import  os

from    py_netasq.commonlib     import asqPath, asqTypes

from    py_netasq               import building as asqBuild
from    py_netasq.building      import core     as asqBuildCore
from    py_netasq.building      import tasks    as asqBuildTasks

import  iniParser

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class fileChecker(iniParser.fileParser) :
    """\
      A build file checker, derivated from fileParser. This class
      should not be instanciated, a class method "checkFile" is provided to
      perform file validation.
      
      
      static void       checkFile(string filename, object display)
      
      constructor       fileChecker(dict defaults)
      
      void              report(object display)
      
      property object   status  check routine result
    """

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def checkFile(cls, filename, display, sections=None, defaults=None) :
        """\
          static void checkFile(string filename, object display, list sections, dict defaults)
          
          Open and parse given file, then check dataset. Given logger
          object 'display' is used to display the check report.
          If given 'filename' does not exist then an exception may be
          raised.
          The 'sections' parameter defines a selective list of sections
          to consider, if 'sections' equals None, then all defined sections
          are taken into account.
          The 'defaults' parameter can be used to define or override values
          found in "DEFAULT" section.
          
          param filename              config file path
          param display               logger object
          param sections              sections to consider
                                        Defaults to None
          param defaults              instance wide default values
                                        Defaults to None
        """
        filename = asqPath.normalize(os.path.abspath(filename))
        aParser  = fileChecker(None)
        
        aParser.readFile(filename)
        if asqTypes.isNonEmptyInstance(defaults, dict) :
          aParser.updateDefaults(defaults, '')
        
        aParser.report(display, sections)


    # set class methods
    checkFile = classmethod(checkFile)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __init__(self, defaults=None) :
        """\
          constructor fileChecker(dict defaults)
          
          Create a new build file validator, 'defaults' dictitonary will be
          used to set the "DEFAULT" section's options; those options can be
          used as variables.
          
          param defaults              DEFAULT options, defaults to None
        """
        super(fileChecker, self).__init__(defaults)
        self._status = asqBuildCore.checkResult()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def report(self, display, sections=None) :
        """\
          void report(object display, list sections)
          
          Checks current dataset, and prints out the report using given
          logger object 'display' as output.
          The 'sections' parameter defines a selective list of sections
          to consider, if 'sections' equals None, then all defined sections
          are taken into account.
          
          Since checkResult's use the same logging levels values, each
          messages are displayed through the accurate method
          (display.warning(), display.critical(), ...).
          To avoid information cluttering, only non empty checkResults
          are displayed.
          
          param display               logger object
          param sections              sections to consider
                                        Defaults to None
        """
        self._checkFileSections(sections)
        self._status.report(display)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _handleDataSet(self) :
        """\
          void _handleDataSet(void)
         
          Handles the setup file dataset, normalizes dataset and extracts
          task items and lists. The whole dataset is automatically checked
          whenever a file is loaded.
          
          Note : Right after a file as been loaded, one would better
          directly call the 'report()' from the 'status' property
          ('checker.status.report(display)') instead of the 'report()'
          method ('checker.report(display)'), since the latter will do the
          whole checking routine one more time.
          
          This method is automatically called whenever file is read
          (through '_read()' or 'readFile()').
        """
        asqBuild.logger.trace('fileChecker._handleDataSet(void)') 
        asqBuild.logger.incDepthTrace()
        try :
          
          super(fileChecker, self)._handleDataSet()
          # allowed sections is None : browse through the whole file
          self._checkFileSections(None)
          
        finally :
          asqBuild.logger.decDepthTrace()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _checkFileSections(self, allowed) :
        """\
          void _checkFileSections(list allowed)
          
          Checks the parsed sections according to the given list of 'allowed'
          sections, and updates inner 'status' checkResult instance.
          
          param allowed               list of allowed sections names
        """
        if allowed is None :
          pass
        elif not isinstance(allowed, list) :
          raise TypeError("type mismatch `allowed` parameter, expecting list")
        else :
          allowed[:] = [x.lower() for x in allowed]
        
        self._status.reset()
        
        if not self._testFileVersion() :
          buffer = asqBuildCore.checkResult(iniParser.SECT_FILEVERSION)
          buffer.addMsgError('%-15s unrecognised value' % (iniParser.TOKN_VERSION, ))
          self._status.appendChild(buffer)
        else :
          # first warns about ignored sections
          self._checkIgnored(allowed)
          # inspect tasks build from the config file
          self._checkSettings(allowed)
          self._checkTaskItems(allowed)
          self._checkTaskLists(allowed)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _isSectionAllowed(self, secName, allowed) :
        """\
          bool _isSectionAllowed(string secName, list allowed)
          
          Tells whether or not given section name 'secName' is allowed by
          the 'allowed' list. If 'allowed' is None, then any string value
          is allowed as a section.
          
          param secName               section name to check
          param allowed               list of allowed sections names
        """
        result = False
        
        if not isinstance(secName, basestring) :
          raise TypeError("type mismatch `secName` parameter, expecting string")
        elif allowed is None :
          result = True
        elif not isinstance(allowed, list) :
          raise TypeError("type mismatch `allowed` parameter, expecting list")
        else :
          result = secName.lower() in allowed
        
        return result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _checkSettings(self, allowed) :
        """\
          void _checkSettings(list allowed)
          
          Loops through the 'settings' dictionary and checks each items.
          The reports are appended to the inner 'status' property.
          
          param allowed               list of allowed sections
        """
        # [FileVersion] section
        if self._isSectionAllowed(iniParser.SECT_FILEVERSION, allowed) :
          buffer = self._checkSection(
              iniParser.SECT_FILEVERSION, iniParser.SECT_FILEVERSION_DEFAULTS
            )
          self._status.appendChild(buffer)
        # [Config] section
        if self._isSectionAllowed(iniParser.SECT_CONFIG, allowed) :
          buffer = self._checkSection(
              iniParser.SECT_CONFIG     , iniParser.SECT_CONFIG_DEFAULTS
            )
          self._status.appendChild(buffer)
        # [Logging] section
        if self._isSectionAllowed(iniParser.SECT_LOGGING, allowed) :
          buffer = self._checkSection(
              iniParser.SECT_LOGGING    , iniParser.SECT_LOGGING_DEFAULTS
            )
          self._status.appendChild(buffer)


    def _checkTaskItems(self, allowed) :
        """\
          void _checkTaskItems(list allowed)
          
          Loops through the 'taskItems' dictionary and checks each items.
          The reports are appended to the inner 'status' property.
          
          param allowed               list of allowed sections
        """
        # [Mailing] section
        if self._isSectionAllowed(iniParser.SECT_MAILING, allowed) :
          buffer = self._checkTask(self.taskItems, iniParser.SECT_MAILING)
          self._status.appendChild(buffer)


    def _checkTaskLists(self, allowed) :
        """\
          void _checkTaskLists(list allowed)
          
          Loops through the 'taskLists' dictionary and checks each items.
          The reports are appended to the inner 'status' property.
          
          param allowed               list of allowed sections
        """
        if self._isSectionAllowed(iniParser.SECT_INITIALIZE, allowed) :
          buffer = self._checkTask(self.taskLists, iniParser.SECT_INITIALIZE)
          self._status.appendChild(buffer)
          
        if self._isSectionAllowed(iniParser.SECT_FINALIZE, allowed) :
          buffer = self._checkTask(self.taskLists, iniParser.SECT_FINALIZE)
          self._status.appendChild(buffer)
        
        for secName in self._getTasksOrderList() :
          if not self._isSectionAllowed(secName, allowed) :
            continue
          else :
            buffer = self._checkTask(self.taskLists, secName)
            self._status.appendChild(buffer)


    def _checkIgnored(self, allowed) :
        """\
          void _checkIgnored(list allowed)
          
          Adds a warning to the check result for each one of the ignored
          section. If 'allowed' is different from None, then nothing is
          done.
          If 'allowed' is set, then we're on some kind of restricted mode;
          There's no need to warn that a  non-allowed section will be
          ignored.
          
          param allowed               list of allowed sections
        """
        if allowed is None :
          for section in self.ignored :
            buffer = asqBuildCore.checkResult(section)
            buffer.addMsgInfo('ignored section')
            self._status.appendChild(buffer)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _checkSection(self, section, expected) :
        """\
          void _checkSection(string section, dict expected)
          
          Checks the 'section', 'expected' gives the list of expected
          options with their default values (assumed when option is missing
          from the section).
          
          param section               section name
          param expected              section's options with default values
        """
        result = asqBuildCore.checkResult(section)
        
        if not self.has_section(section) :
          result.addMsgInfo('section is missing')
        for (option, defval) in expected.iteritems() :
          self._checkOption(result, section, option, defval)
        
        return result


    def _checkOption(self, result, section, option, defval) :
        """\
          void _checkOption(
                        object result, string section,
                        string option, string defval )
          
          Checks the value of 'option', diagnostic message is added to the
          result object. 'defval' is the default value assumed when option
          is missing from the 'section'.
          
          param result                check result
          param section               section name
          param option                option name
          param defval                default value for option
        """
        try :
          self.get(section, option, raw=False, vars=None)
        except (ConfigParser.NoOptionError, ConfigParser.NoSectionError), e :
          if defval is None :
            result.addMsgError('%-15s missing' % (option, ))
          else :
            result.addMsgWarning('%-15s missing, assuming "%s"' % (option, defval))
        except Exception, e :
          result.addMsgCritical('%-15s %s' % (option, str(e).strip()))


    def _checkTask(self, aDict, section) :
        """\
          void _checkTask(dict aDict, string section)
          
          Calls up the "check()" method for the task object corresponding
          to the given 'aDict[section]'.
          
          param aDict                 tasks (lists or items) dictionary
          param section               section name == task name
          return                      a check result instance
        """
        assert isinstance(aDict, dict)
        result = asqBuildCore.checkResult(section)
        
        if section not in aDict :
          result.addMsgInfo('section is missing')
        else :
          aDict[section].check(result, self.dirname)
        
        return result


  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _getStatus(self) :
        """\
          object _getStatus(void)
          
          Getter - status property.
          Returns the checkResult instance.
          
          return                      check result
        """
        return self._status


  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    status      = property(
        doc  = "checkResult instance",
        fget = _getStatus,
        fset = None,
        fdel = None
      )


  # /class fileChecker  - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -