#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
Defines a basic list of task items, which is also a task by itself.

Require
  Python        2.2

Classes
  class         taskList(bool enabled)
                  extends py_netasq.building.core::task
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.10 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  os

from    py_netasq.commonlib     import asqSequence
from    py_netasq.commonlib     import asqTypes

from    py_netasq               import building as asqBuild

import  task                                    as asqTask
import  taskItem                                as asqTaskItem


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class taskList(asqTask.task, list) :
    """\
      A list of task items, and also a task by itself; a taskList
      does define the default attributes values for each of the owned
      taskItems.
      Unlike taskItem, empty strings and 'None' values share the same
      meaning : "no value for that task attribute".
      
      void              clear(void)
      
      object            iterMerged(void)
      object            appendNewTask(bool enabled)
    """

    # Properties defined for that task type and its descendant.
    # Note : attribute names should be "lower case" only.
    attributes    = (
        'taskname',             # task name (should held section name)
        'count',                # task item count
        'tasktype',             # task type as a string
        'enabled',              # is task active ?
        'workdir',              # task base directory
        'comments',             # about this task
        'others'                # unrecognized attributes
      )

    # task type compatibility list
    compatibility = (
        asqTask.task.classTaskType,
      )

    # task item class
    taskClass = asqTaskItem.taskItem

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __setattr__(self, name, value) :
        """\
          void __setattr__(string name, var value)
          
          Special method - Called when an attribute assignment is attempted.
          This is called instead of the normal mechanism (i.e. store the
          value in the instance dictionary). name is the attribute name,
          value is the value to be assigned to it.
          
          If name is not found in defined attributes list, and is not a
          mangled name (_classname__name), then (name, value) is added to
          the object's "others" dictionnary.
          
          Each time an attribute value is set, 'normalize()' should be
          called to ensure consistency.
          
          param name                  attribute to set
          param value                 new value
        """
        asqBuild.logger.trace('taskList.__setattr__(%s, %s)', name, repr(value))
        asqBuild.logger.incDepthTrace()
        try :
          
          if 'count' != name :
            super(taskList, self).__setattr__(name, value)
          else :
            # pad list to at least match given item count
            buffer = asqTypes.asInteger(value, radix=0, default=0)
            asqSequence.stuffSequence(self, buffer, None)
          
        finally :
          asqBuild.logger.decDepthTrace()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def debug(self) :
        """\
          void debug(void)
          
          Prints out the main attributes values through the logging
          utility.
        """
        length = len(self)
        super(taskList, self).debug()
        
        for i in xrange(length) :
          asqBuild.logger.debug('')
          asqBuild.logger.debug('* item %d / %d', i, length-1)
          asqBuild.logger.incDepthDebug()
          self[i].debug()
          asqBuild.logger.decDepthDebug()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _attributeNonZero(self, name) :
        """\
          bool _attributeNonZero(string name)
          
          Callback used by self.__nonzero__()
          Checks that attribute is not empty.
          Should be overriden to set the behavior for children classes.
          
          param name                  attribute name
          return                      is attribute NOT empty ?
        """
        if 'count' == name :
          return 0 < len(self)
        else :
          return super(taskList, self)._attributeNonZero(name)


    def _attributeDebug(self, name) :
        """\
          void _attributeDebug(string name)
          
          Callback used by self.debug()
          Prints out debug information for given attribute.
          Should be overriden to set the behavior for children classes.
          
          param name                  attribute name
        """
        if 'count' == name :
          self._debugKeyVal(name, len(self))
        else :
          super(taskList, self)._attributeDebug(name)


    def _attributeToList(self, aList, name) :
        """\
          void _attributeToList(list aList, string name)
          
          Callback used by self.toList(...)
          Appends attribute to given "aList" list.
          Should be overriden to set the behavior for children classes.
          
          param aList                 list of attributes
          param name                  attribute name
        """
        if 'count' == name  :
          aList.append( (name, len(self)) )
        else :
          super(taskList, self)._attributeToList(aList, name)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def check(self, status=None, baseDir=None) :
        """\
          object check(object status, string baseDir)
          
          Checks attribute values, fills given status with gathered error
          or warning messages, and then returns the 'status' object. If
          'status' is None, then a new "checkResult" object will be
          instanciated.
          Calls the check method on each item of  the list and appends
          returned checkResult objects to the status.
          
          param status                checkResult object
                                        Defaults to None
          param baseDir               base for absolute path
                                        Defaults to None
          return                      check result
        """
        if status is None :
          asqBuild.logger.trace('taskList.check(status=None)')
        else :
          asqBuild.logger.trace('taskList.check(status)')
        asqBuild.logger.incDepthTrace()
        try :
          
          offset = 0
          status = super(taskList, self).check(status, baseDir)
          for ersatz in self.iterMerged() :
            try :
              asqBuild.logger.trace('status.appendChild(self[%d].check(status=None))', offset)
              status.appendChild(ersatz.check(None, baseDir))
              offset += 1
            finally :
              del ersatz
          
          return status
          
        finally :
          asqBuild.logger.decDepthTrace()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def normalize(self) :
        """\
          void normalize(void)
          
          Correct values types for some peculiar attributes; e.g. the
          "enabled" attribute must be a boolean value, None value will be
          converted to False.
          Empty string attributes (and only string values) will be replaced
          by None values.
        """
        asqBuild.logger.trace('taskList.normalize()')
        asqBuild.logger.incDepthTrace()
        try :
        
          super(taskList, self).normalize()
          for key in self.attributes :
            buffer = self.__dict__[key]
            if not isinstance(buffer, basestring) :
              continue
            elif 0 == len(buffer) :
              object.__setattr__(self, key, None)
        
        finally :
          asqBuild.logger.decDepthTrace()


    def clear(self) :
        """\
          void clear(void)
          
          Empty tasks list.
        """
        asqBuild.logger.trace('taskList.clear()')
        asqBuild.logger.incDepthTrace()
        try :
        
          self[:] = []
       
        finally :
          asqBuild.logger.decDepthTrace()


    def reset(self) :
        """\
          void reset(void)
          
          Set object's attributes to their default values.
        """
        super(taskList, self).reset()
        self.clear()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def iterMerged(self) :
        """\
          object iterMerged(void)
          
          Generator function, returns a generator which will loop through
          the tasklist's items and yield merged task items clones.
          
          return                      task item generator
        """
        for i in xrange(len(self)) :
          result = self[i].clone()
          result.mergeWithTask(self)
          yield result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def appendNewTask(self, enabled=False) :
        """\
          object appendNewTask(bool enabled)
          
          Returns a brand new task, which has been added to taskList.
          Somehow, it looks a little bit like a "factory" function.
          
          param enabled               is task enabled, defaults to False
          return                      task object
        """
        result = self.taskClass(enabled)
        self.append(result)
        return result


  # /class taskList - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -