#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
A simple task used by the automated building system, adds
 "mergeWithX" methods and "command" generation to the task
 object.

Require
  Python        2.2

Classes
  class         taskItem(bool enabled)
                  extends py_netasq.building.core::task
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.15 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  os

from    py_netasq.commonlib     import asqCommand, asqMapping
from    py_netasq.commonlib     import asqSequence, asqString

from    py_netasq               import building as asqBuild

import  task                                    as asqTask


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class taskItem(asqTask.task) :
    """\
      A simple task used by the automated building system. For single
      taskItem, there is a huge difference between 'None' attribute value
      and an empty string; each attribute that is equal to 'None', means
      "use taskList value for that attribute", whereas empty string does
      override tasklist values, meaning "no value for that attribute".
     
     
      void              mergeWithList(list aList, string defaultType)
      void              mergeWithTask(object aTask)
     
      str               antDisplay(void)
     
      object            toCmdLine(void)
      void              buildTmpFile(void)
    """

    # task canonical name (ex : 'dcc32', 'cvs') used by antDisplay()
    ant_name    = ''
    # ant margin, used to display the task's ant name
    ant_margin  = 15

    # Properties defined for that task type and its descendant.
    attributes  = asqTask.task.attributes

    # properties that could be overwritten by a "Merge" method
    mergeable   = (
        'workdir',
      )


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __str__(self) :
        """\
          string __str__(void)
          
          Called by the str() built-in function and by the print statement
          to compute the "informal" string representation of an object. This
          differs from __repr__() in that it does not have to be a valid
          Python expression: a more convenient or concise representation may
          be used instead. The return value must be a string object
          
          Returns a pseudo command line: arguments are shortened to reduce
          information cluttering. If current task is found non valid, then
          an empty string will be returned.
          
          return                      pseudo command line string
        """
        if not self.isValid() :
          return ''
        
        result = self.toCmdLine()
        try :
          for i in xrange(len(result)) :
            result[i] = asqString.shortenString(result[i], 25, '...')
          return str(result)
        finally :
          del result


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def debug(self) :
        """\
          void debug(void)
          
          Prints out the main attributes values through the logging
          utility.
        """
        self._debugKeyVal('str(self)', str(self))
        super(taskItem, self).debug()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _attributeToList(self, aList, name) :
        """\
          void _attributeToList(list aList, string name)
          
          Callback used by self.toList(...)
          Appends attribute to given "aList" list.
          Should be overriden to set the behavior for children classes.
          
          param aList                 list of attributes
          param name                  attribute name
        """
        if 'tmpfile' != name :
          super(taskItem, self)._attributeToList(aList, name)


    def _attributeIsPath(self, name) :
        """\
          bool _attributeIsPath(string name)
          
          Callback used by self.completePaths(...)
          Returns True if attribute is a path, please note that the
          'workdir' attribute should NOT be considered as a path here.
          Should be overriden to set the behavior for children classes.
          
          param name                  attribute name
          return                      is attribute a path
        """
        if 'tmpfile' == name :
          return True
        else :
          return super(taskItem, self)._attributeIsPath(name)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _attributeMerge(self, name, value) :
        """\
          void _attributeMerge(string name, var value)
          
          Callback used by mergeWithList(...)
          Merge attribut value with given one.
          Should be overriden to set the behavior for children classes.
          
          param name                  attribute name
          param value                 attribute value
        """
        if name.startswith('_') :
          pass
        elif name not in self.attributes :
          if self.others.get(name, None) is None :
            self.others[name] = value
        elif name in self.mergeable :
          if self.__dict__[name] is None :
            self.__setattr__(name, value)
        
        #~ else :
          #~ print 'else ???'
          #~ print self.attributes
          #~ print self.mergeable


    def _fillCmdLine(self, aCmdLine) :
        """\
          void _fillCmdLine(object aCmdLine)
          
          Callback used by toCmdLine()
          Build the task command line, task is assumed valid.
          Should be overriden to set the behavior for children classes.
          
          param aCmdLine              cmdLine object
        """
        asqBuild.logger.trace('taskItem._fillCmdLine(aCmdLine)')
        asqBuild.logger.incDepthTrace()
        try :
          
          assert aCmdLine is not None
          
        finally :
          asqBuild.logger.decDepthTrace()


    def _fillTmpFile(self, aFile) :
        """\
          void _fillTmpFile(object aFile)
          
          Callback used by buildTmpFile()
          Dumps config values to given file object, which must have
          been opened.
          Should be overriden to set the behavior for children classes.
          
          param aFile                 file object
        """
        assert aFile is not None


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  
    def mergeWithList(self, aList, defaultType=None) :
        """\
          void mergeWithList(list aList, string defaultType)
          
          Replaces mergeable attributes equal to 'None' with values provided
          by 'aList' (Note : list should be a valid map).
          If there is no "tasktype" value found in list, then 'defaultType'
          is assumed.
          If list is compatible with current object, then unrecognised
          attributes are added to the "others" dict. If not, then only
          the commonly shared attributes will be considered, all other
          attributes will simply be ignored.
           
          param aList                 list of attributes
          param defaultType           assumed tasktype.
                                        Defaults to None
        """
        asqBuild.logger.trace('taskItem.mergeWithList(aList, %s)', defaultType)
        asqBuild.logger.incDepthTrace()
        try :
        
          # create a copy, avoid making unnecessary changes to original list
          buffer = dict(aList)
          tkType = asqMapping.extractItem(buffer, 'tasktype', defaultType)
          
          if not self.isCompatibleWith(tkType) :
            # class attributes list may be reduced (!) by child class
            filter = asqSequence.intersection(asqTask.task.attributes, self.attributes)
            asqMapping.filterKeys(buffer, filter, False)
          
          self.beginUpdate()
          try :
            for (key, val) in buffer.iteritems() :
              self._attributeMerge(key, val)
          finally :
            self.endUpdate()
            self.normalize()
            del buffer
        
        finally :
          asqBuild.logger.decDepthTrace()
  
  
    def mergeWithTask(self, aTask) :
        """\
          void mergeWithTask(object aTask)
          
          Replaces mergeable attributes equal to 'None' with values provided
          by 'aTask'.
          If 'aTask' is compatible with current object, then unrecognised
          attributes are added to the "others" dict. If not, then only
          the commonly shared attributes will be considered, all other
          attributes will simply be ignored.
           
          param aTask                 another task
        """
        buffer = aTask.toList(True)
        try :
          self.mergeWithList(buffer)
        finally :
          del buffer


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def antDisplay(self) :
        """\
          string antDisplay(void)
          
          Returns a "java ant" look alike string showing the task's action.
          eg.
            'dcc32       Compiling : E:\projects\...\src\dummy.dpr'
          
          Note : One had better override the '_antAction' method to change
            behavior for heir classes.
          
          return                      task action
        """
        result = list()
        chunks = list()
        
        chunks.append( self._antName() )
        if not self.enabled :
          chunks.append('disabled -')
        
        if not self.isValid() :
          chunks.append('invalid task settings')
          result.append(' '.join(chunks))
        else :
          buffer = self._antAction()
          if 0 == len(buffer) :
            chunks.append('')
            result.append(' '.join(chunks))
          # if we get a simple string, then only add it to the line
          elif isinstance(buffer, basestring) :
            chunks.append(buffer)
            result.append(' '.join(chunks))
          elif not isinstance(buffer, list) :
            chunks.append('')
            result.append(' '.join(chunks))
          else :
            # here we got a list of strings
            # remove the first one, then add all other lines
            chunks.append(buffer.pop(0))
            result.append(' '.join(chunks))
            
            if self.enabled :
              for item in buffer :
                chunks[:] = []
                chunks.append( self._antMargin() )
                chunks.append( item )
                result.append(' '.join(chunks))
        
        if 1 < len(result) :
          return result
        else :
          return ''.join(result)


    def _antName(self) :
        """\
          string _antName(void)
          
          Builds the ant like string prefix.
          eg:
            `     [dcc32]'
          
          return                      task prefix
        """
        result = '[%s]' % (self.ant_name, )
        return result.rjust(self.ant_margin)


    def _antMargin(self) :
        """\
          string _antMargin(void)
          
          Builds a ant like left margin.
          
          return                      string made of space
        """
        return ' ' * self.ant_margin


    def _antAction(self) :
        """\
          string _antAction(void)
          
          Callback used by antDisplay()
          Returns a short string explaining the task's action.
          Should be overriden to set the behavior for children classes.
          
          Note : this function may also return a list of strings instead
          (without the left margin).
          
          return                      task's action
        """
        return ''


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def toCmdLine(self) :
        """\
          object toCmdLine(void)
          
          Returns a command line as an cmdLine instance. If current task
          is found empty or invalid, then an empty command is returned.
          A cmdLine instance should always be returned, "None" is not
          an acceptable value.
          
          return                      cmdLine instance
        """
        asqBuild.logger.trace('taskItem.toCmdLine()')
        asqBuild.logger.incDepthTrace()
        try :
          
          result = asqCommand.cmdLine()
          self._fillCmdLine(result)
          return result
          
        finally :
          asqBuild.logger.decDepthTrace()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def buildTmpFile(self, aPath=None) :
        """\
          void buildTmpFile(string aPath)
          
          Dumps config values to the file indicated by the 'tmpfile'
          property, if it does exist and is different from None.
          Previous file content will be lost, if file does not exist
          it will be created.
          If "aPath" is set, it will replace the tmpfile attribute value.
          
          Some tasks need to build a file from scratch, that file
          would then be handed to the executable as a config file.
           
          param aPath                 path to file
                                        defaults to None
        """
        asqBuild.logger.trace('taskItem.buildTmpFile(%s)', str(aPath))
        asqBuild.logger.incDepthTrace()
        try :
          
          if aPath is not None :
            self.tmpfile = aPath
          if getattr(self, 'tmpfile', None) is None :
            return
          buffer = open(self.tmpfile, 'w')
          try :
            buffer.seek(0, 0)
            buffer.truncate(0)
            self._fillTmpFile(buffer)
          finally :
            buffer.close()
          
        finally :
          asqBuild.logger.decDepthTrace()


  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # /class taskItem - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -